% FGS forward solver default parameters overloading from sources
% P = fgsp(P)
%
% .algoNL             Plasma convergence method
% .xscal              Scale solution x to be of order ~1
% .workers            Number of workers for parallel computation
% .insrc              Data input source
%
% Solver parameters:
% .algoF              Algorithm for root finding (jfnk,newton,broyden)
% .tolF               Stopping tolerance for root finding algorithm
% .kmax               Maximum iterations for root finding algorithm
% .nrestarts          Number of maximum randomized restarts of the solver
% .restart_random     Randomized restarting if Newton solver gets stuck
% .debuglinesrch      Trigger plotting of linesearch space
% .userowmask         Reduce Newton problem size when possible
% .robust_on          Allow subiteration with reduced steps in case of NaN in the operator
% .nrobust            Max number of subiterations for robust decrease of dx
% .algoGMRES          Algorithm variant for GMRES method (direct_inversion,matlab_gmres,sim,giv,aya)
% .mkryl              Maximum krylov subspace dimension for GMRES
% .usepreconditioner  If true, use preconditioner for GMRES
% .prectype           Type of preconditioner to use (user, inv, ilu)
% .precupdate         Frequency at which to update preconditioner (once, time, newton)
% .precside           Left or right preconditioning for algoGMRES=sim/giv/aya (left, right)
% .Prec               User-provided preconditioner
% .jacobian_handle    Use function handle form of jacobian (algoF=newton)
% .epsilon_d          Step size to approximate Jacobian numerically
% .jacobian_sparse    Use sparse-matrix form of jacobian (algoF=newton, algoNL=Newton-GS)
%
% Broyden Solver parameter (if algoF=broyden)
% .use_inverse          Approximate the inverse of the Jacobian in broyden's solver. Default true
% .large_system         Udpate the inverse matrix only at the end of the solver call. Default true
% .is_factored          Keep a low rank approximation of the inverse matrix (Default true)
% .ldim                 Rank of the inverse matrix approximation. Only used if is_factored=true. Default 10
% .group_size           Update of rank 'group_size' of the inverse. Default 4
%
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [P] = fgsp(P,varargin)

D = meqp(); % meq default parameters

%% Default Parameters
% NL solver parameters
D.mkryl = 150; %
D.nrestarts = 5; % number of maximum randomized restarts of the solver
D.restart_random = true; % default randomized restarting if Newton solver gets stuck
D.tolF = 1e-10; % default value of tol (minimum tolerance max(abs(F(x))))
D.kmax = 50; % default value of kmax (maximum number of Newton steps)
D.kinner = 10; % maximum inner iterations in Picard algorithm stabilizer  (only for algoNL='Picard')
D.kpic = 50;   % maximum Picard loop iterations (only for algoNL='Picard')
D.algoNL = 'all-nl'; % Plasma convergence method:
%   'all-nl': entire plasma state is unknown of nonlinear system.
%   'Picard': solve plasma with inner Picard loop.
%   'Newton-GS': entire plasma state is unknown of nonlinear system with different residual formulation in fgsF_GS
D.algoF = 'jfnk'; % 'jfnk' 'newton','broyden' Specify the algorithm to be used for nonlinear solver
D.algoGMRES = 'aya'; % GMRES with Ayachour's method
D.anajac = true; % Use analytical Jacobian
D.jacobian_handle = false; % Use function handle form of jacobian
D.epsilon_d = 1e-8; % default values of epsilon_d (finite step size used to approximate the Jacobian)
D.usepreconditioner = 0; % Preconditioning matrix
D.prectype = 'inv';
D.precupdate = 'once';
D.precside = 'right';
D.Prec = [];
D.debuglinesrch = false; % Plotting for linesearch
D.jacobian_sparse = false;

% Newton-GS preconditioner parameters
D.ilu_algo = 'crout';
D.ilu_droptol = 1e-3;

% Broyden solver parameter
D.use_inverse = true; % approximate the inverse of the Jacobian in broyden's solver
D.large_system = true; % Parameter of broyden: do not udpate the inverse matrix at each iteration, but only at the end
D.is_factored = true; % Parameter of broyden: Keep a low rank approximation of the inverse matrix
D.ldim = 10; % Rank of the inverse matrix approximation. Only used if is_factored=true
D.group_size = 4; % Update of rank group_size of the inverse
D.ratio_gb = 0.5; % Linear interpolation between good and bad broyden update (good=1 and bad=0). When is_factored=true should be <1.

D.xscal = true; % Scale x to be of order ~1
D.insrc = 'fbt'; % 'fbt' 'liu'
D.userowmask = true; % Use full system size as default
D.agcon = {'Ip','bp','qA'}; % default constraints (see meqagcon for details)
D.LYall = true; % return all time slices even if not converged
D.tolPicard = 1e-12; % Picard loop convergence tolerance, used for algoNL='Picard'

% Initialization source
D.insrc = 'fbt'; % 'fbt' 'liu'
D.shot = NaN;

%% Overwrites parameters from input and set defaults
for k = 1:2:length(varargin), P.(varargin{k}) = varargin{k+1}; end
for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
end

%% order
P = orderfields(P);

%% Checks
assert(~P.usepreconditioner || ~strcmp(P.algoNL,'Picard'),'usepreconditioner must be false if algoNL=''Picard''')
assert(P.ilim~=2 || ~P.anajac,'Analytical jacobian cannot be used with ilim=2')
if isequal(P.algoF,'broyden')
  assert(P.anajac,'broyden:anajac','Must set anajac=true when algoF=''broyden''')
  assert(~strcmp(P.algoNL,'Newton-GS'),'broyden:Newton_GS','Cannot use Broyden method with Newton-GS')
end
assert(~P.jacobian_sparse || strcmp(P.algoNL,'Newton-GS'),'Sparse jacobian is only available with algoNL=Newton-GS')
end
