function [L,LX,LY] = fgs(tok,shot,t,varargin)
%FGS - FORWARD GRAD-SHAFRANOV SOLVER
% [L,LX,LY] = FGS(TOK,SHOT,T,'PAR',VAL,...)
% Solve the static free-boundary grad shafranov problem. Given conductor
% currents and internal plasma constraints, it returns the full
% equilibrium.
%
% L = FGS('TOK',SHOT,...) returns only the pre-computed quantities as provided by 
% FGSP<TOK>, FGSG<TOK> and FGSC for later use.
% [L,LX] = FGS('TOK',SHOT,...) additionally returns the data in LX
% generated from FGSX<TOK>
%
% For general help,see README.MD and MEQHELP
% For help on LY,  see MEQT and FBTT
% For help on LX,  see FGSX and FGSX<TOK>
% For help on L.P, see MEQP, MEQP<TOK>, FGSP and FGSP<TOK> 
% For help on L.G, see MEQG, MEQG<TOK>, FGSG and FGSG<TOK> 
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

assert(ischar(tok),'tok must be a string');
tok = lower(tok); % tok: 'tcv' 'ids'
meqpath(mfilename,tok); % check required files and add path if necessary
assert(isnumeric(shot),'shot must be numeric');
if nargin>=4, assert(~mod(numel(varargin),2),'varargin must come in ''parameter'',value pairs'); end

% Tokamak specific functions
fgsptok = str2func(['fgsp' tok]);
fgsgtok = str2func(['fgsg' tok]);
fgsxtok = str2func(['fgsx' tok]);

% Algorithm configuration parameters, geometry and anciliary data
P = fgsptok(shot(1),varargin{:}); % parameters from sources
G = fgsgtok(shot(1),P);  % geometry from sources
P = fgsp(P); 
G = fgsg(G,P); 
L = fgsc(P,G); % Final assembly of all parameters

if nargout >= 2
  assert(isnumeric(t)||isempty(t),'time must be numeric');
  % Get inputs data for simulation
  LX = fgsxtok(t,L);
  LX = fgsx(L,LX); 
end

if nargout >= 3
  % Solve problem
  LY = fgst(L,LX);
end

end
