%FGET  Forward Grad-shafranov Evolution solver
% LY = FGET(L,LX) solves the Grad-Shafranov equation and the active and passive coil evolution
%
% Debug levels (L.P.debug):
%     0: progress bar with dots
%    >0: problem information and information per time step
%    >1: solver iterations per time step
%    >3: display full parameter structure 
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [LY,xnl,xnldot,Prec,psstate,Tstate,nnoc,alarm,solverinfo] ...
  = fget(L,LX,varargin)

%% Arguments
LYp_init = true;    xnl_init = true; xnldot_init = true;  Prec_init = true;
psstate_init = true; Tstate_init = true;  nnoc_init = true; dt_init = true;

for k = 1:2:length(varargin)
  switch varargin{k}
    % Initialise iterator state via arguments
    case 'LYp'    , LYp     = varargin{k+1};     LYp_init = false;
    case 'xnl'    , xnl     = varargin{k+1};     xnl_init = false;
    case 'xnldot' , xnldot  = varargin{k+1};  xnldot_init = false;
    case 'Prec'   , Prec    = varargin{k+1};    Prec_init = false;
    case 'psstate', psstate = varargin{k+1}; psstate_init = false;
    case 'Tstate' , Tstate  = varargin{k+1};  Tstate_init = false;
    case 'nnoc'   , nnoc    = varargin{k+1};    nnoc_init = false;
    case 'dt'     , dt      = varargin{k+1};      dt_init = false;
    % Additional output arguments
    case 'argout', L.argoutc = meqargoutc(varargin{k+1},L.argoutc);

    % Additional parameters
    otherwise, L.P.(varargin{k}) = varargin{k+1};
  end
end
is_iterator = ~xnl_init;

%% Debug levels
dispprogress = (L.P.debug==0) && ~is_iterator; % display progress bar
dispinfo     = L.P.debug>0 && ~is_iterator; % display problem information
dispiter     = L.P.debug>0; % display information per time step
dispdetails  = L.P.debug>1; % displays lower-level solver details, forces dispiter header display at each time step
dispinputs   = L.P.debug>3; % display parameter inputs

plotiter     = L.P.debugplot==1; % plot equilibrium per time step (if >1 more complex plots will appear)

if dispinfo, meqinfo(L,LX); end
if dispiter, tstart=tic; end
if dispinputs
  fprintf('\nCall to fget with these parameters:\n')
  dispparams(L.P);
end

%% Check linearization is present
if L.P.lin && ~isfield(L,'lin')
  error('L.lin is necessary for all fget calls with lin=true')
end

%% Initial state computed using fgs
LXt = meqxk(LX,1); % get only first time index
if LYp_init
  LYp = rmfield(LXt,{'Vadot0','Iadot0','SQ'}); % initial LY
  % Add missing parts of output structure
  LYp.Uf = zeros(L.G.nf,1);
  LYp.Um = zeros(L.G.nm,1);
  LYp.Un = zeros(L.nn  ,1);
  LYp.res = 0;
  itstart = 2;
else
  itstart = 1;
  assert(numel(LX.t)==1,...
    'must pass only a single LX time slice for iterator mode')
end

%% Time init
nt = numel(LX.t);
if dt_init
  if nt>1 % init dt from LX.t differences
    dt = (LX.t(nt) - LX.t(1))/(nt - 1);
    if any(abs(diff(LX.t)-dt) > dt*sqrt(eps))
      error('times must be equally spaced when passing multiple LX slices')
    end
  elseif ~is_iterator
    % In this case we are just calling fget with a single time slice.
    dt = 0; % dt can not be inferred from single-slice LX 
    % but some value is used to enable calling fgess, fgepre, meqps.
  else
    error('can''t initialize fget as iterator with single-slice LX')
  end
elseif is_iterator
  % check time 
  if abs( (LXt.t - LYp.t) - dt ) > 10*eps()
    error('unexpected time step, found %f expected %f, delta=%f', ...
      (LXt.t - LYp.t),dt, LXt.t - LYp.t - dt)
  end 
end

%% Pack linearized system into state space representation, with discrete time
if L.P.lin
  [sys_tmp,L.lin.ind,~,L.lin.yo,L.lin.x0dotL,L.lin.y0] = fgess(L,dt);
  % store ABCD in matrices since much faster access
  L.lin.sysA = sys_tmp.A;  L.lin.sysB = sys_tmp.B;
  L.lin.sysC = sys_tmp.C;  L.lin.sysD = sys_tmp.D;
  L.lin.iut = [sys_tmp.InputGroup.Va,sys_tmp.InputGroup.Co,sys_tmp.InputGroup.Ini,sys_tmp.InputGroup.dCodt];
  L.lin.OutputGroupString = sys_tmp.OutputGroup;
end

%% Init controller
if ~isempty(L.P.ctrlfct)
  % NB: Combined use of iterator mode and controller is not supported.
  % Typically one one would use iterator mode when calling with an external controller.
  if ~LYp_init
    error('FGET:NoFGETIteratorAndController','Use of controller for fget iterator mode is not supported');
  end

  fbcontrol = true;
  [cstate,Vactr] = L.P.ctrlfct('init',[],L.P.ctrlpar,L,LYp);
  if L.P.ctrlpar.dt < (dt-1000*eps) && ~(dt==0) % don't test if dt==0 (single slice, non-iterator call)
    error('controller time step %3.3e is smaller than simulation time step %3.3e',L.P.ctrlpar.dt,dt)
  end
else
  fbcontrol = false;
  Vactr = LXt.Va;
end

%% Init power supply
if psstate_init, [psstate,~] = meqps(L.G,[],dt,Vactr); end

%% Init protection limits
if Tstate_init, [Tstate,LYp,~] = meqlim(L,LYp,[],[]); end

%% NL state vector initialization
if xnl_init
  xnl = L.LX2x(LYp);
end

if xnldot_init
  xnldot = zeros(L.nN,1);
end

if nnoc_init
  nnoc = 0;
end
%% Index and other initialization
itend = nt;
itc = 1; itctr = 1;
dobreak = false; alarm = false;

%% Preconditioner initialization
% We don't have a solveF parameter structure yet
P.algoGMRES   = L.P.algoGMRES;
if L.P.usepreconditioner && ~L.P.lin % Preconditioner not needed for linear sim
  P.prectype    = L.P.prectype;
  P.precupdate  = L.P.precupdate;
  P.ilu_algo    = L.P.ilu_algo;
  P.ilu_droptol = L.P.ilu_droptol;

  if Prec_init
    P.Prec = L.P.Prec;
  else
    P.Prec = Prec; % Use value passed by iterator
  end

  if Prec_init && (itstart>itend)
    % Need to initialize Prec for iterator
    Prec = fgepre(L,LYp,P,dt);
  end
else
  P.prectype = 'user';
  P.precupdate = 'once';
  P.Prec = [];

  if Prec_init && (itstart>itend)
    % Need to initialize Prec for iterator
    Prec = []; % prec for iterator
  end
end

%% Time loop init and checks
% Allocate LY
LY = repmat(LYp,nt,1);
prev_H=[];

dotimestep = (itstart<=itend);
% Check that when we do some time step in rzp, we call the linear solver
if dotimestep && ~L.P.lin && strcmp(L.code,'rzp')
  error('rzp:linearonly','rzip is only a linear model for now, so can not run nonlinear time stepper');
end

%% Time loop
for it = itstart:itend
  LXt = meqxk(LX,it); % Input data for this time step

  % Voltage command to Power Supply
  if fbcontrol
    Vacmd = Vactr;
  else
    Vacmd = LXt.Va;
  end

  % Power supply
  [psstate,Vact] = meqps(L.G,psstate,dt,Vacmd);
  LXt.Va = Vact; % overwrite LXt.Va with power supply output

  % Time step
  if L.P.lin
    % Euler implicit dt step L
    LYt = fgetkl(L,LXt,LYp);
  else
    % Euler Implicit dt step NL
    xnlp = xnl;

    % Update preconditioner
    if (strcmp(P.precupdate,'time') || (it==itstart && Prec_init && strcmp(P.precupdate,'once')))
      % Use previous state
      Prec = fgepre(L,LYp,P,dt);
    end

    [xnl,LYt,prev_H,solverinfo] = fgetk_implicit(xnlp,xnldot,L,LXt,LYp, Prec, prev_H);
    xnldot = (xnl-xnlp)/dt;
  end
  % copy common things
  LYt.t       = LXt.t;
  LYt.shot    = LXt.shot;
  LYt.tokamak = LXt.tokamak;

  %% Derivatives
  LYt = meqLYdot(LYt,LYp);

  % Check coils limits and add normalized distance from limits to outputs
  [Tstate,LYt,alarm] = meqlim(L,LYt,dt,Tstate); % limits

  % Additional output arguments
  for k = 1:numel(L.argoutc)
    if isempty(L.argoutc{k}.file)
      LYt.(L.argoutc{k}.fld) = eval(L.argoutc{k}.exp);
    end
  end

  LY(it) = LYt; % Store in LY array
  LYp    = LYt; % Store for next time step
  if alarm && L.P.stoponalarm
    fprintf('stop on protection alarm'); dobreak = true;
  end

  if ~L.P.lin
    if ~LYt.isconverged
      nnoc = nnoc+1; % increment counter for non-convergent time steps
      if nnoc > L.P.nnoc
        dobreak=true;
        if dispiter
          msg = sprintf('Stopping due to %d consecutive non-convergence',nnoc);
          meqmsge('i',mfilename,L.P.tokamak,LXt.t,solverinfo.niter,...
            L.P.shot,msg,'FGE:TooManyNonConverged');
        end
      elseif dispiter
        msg = sprintf('%d/%d consecutive non-convergent slices',nnoc,L.P.nnoc);
        meqmsge('i',mfilename,L.P.tokamak,LXt.t,solverinfo.niter,...
          L.P.shot,msg,'FGE:NonConvergenceCounter');
      end
    else
      nnoc = 0; % reset counter
    end
  end

  %% Controller Step
  if fbcontrol
    if mod(LYt.t,L.P.ctrlpar.dt)<1000*eps
      % controller will get simulation info from previous controller time step
      itcontrol = it-round(L.P.ctrlpar.dt/dt)+1-itc;
      if itcontrol>=1 % else, not enough simulation times to call controller yet
        % call controller step
        [cstate,Vactr,~,ctdebugt] = L.P.ctrlfct('step',cstate,L.P.ctrlpar,L,LY(itcontrol));
        if ~isempty(ctdebugt), ctdebug(itctr) = ctdebugt; end %#ok<AGROW>
        itctr = itctr+1; % controller counter
      end
    end
  end

  %% Debugging display/outputs
  if dispprogress && (it==itstart || it==itend || ~rem(it, ceil(itend/10)))
    meqprogress(it,itend,it==itstart); % progress bar with progressing dots
  elseif dispiter
    disp_header = dispdetails || (~rem(it-itstart,10)); % triggers progress display header
    if disp_header 
      meqdebug(L);
    end
    meqdebug(L,LYt,it);
  end
  if dobreak, break; end
  
  if plotiter
    clf;
    meqplott(L,LYt);
    drawnow;
  end

end % time loop

if dobreak && ~isempty(it), LY = LY(1:it); end % chuck the rest if stopped prematurely

if dispiter, fprintf('Time elapsed (fget.m): %4.3f[s]\n',toc(tstart)); end

LY = meqlpack(LY); % pack into structure of arrays
if isfield(LX,'shot'), LY.shot = LX.shot; end
if isfield(LX,'tokamak'), LY.tokamak = LX.tokamak; end
if ~isempty(L.P.ctrlfct) && exist('ctdebug','var')
  % append controller debug info to LY if desired
  LY.ctdebug = meqlpack(ctdebug);
end

end
