function [sys,ind,x0,yo,x0dot,y0] = fgess(L,Ts,outputlist)
% fgess - Compute state-space model matrices
% [sys,ind,x0,yo,x0dot,y0] = fgess(L,Ts{,measlist})
%
%  L: L structure containing L.lin (from fgel)
%  Ts: Sample time (0=continuous time, default)
%  outputlist: cell array with list of outputs to pass. Default = 'all';
%
% state: x  = {Ia,Iu,(Ip)}, inputs: u = {Va, Co, dCoidt, (Ini)}. 
% outputs: defined via outputlist. Default 'all'.
%
% Quantities between parenthesis () are used only when cde is included.
% State space form (as in Walker et al. FSD 2006)
% S dxdt = K x + U u -> dxdt = A x + B u
%                          y = C x + D u + yo
% or
%                  d(x-x0)dt = A (x-x0) + B (u-u0) + x0dot
%                          y = C (x-x0) + D (u-u0) + y0
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Check inputs
assert(isfield(L,'lin'), 'fgel needs to be called before fgess to compute the linearization')

%% Choose if continuous/discrete time system output
if nargin==1
  Ts = 0; % continuous-time system
else
  assert(isscalar(Ts),'Ts must be scalar')
end

%% Outputlist
if nargin<=2
  outputlist = 'all';
end

% The list must include all state variables Iy,Fx,ag,Ia,Iu
if isequal(outputlist,'all')
  measlist = {'Bm','Ff','Ia','Iu','Ip','Ft','rIp','zIp','Iy','Fx','FA','FB','ag','rA','zA','bp','qA','li','Wk'};
  if L.nn, measlist = [measlist,{'Fn','Brn','Bzn'}]; end % optional
  if L.P.idoublet,  measlist = [measlist,{'IpD','rIpD','zIpD','bpD'}]; end % optional
else
  measlist = outputlist;
end
nmeas = numel(measlist);

%% Define state related matrices S,K
% S dxdt = K x + U u 
% S is the matrix formed by the mutual inductances Mee (+ Lp if cde) 
% and the correction coming from plasma motion Xee (+ Xeo, Xpe, Xpo if cde). 
% K is the resistance matrix, formed by the resistances Re (+ Rp if cde)
% U is the actuator matrix, formed the matrices that connect the actuator
% Va (+ Ini if cde) to the state evolution. It also connect the disturbances Co and dCodt 
% through Xeo and Xpo (they vary if Ip is a state or a Co).

% State indices
iL = find(L.lin.maskL);
[~,ind.iSIa] = ismember(L.ind.ixa,iL);
[~,ind.iSIu] = ismember(L.ind.ixu,iL);
[~,ind.iSIp] = ismember(L.ind.ixg,iL);
ind.iSIp = ind.iSIp(ind.iSIp>0);
ind.iSIe = [ind.iSIa,ind.iSIu];
nS = numel(iL);

% S matrix
S = L.lin.S;

% K matrix
K = L.lin.K;

% Labels for state
state_lbl = L.dimN(L.lin.maskL);

% linearization state
x0 = L.lin.x0L;

% linearization inputs
u0 = L.lin.u0L;

%% Define indices for input vector u  = {Va,Co,Ini, d...dt}
[ind, nU] = indxgen(ind,{'iUVa',L.G.na,'iUCo',L.nC,'iUIni',L.nD,'iUdCodt',L.nC},0);

% U matrix
U = [L.lin.U,L.lin.Udot(:,L.ind.iuC)];

% Labels for u
labelVa    = cellstr(strcat('Va_'    ,L.G.dima));
labelCo    = cellstr(strcat('Co_'    ,L.agconc(:,3),num2str([L.agconc{:,4}]','_%03d')));
labelIni   = cellstr(num2str((1:L.nD)','Ini_%03d_S'));
labeldCodt = cellstr(strcat('dCodt_' ,L.agconc(:,3),num2str([L.agconc{:,4}]','_%03d')));
input_lbl  = [labelVa;labelCo;labelIni;labeldCodt];

%% Define the measurements matrices from state C and D, such that y = C dx + D du + y0

% Define indices for measurement vector y
namedimpair = cell(1,2*nmeas);
for ii=1:nmeas
  mymeas = measlist{ii};
  namedimpair(2*ii-1:2*ii) = {['iM' mymeas],prod(meqsize(L,mymeas))};
end
% Indexing for measurements
[ind, nM] = indxgen(ind,namedimpair,0); % Gen index structure

C = zeros(nM,nS);
for ii = measlist
  myval = L.lin.(['d' ii{:} 'dxL']); nval = size(myval,1);
  myind = ind.(['iM' ii{:}]);
  % For A/B-suffixed quantities size(myval,1)=nA/nB while numel(myind)=L.nD
  C(myind(1:nval),:) = myval;
end

D = zeros(nM,nU);
for ii = measlist
  myval = L.lin.(['d' ii{:} 'du']); nval = size(myval,1);
  myind = ind.(['iM' ii{:}]);
  % For A/B-suffixed quantities size(myval,1)=nA/nB while numel(myind)=L.nD
  D(myind(1:nval),1:(L.G.na+L.nC+L.nD)) = myval; % No explicit dependence on dCodt
end

% Labels for y
meas_lbl = cell(1,nmeas);
for ii = 1:nmeas
  mymeas = measlist{ii};
  switch mymeas
    case {'Bm','Ff','Ia','Iu'} % use L.G.dim* directly
      meas_lbl{ii} = cellstr(strcat([mymeas,'_'],L.G.(['dim',mymeas(end)])));
    otherwise
      prodsize = prod(meqsize(L,mymeas));
      if prodsize==1
        meas_lbl{ii} = {mymeas};
      else
        formatstring = sprintf([mymeas,'_%%0%dd'],1+floor(log10(prodsize))); % correct padding zeros
        meas_lbl{ii} = cellstr(num2str((1:prodsize)', formatstring));
      end
  end
end
% Join all cell arrays
meas_lbl = vertcat(meas_lbl{:});

%% Pack measurements of linearization state
y0 = zeros(nM,1);
for ii = 1:nmeas
  mymeas = measlist{ii};
  myval = L.lin.(mymeas); nval = numel(myval);
  myind = ind.(['iM' mymeas]);
  % For A/B-suffixed quantities numel(myval)=nA/nB while numel(myind)=L.nD
  y0(myind(1:nval)) = myval;
end
% Compute y offset as in Walker et al. FST 2006
yo = y0 - C*x0 - D*u0;

%% Generate the state space representation
if Ts == 0
  % Continuous time representation
  A = S\K;
  B = S\U;
  sys = ss(A,B,C,D);
  x0dot = -S\L.lin.res0L;
else
  % Discrete time representation. Euler Implicit scheme.
  A = (S-Ts*K)\S;
  B = (S-Ts*K)\(Ts*U);
  sys = ss(A,B,C,D,Ts);
  x0dot = -(S-Ts*K)\(Ts*L.lin.res0L);
  % Warning when using lsim
  % Matlab ss for discrete system considers
  % x[n+1] = A x[n] + B u[n]
  % y[n]   = C x[n] + D u[n]
  % While typically the fgetk linear stepper considers
  % y[n+1]   = C x[n+1] + D u[n+1]
end
sys = set(sys,'StateName', state_lbl, 'InputName',input_lbl, 'OutputName',  meas_lbl);

assert(size(B,2)==numel(L.lin.u0L),'size mismatch between size(B,2) and u0L')

%% Add input delays
if isfield(L.G,'Vadelay')
  if ~all(L.G.Vadelay==0)
    warning('Found nonzero values of L.G.Vadelay. These are not included in the state-space system')
  end
end

%% Group outputs
for ii= measlist
  sys.OutputGroup.(ii{:}) = ind.(['iM' ii{:}]);
end

%% Group inputs
inlist = {'Va','Co','Ini','dCodt'};
for ii = inlist
  sys.InputGroup.(ii{:}) = ind.(['iU' ii{:}]);
end

end

function [ind,indend] = indxgen(ind, label_dim_pair, istart)
% Generate subsequent index structure from couple of {label, dimension}
% Extends the ind structure with the new inputs
% istart -> index offset to generate the new index starting from istart + 1

for ii=1:2:numel(label_dim_pair)
  indend = istart+label_dim_pair{ii+1};
  ind.(label_dim_pair{ii}) = istart+1:indend;
  istart = indend;
end
end

