function [L] = fgel(L,LX)
% FGE linearization around initial equilibrium guess LX.
% Takes linearizations around the GS equilibrium using fgeFlin
% and computes linearizations for the coil current equation.
% and fills L.lin structure with linearization data to be used by fgess() 
% or fget.m
%
% The FGE equations can be separated into static equations and dynamic
% equations. The static equations, or FGS equations are the Grad-Shafranov
% equation and the plasma profile constraints. The dynamic equations are
% the circuit equations and the current diffusion equations. We write them
% as:
%   - e(x,u) = 0                    for the static ones
%   - f(x)*d/dt​(g(x)) + h(x,u) = 0  for the dynamic ones
% where x is the full FGE state and u is the vector or external inputs. In
% particular we assume that the dynamic equations are linear in the time
% derivative of x.
%
% The linearized equations around a solution (x,u) are then
%   - de/dx*dx + de/du*du = 0
%   - f(x)*g'(x)*d(dx)/dt + dh/dx*dx + dh/du*du = 0
% We can write this in the general form S*d(dx/dt) + K*dx + U*du = 0
% with S = [0;f(x)*g'(x)], K = [de/dx;dh/dx], U = [de/du;dh/du]
%
% After time discretization, the FGE operator depends on the current state
% x, the current inputs u and the past state x0. For a time step dt>0, we
% define it as
%   F(x,u,x0) = [e(x,u);
%                f(x)*(g(x)-g(x0))/dt+h(x,u)]
% When dt=0 we zero out the time derivative terms and we define F with
%   F(x,u,x0) = [e(x,u);
%                h(x,u)]
%
% Coming back to the definitions of S,K and U, we see that:
%   - K =     dF/dx   with dt=0
%   - U =     dF/du   with dt=0
%   - S = -dt*dF/dx0  with dt>0 at x0=x
%
% If we define F_(x,u,x0,xdot) = F(x,u,x0-xdot*dt) we have
%   - K = dF_/dx    with dt=0
%   - U = dF_/du    with dt=0
%   - S = dF_/dxdot with dt>0 at xdot=(x-x0)/dt
%
% This structure was used in the fgeF implementation of the FGE operator.
% When required it outputs the analytical jacobians Jx = dF_/dx, Ju =
% dF_/du and Jxdot = dF_/dxdot at xdot=0 and x0=x.
% Notes about this convention for Jxdot:
%   - in general Jxdot=[0;f(x)*g'(x)] is not equal to
% -dt*dF/dx0 = [0;f(x)*g'(x0)]. However in many cases g is actually linear
% in x such that g'(x)=g'(x0) even if x~=x0, a counter example is the
% cde_ss_0D equation which includes a time-derivative of FA and FB which
% are non-linear functions of the state.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

assert(nargin==2,'not enough input arguments')
assert(numel(LX.t)==1,'fgel works for one time slice only, use meqxk() to slice')

% if no plasma, equations are already linear
if all(LX.Ip == 0)
  L.lin = struct();
  return
end

%% Compute linearization
if L.P.debug,  fprintf('Linearizing around equilibrium... '); end

%% Store linearization points
lin.Iy = LX.Iy;
lin.ag = LX.ag;
lin.Ie = [LX.Ia;LX.Iu];
lin.Ia = LX.Ia;
lin.Iu = LX.Iu;
lin.Bm = LX.Bm;
lin.Ff = LX.Ff;
lin.rA = LX.rA;
lin.zA = LX.zA;
lin.zIp = LX.zIp;
lin.rIp = LX.rIp; 
lin.zIp = sum(L.zzy(:).*LX.Iy(:));
lin.rIp = sum(L.rry(:).*LX.Iy(:));
lin.Ip = LX.Ip;
if L.nn
  lin.Fn = LX.Fn; 
  lin.Brn = LX.Brn;
  lin.Bzn = LX.Bzn;
end
lin.Ini = LX.Ini;
lin.SQ  = LX.SQ;
lin.Fx  = LX.Fx; 
lin.FA  = LX.FA;
lin.FB  = LX.FB;
lin.Ft  = LX.Ft;
lin.bp  = LX.bp;
lin.qA  = LX.qA;
lin.li  = LX.li;
lin.Wk  = LX.Wk;

lin.Rp = LX.Rp;
lin.Lp = LX.Lp;

% Doublet observers
lin.IpD = LX.IpD;
lin.rIpD = LX.rIpD;
lin.zIpD = LX.zIpD;
lin.bpD = LX.bpD;

%% constraining quantities in LX
lin.Co = LX2Co(L,LX); %constraint quantities [Ip,bp,li/qa] (only li for rzip for now)

%% Post-processing quantity linearizations
[dIydx, dFxdx, dFAdx, dFBdx, drAdx, dzAdx, dFndx, dBrndx, dBzndx, dFtdx, ~, dbpdx, dbpDdx, dlidx, dqAdx, dWkdx] = meqdFdI(L,LX);

dagdx = spdiags(L.xscal(L.ind.ixg),L.ind.ixg(1)-1,L.ng  ,L.nN);
dIadx = spdiags(L.xscal(L.ind.ixa),L.ind.ixa(1)-1,L.G.na,L.nN);
dIudx = spdiags(L.xscal(L.ind.ixu),L.ind.ixu(1)-1,L.G.nu,L.nN);

dBmdx = L.G.Bmx(:,L.lxy)*dIydx + L.G.Bma*dIadx + L.G.Bmu*dIudx;
dFfdx = L.G.Mfx(:,L.lxy)*dIydx + L.G.Mfa*dIadx + L.G.Mfu*dIudx;

% rIp, zIp, Ip variations
drIpdx = L.rry(:).'*dIydx;
dzIpdx = L.zzy(:).'*dIydx;
dIpdx  = sum(dIydx,1);

% Doublet observers
drIpDdx = zeros(L.nD,L.nN);
dzIpDdx = zeros(L.nD,L.nN);
dIpDdx  = zeros(L.nD,L.nN);
for ii=1:L.nD
  if ii == 1
    O = (LX.Opy(:)<= 1); % Include vacuum. This ensures sum(dIpDdx)==dIpdx.
  else
    O = (LX.Opy(:)==ii);
  end
  drIpDdx(ii,:) = (L.rry(:).*O).'*dIydx;
  dzIpDdx(ii,:) = (L.zzy(:).*O).'*dIydx;
   dIpDdx(ii,:) = (          O).'*dIydx;
end

%% Reduced equations

x0 = L.LX2x(LX);
u0 = [LX.Va(:);lin.Co;LX.IniD];

[res0,Jx,Ju,Jxdot] = L.codeFlin(L,LX);
[S,K,U,Udot,maskL,dxdxL,dxdu] = fgselim(L,Jx,Ju,Jxdot);

lin.x0L = x0(maskL);
lin.u0L = [u0;zeros(L.nC,1)]; % adding Codot to [Va;Co;IniD]
lin.S = S;
lin.K = K;
lin.U = U;
lin.Udot = Udot;
lin.res0L = res0(L.ind.irD); % Residual from evolutive equations
% Function to get reduced linear state
lin.LX2x = @(LX) LX2x(LX,L.LX2x,maskL);
% Store link between full and reduced linear states
lin.maskL = maskL;
lin.dxdxL = dxdxL;
lin.dxdu  = dxdu;
% Store jacobian for construction of preconditioner
lin.Jx = Jx;
lin.Ju = Ju;
lin.Jxdot = Jxdot;

%%

lin.dBmdxL = dBmdx*dxdxL;
lin.dFfdxL = dFfdx*dxdxL;
lin.dIadxL = dIadx*dxdxL;
lin.dIudxL = dIudx*dxdxL;
lin.dagdxL = dagdx*dxdxL;
lin.dIydxL = dIydx*dxdxL;

lin.dFxdxL = dFxdx*dxdxL;
lin.dFAdxL = dFAdx*dxdxL;
lin.dFBdxL = dFBdx*dxdxL;
lin.drAdxL = drAdx*dxdxL;
lin.dzAdxL = dzAdx*dxdxL;
lin.dFndxL = dFndx*dxdxL;
lin.dBrndxL = dBrndx*dxdxL;
lin.dBzndxL = dBzndx*dxdxL;
lin.dFtdxL = dFtdx*dxdxL;
lin.dbpdxL = dbpdx*dxdxL;
lin.dbpDdxL = dbpDdx*dxdxL;
lin.dlidxL = dlidx*dxdxL;
lin.dqAdxL = dqAdx*dxdxL;
lin.dWkdxL = dWkdx*dxdxL;

lin.drIpdxL = drIpdx*dxdxL;
lin.dzIpdxL = dzIpdx*dxdxL;
lin.dIpdxL  =  dIpdx*dxdxL;

% Doublet observers
lin.drIpDdxL = drIpDdx*dxdxL;
lin.dzIpDdxL = dzIpDdx*dxdxL;
lin.dIpDdxL  =  dIpDdx*dxdxL;

%%

lin.dBmdu = dBmdx*dxdu;
lin.dFfdu = dFfdx*dxdu;
lin.dIadu = dIadx*dxdu;
lin.dIudu = dIudx*dxdu;
lin.dagdu = dagdx*dxdu;
lin.dIydu = dIydx*dxdu;

lin.dFxdu = dFxdx*dxdu;
lin.dFAdu = dFAdx*dxdu;
lin.dFBdu = dFBdx*dxdu;
lin.drAdu = drAdx*dxdu;
lin.dzAdu = dzAdx*dxdu;
lin.dFndu = dFndx*dxdu;
lin.dBrndu = dBrndx*dxdu;
lin.dBzndu = dBzndx*dxdu;
lin.dFtdu = dFtdx*dxdu;
lin.dbpdu = dbpdx*dxdu;
lin.dbpDdu = dbpDdx*dxdu;
lin.dlidu = dlidx*dxdu;
lin.dqAdu = dqAdx*dxdu;
lin.dWkdu = dWkdx*dxdu;

lin.drIpdu = drIpdx*dxdu;
lin.dzIpdu = dzIpdx*dxdu;
lin.dIpdu  =  dIpdx*dxdu;

% Doublet observers
lin.drIpDdu = drIpDdx*dxdu;
lin.dzIpDdu = dzIpDdx*dxdu;
lin.dIpDdu  =  dIpDdx*dxdu;

%%

% dCodxL must be 0.
% dCodu  must be 1 for the corresponding index.
% From the linearization dCodxL~0, dCodu~1
% but the small error may accumulate for long simulations because its value 
% is being fed back to compute the time derivative in fgetkl(). 
% Hence, forced to be the analytic values.

for ii = 1:L.nC
  Coname = L.agconc{ii,3};
  Coii   = L.agconc{ii,4};
  dCoiidxL = sprintf('d%sdxL',Coname);
  dCoiidu  = sprintf('d%sdu' ,Coname);
  
  lin.(dCoiidxL)(Coii,        :) = 0;
  lin.(dCoiidu )(Coii,        :) = 0;
  lin.(dCoiidu )(Coii,L.G.na+ii) = 1;
end


%%

L.lin = lin;

if L.P.debug,  fprintf('done\n'); end

end

function x = LX2x(LX,fun,maskL)
  x = fun(LX);
  x = x(maskL);
end
