function [Ei,Vi,LYi] = fgeeig(L,varargin)
% FGEEIG Calculate unstable eigenvalues of system using fgess
% 
% [Ei,Vi,LYi] = fgeeig(L,varargin)
% Returns unstable eigenvalues (Ei) of system returned by fgess(L)
% eigenmodes (Vi) and array of perturbed equilibrium information (LYi)
% 
% optional parameters-value pairs: 
%  LX:     meq equilibrium. If given it won't be used the lin structure
%          inside L. Also the calculation will be faster, since it doesn't
%          need to compute the full state-space, but only the force
%          balanace linearisation and invert it.
%  nmodes: number of modes to compute, default: most unstable
%  imodes: specific mode indices to plot
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

assert(isfield(L,'Mee'),'missing Mee field in L, is this an fge structure?')

p=inputParser;
p.addParameter('LX',[]); % meq equilibrium
p.addParameter('nmodes',1); % number of modes to plot
p.addParameter('imodes',[]); % number of modes to plot
parse(p,varargin{:}); P = p.Results;

if isempty(P.imodes)
  nmodes = P.nmodes; imodes = 1:nmodes; % plot some modes
else
  imodes = P.imodes; nmodes = numel(imodes); % plot specific modes
end

if isempty(P.LX)
  % Compute state-space system corresponding to linearized equilibrium stored in L
  assert(isfield(L,'lin'),'missing lin field in L, did you run fgel()?')
  sys = fgess(L);
  A = sys.A;
  dxdxL =  L.lin.dxdxL;
else
  % Compute A matrix of state-space system corresponding to equilibrium stored in P.LX
  [~,Jx,Ju,Jxdot] = L.codeFlin(L,P.LX);

  [S,K,~,~,~,dxdxL,~] = fgselim(L,Jx,Ju,Jxdot);
  A = S\K;
end

[V,D] = eig(A);
[E,isort] = esort(diag(D));

%%
% init
Ei = zeros(1,nmodes);
Vi = zeros(size(V,1),nmodes);

for jj=1:nmodes
  ieig = imodes(jj);
  Ei(jj) = E(ieig);
  if L.P.debug>0
    fprintf('growth rate %2.2f\n',E(ieig));
  end
  
  if nargout > 1
    % Find dIy response to eigenvector direction  
    dIydx = [eye(L.ny),zeros(L.ny,L.ng+L.ne)].*L.xscal.';
    dIydV = dIydx*dxdxL;
    Veig = V(:,isort(ieig)); % eigenvector
    % Adjust eigenvector sign to make it unambiguous
    Veig = Veig*sign(sum(L.zzy(:).*(dIydV*Veig)));
    Vi(:,jj) = Veig; % store
    
    if nargout > 2
      dIy = reshape(dIydV*Veig,L.nzy,L.nry);
      
      dIe = Veig(1:L.ne);
      if isfield(L.G,'Mza')
        LY.Fz = reshape([L.G.Mza,L.G.Mzu]*dIe + L.G.Mzy*dIy(:),L.nzz,L.nrz);
      end
      LY.Fx = meqFx(L,dIy,dIe);
      LY.Ia = dIe(1:L.G.na); 
      LY.lB = false; [LY.rX,LY.zX] = deal([]);
      LY.Iy = reshape(dIy,L.nzy,L.nry);
      LY.Iu = dIe(L.G.na+(1:L.G.nu));
      LY.Iv = L.G.Tvu*LY.Iu;
      LY.Iy = dIy;
      LY.Ip = 0; % stresses that this is not a real plasma
      LY.t  = NaN;
      LYi(jj) = LY; %#ok<AGROW>
    end
  end
end
