function [Jx,Ju,Jxdot] = fgeFvacJac(L)
% FGEFVACJAC Vacuum part of fgeF jacobians
%
%   [Jx,Ju,Jxdot] = fgeFJacLin(L)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Identify algoNL variant
% xHasIy: state has Iy, rHasFx: residual has Fx
switch L.P.algoNL(end)
  case 'l' % all-nl
    xHasIy = true;
    rHasFx = false;
  case 'x' % all-nl-Fx
    xHasIy = false;
    rHasFx = true;
  case 'S' % Newton-GS
    xHasIy = false;
    rHasFx = false;
end

%% Scales
% state/input
xscalGS = L.xscal(L.ind.ixGS);
if L.isEvolutive
  % Ie is a state
  ixe = [L.ind.ixa,L.ind.ixu];
  ire = [L.ind.ira,L.ind.iru];
  xscale = L.xscal(ixe);
else
  % Ie is an input
  iue = [L.ind.iua,L.ind.iuu];
  xscale = 1;
end
% residual
resscalGS = L.resscal(L.ind.irGS);
% ag constraints are already rescaled
% cde resscal not needed here
if L.isEvolutive
  resscalc = L.resscal(      ire);
end

%% Allocate

if ~xHasIy && ~rHasFx && L.P.jacobian_sparse
  % Sparse jacobian, algoNL=Newton-GS
  % Ry~Iy+dlst(Fx)
  if L.P.icsint
    % Iy depends on all grid points
    nnz_dRydFx = L.nx*L.ny;
  else
    % Iy depends only on local + 12 points per domain from meqpdom stencil, dlst has 5-point stencil
    nnz_dRydFx = (5+12*L.nD)*L.ny;
  end
  % Add Fb residual, dRydag and ag + CDE constraints
  nb = L.nx - L.ny;
  nnz_ =  nnz_dRydFx + nb*(L.ny+1) + L.ng*(L.ng+2*L.nx);
  if L.isEvolutive
    % Add dRxdIe and circuit equations
    nnz_ = nnz_ + nb*L.ne + nnz(L.Tye) + L.ne*(L.nx+L.ne);
  end
  Jx = sparse([],[],[],L.nN,L.nN,nnz_);
else
  % Full jacobian
  Jx = zeros(L.nN,L.nN);
end
Ju = zeros(L.nN,L.nuN);
Jxdot = zeros(L.nrD,L.nN);

%% GS residuals
if xHasIy
  %   resGS = Iyp(Fx(Iy,Ie),ag) - Iy
  % dGS/dGS = dIyp/dIy - I, Iyp=0 for vacuum 
  Jx(L.ind.irGS,L.ind.ixGS) = -eye(L.ny);
  % dGS/d[ag,Ie] zero for vacuum
  % dGS/d[Va,Co,Ini] zero
elseif rHasFx
  %   resGS = L.Mxy*Iyp(Fx,ag) + L.Mxe*Ie - Fx
  Jx(L.ind.irGS,L.ind.ixGS) = -eye(L.nx);
  % dGS/dag zero for vacuum
  % dGS/dIe 
  if L.isEvolutive
    Jx(L.ind.irGS,ixe) = (resscalGS.*L.Mxe).*xscale.';
  else
    Ju(L.ind.irGS,iue) = (resscalGS.*L.Mxe);
  end
  % dGS/d[Va,Co,Ini]  zero
else
  %   resGS =   Txb*(Fb - Mby*Iyp - Mbe*Ie) ...
  %           + Txy*(rhsf*(Iyp + L.Tye*Ie) - dlst*Fx)

  % inner part of dRx_dFx is given by dlst contribution and
  % rhs_factor * dIy_dFx contribution
  dRx_dFx = L.Txy * L.dlst;

  % boundary part of dRx_dFx is given by the jacobian which dictates
  % how reference boundary changes with values on inner grid
  dRx_dFx = dRx_dFx + L.Txb * L.Txb.';

  % Add to Jx
  Jx(L.ind.irGS,L.ind.ixGS) = (resscalGS.*dRx_dFx).*xscalGS.';
  % dGS/dag zero for vacuum
  % dGS/dIe
  JGSe = - L.Txb * (sparse(L.Mbe)) ...
         + L.Txy * (spdiags(L.rhsf, 0, L.ny, L.ny) * L.Tye);
  if L.isEvolutive
    Jx(L.ind.irGS,ixe) = (resscalGS.*JGSe).*xscale.';
  else
    Ju(L.ind.irGS,iue) = (resscalGS.*JGSe);
  end
  % dGS/d[Va,Co,Ini]  zero
end

%% ag and CDE residuals
% In vacuum equivalent to ag=0
Jx([L.ind.irC,L.ind.irp],L.ind.ixg) = eye(L.ng);

%% Circuit equation residuals
if L.isEvolutive
  % Jx depends on value of dt, will output Jx value for 1/dt=0 such that
  % Jx(dt>0) = Jx(1/dt=0) + 1/dt*Jxdot
  
  % dcirc/dIe
  Jx(ire,ixe) = (resscalc.*diag(L.Re)).*xscale.';
  
  % dcirc/dVa
  Ju(L.ind.ira,L.ind.iua) = -diag(resscalc(1:L.G.na));
  
  % dcircdxdot
  irDe = ire - (L.nN-L.nrD);
  % dcirc/d[GS,Ie]dot
  if xHasIy
    JeGSdot = L.Mey;
    JeIedot = L.Mee;
  else
    JeGSdot = - (L.Mey./L.rhsf.') * L.dlst;
    JeIedot = L.Mee - L.Mey*L.Tye;
  end
  Jxdot(irDe,L.ind.ixGS) = (resscalc.*JeGSdot).*xscalGS.';
  % dcirc/dIedot
  Jxdot(irDe,      ixe ) = (resscalc.*JeIedot).*xscale.';
end

end
