function [Ps,dxdus,dxdxps] = fgeFJac_inverse_picard_approx(L,idts)
% FGEFJAC_INVERSE_PICARD_APPROX Approximation of inverse of fgeF jacobian based on Picard iteration
%
%   [Ps,dxdus,dxdxps] = fgeFJac_inverse_picard_approx(L,idts)
%
% This routine provides approximations of the inverse of the jacobian of
% the FGE operators with the assumption dIy/Fx=0, dCo/dFx=0
%
% Inputs:
%   L: FGE structure of consolidated inputs
%   idts: list of values of 1/dt for each slice to be considered
%
% Outputs:
%   Ps: cell array of function handles taking the values for each time slice
%       of Jxg as input and returning the approximate inverse of the Jacobian
%       as a function handle
%   dxdus: cell array of approximate values for each time slice of -Jx\Ju with
%       u=[Ie] for FGS and u=[Va] for FGE.
%   dxdxps: cell array of approximate value for each time slice of -Jx\Jxp(=1/dt*Jx\Jxdot).
%       This only applies to FGE, for FGS the matrices are empty.
%
% Note that the approximate values returned in dxdus and dxdxps values are
% independent of the value of x or u, so they can be computed only once.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

nts = numel(idts);
Ps = cell(nts,1);

% Indices into the state
ixGS =  L.ind.ixGS;
ixe  = [L.ind.ixa,L.ind.ixu];
% Indices into the resicual
irGS =  L.ind.irGS;
ire  = [L.ind.ira,L.ind.iru];
irDe = ire - (L.nN-L.nrD);

% State scale
xscalGS = L.xscal(ixGS);
xscale  = L.xscal(ixe);
% Residual scale
rscalGS = L.resscal(irGS);
rscalc  = L.resscal(ire);

if L.isEvolutive
  % FGE, u = [Va]
  na = L.G.na;
  dxdus  = repmat({zeros(L.nN,na  )},nts,1);
  dxdxps = repmat({zeros(L.nN,L.nN)},nts,1);
  for its = 1:nts
    % JGSe
    JGSe = L.Jx(irGS,ixe);
    % Jee
    if strcmp(L.P.algoNL,'all-nl')
      Jee = L.Jx(ire,ixe) + L.Jxdot(irDe,ixe).*idts(its);
    else
      % Tye part later cancels out
      Jee = rscalc.*(L.Mee.*idts(its) + diag(L.Re)).*xscale.';
    end
    iJee = Jee\eye(L.ne,L.ne);
    % JeGS
    JeGS = L.Jx(ire,ixGS) + L.Jxdot(irDe,ixGS).*idts(its);
    if strcmp(L.P.algoNL,'Newton-GS')
      % This enters the inverse of the Jx approximation
      Jey  = (rscalc.*(L.Mey./L.rhsf.')./rscalGS(L.lxy).').*idts(its);
    end
    % Inverse of approximate jacobian
    switch L.P.algoNL
      case {'all-nl','all-nl-Fx'}
        Ps{its} = @(Jxg) fgeF_picard_Prec(L,Jxg,JGSe,iJee,JeGS);
      case 'Newton-GS' 
        Ps{its} = @(Jxg) fgeF_picard_Prec(L,Jxg,JGSe,iJee,Jey );
    end
    % dxdu
    dxdus{its} (ire ,  : ) = iJee(:,1:na).*rscalc(1:na).';
    if ~strcmp(L.P.algoNL,'all-nl')
      dxdus{its} (irGS,  : ) = (1./xscalGS.*L.Mxe.*xscale.')*dxdus{its}(ire,:);
    end
    % dxdxp
    dxdxps{its}(ire ,ixGS) = (iJee*          JeGS           );
    if strcmp(L.P.algoNL,'all-nl')
      dxdxps{its}(ire ,ixe ) = (iJee*(rscalc.*L.Mee.*xscale.')).*idts(its);
    else
      dxdxps{its}(ire ,ixe ) = (iJee*(rscalc.*(L.Mee-L.Mey*L.Tye).*xscale.')).*idts(its);
      dxdxps{its}(irGS,  : ) = (1./xscalGS.*L.Mxe.*xscale.')*dxdxps{its}(ire,:);
    end
  end
else
  % FGS, u = [Ie], xdot = [], nts=1 is enforced
  % dxdu
  switch L.P.algoNL
    case 'all-nl'
      dxdus = {zeros(L.nN,L.ne)};
    case {'all-nl-Fx','Newton-GS'}
      dxdus = {[L.Mxe./xscalGS;zeros(L.ng,L.ne)]};
  end
  % dxdxp
  dxdxps = {zeros(L.nN,L.nN)};
  % Inverse of approximate jacobian
  Ps = {@(Jxg) fgeF_picard_Prec(L,Jxg,[],[],[])};
end
end

function P = fgeF_picard_Prec(L,Jxg,JGSe,iJee,JeGS)
% FGEF_PICARD_PREC Compute inverse of approximate Jacobian as function handle

iGS =  L.ind.ixGS;
ig  =  L.ind.ixg; % is also [L.ind.irC;L.ind.irp]
ie  = [L.ind.ixa,L.ind.ixu];

% Process Jxg
iJgg = Jxg(ig,:)\eye(L.ng);
JGSg = Jxg(iGS,:);

switch L.P.algoNL
  case {'all-nl','all-nl-Fx'}
    P = @(r) all_nl_Prec(r,ig,iGS,iJgg,JGSg,ie,iJee,JeGS,JGSe);
  case 'Newton-GS'
    Iyf = L.resscal(L.ind.irGS(L.lxy)).*L.rhsf.*L.Fx0; % convert residual to Iy/Fx0
    P = @(r) GS_Prec(r,ig,iGS,iJgg,JGSg,ie,iJee,JeGS,JGSe,...
      L.lxy,Iyf,[L.nzy,L.nry],L.cx,L.cq,L.cr,L.cs,L.ci,L.co);
end
end

function dx = all_nl_Prec(r,ig,iGS,iJgg,JGSg,ie,iJee,JeGS,JGSe)
% ALL_NL_PREC Apply inverse of approximate jacobian for algoNL=all-nl or all-nl-Fx
dxg = iJgg*r(ig,:);
dxGS = -r(iGS,:)+JGSg*dxg;
if ~isempty(ie)
  % FGE
  dxe = iJee*(r(ie,:)-JeGS*dxGS);
  if any(JGSe)
    dxGS = dxGS + JGSe*dxe;
  end
else
  % FGS
  dxe = [];
end
dx = [dxGS;dxg;dxe];
end

function dx = GS_Prec(r,ig,iGS,iJgg,JGSg,ie,iJee,JeGS,JGSe,...
                      lxy,Iyf,szy,cx,cq,cr,cs,ci,co)
% GS_PREC Apply inverse of approximate jacobian for algoNL=Newton-GS
dxg = iJgg*r(ig,:);
drhs = r(iGS,:) - JGSg*dxg;
if ~isempty(ie)
  % FGE
  dxe = iJee*(r(ie,:)+JeGS*drhs(lxy,:));
  drhs = drhs - JGSe*dxe;
else
  % FGS
  dxe = [];
end
nr = size(r,2);
dxx = gszrmex(drhs(~lxy,:),reshape(-drhs(lxy,:)./Iyf,[szy,nr]),cx,cq,cr,cs,ci,co,0);
dx = [reshape(dxx,[],nr);dxg;dxe];
end
