% FGEF Physics operator for FGE/FGS
% [res, LY, Jx, Ju, Jxdot, rowmask] = fgeF(x,L,LX,LYp,opts,aux[,u,xdot])
% Computes F(x) operator for FGE, x s.t. F(x) = 0 is the solution
% Returns the (scaled) residual of
% * GS equation res(L.ind.irGS)
% * Constraint equations res(L.ind.irC)
% * CDE equation (optional when running FGE with CDE) res(L.ind.irp)
% * Circuit equations res([L.ind.ira,L.ind.iru])
%
% x is the nonlinear state vector:
%   - for algoNL=all-nl,                x = [Iy(:);ag(:);Ie(:)]./L.xscal
%   - for algoNL=all-nl-Fx or Newton-GS x = [Fx(:);ag(:);Ie(:)]./L.xscal
%
% L is the ancilliary data structure
% LX contains input data
% LYp contains the previous equilibrium.
% for FGS, LYp=LX, for FGE, LYp and LX will be different
%
% opts is a structure generated by OPTSF specifying what operations are
% to be done
%
% aux is a cell array with externally provided quantities:
%  - aux{1} is the boundary flux due to external currents
%  - aux{2} is the current density due to external currents in the
%  computational grid
%  - aux{3} is the value of the Opy matrix to be imposed to ease meqopt
%  convergence (LIU or FBT only)
%  - aux{4} is the value of dz (LIU stabilization parameter)
%  - aux{5} is the value of Opy for use with dz
%  - aux{6} is the value of doSQP from meqopt, when false only the picard
%  approximate value of the jacobians is required
% 
% The last 2 arguments u and xdot are optional and when provided supersedes 
% the external parameters provided via LX or LYp. It should only be used to
% compute Ju and Jxdot using finite differences and for testing purposes.
%
% For FGS:
% Jx is the Jacobian w.r.t. x
% Ju is the Jacobian w.r.t. u = [Ia;Iu;Co]
% For FGE
% Jx is the Jacobian w.r.t. x
% Ju is the Jacobian w.r.t. u=[Va;Co;Ini]
% Jxdot is the Jacobian w.r.t xdot (and rows corresponding to static
%   equations are ommitted).
%
% See the help section of FGEL for a discussion on he convention chosen for
% the jacobians and their relation to the linearized equation.
%
% rowmask is a vector with non-zero values for rows where Jx has only one
% non-zero element on the diagonal. And the value is the value of the
% diagonal element.
%
% See also: optsF fgeFJac fgeFlin fgel
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [res, LY, Jx, Ju, Jxdot, rowmask] = fgeF(x,L,LX,LYp,opts,aux,u,xdot)

% Check
if nargin < 4
  error('fgeF requires at least 4 input arguments');
end

% Selections of outputs
if nargin<5
  opts = optsF;
end

% external quantities
auxHasFbe = false;
auxHasOpy = false;
auxHasdz  = false;
picardJac = false;
if nargin>=6 && iscell(aux) && numel(aux)>1
  auxHasFbe = ~isempty(aux{1}) && ~isempty(aux{2});
  auxHasOpy = numel(aux)>2 && ~isempty(aux{3});
  auxHasdz  = numel(aux)>3 && ~isempty(aux{4});
  picardJac = numel(aux)>5 && ~aux{6};
end

%% Identify algoNL variant
% xHasIy: state has Iy, rHasFx: residual has Fx
switch L.P.algoNL(end)
  case 'l' % all-nl
    xHasIy = true;
    rHasFx = false;
  case 'x' % all-nl-Fx
    xHasIy = false;
    rHasFx = true;
  case 'S' % Newton-GS
    xHasIy = false;
    rHasFx = false;
end

%% Update LX and LYp based on u (for finite difference version of Ju)
if nargin > 6
  if nargin < 8
    xdot = zeros(L.nN,1);
  end
  [LX,LYp] = fgeF_u2LXLY(u,xdot,L,LX,LYp,xHasIy);
end

%% init
% Indicates whether jacobian of static equations will be computed
dojacstat = opts.dojacx || opts.dojacu || opts.dojacF;
% Indicates whether any form of jacobian will be computed
dojac     = dojacstat || opts.dojacxdot;

res = zeros(L.nN,1);
if opts.dopost, LY = LYp; else, LY = []; end
% Use pre-filled jacobians
if opts.dojacx,    Jx    = L.Jx;    else, Jx    = []; end
if opts.dojacu,    Ju    = L.Ju;    else, Ju    = []; end
if opts.dojacxdot, Jxdot = L.Jxdot; else, Jxdot = []; end
rowmask = zeros(L.nN,1);

%% Time step
if L.isEvolutive
  dt = LX.t - LYp.t;
  if dt == 0
    idt = 0;
  else
    idt = 1./dt;
  end
end

%% Scales
% state/input
xscalGS = L.xscal(L.ind.ixGS);
xscalg  = L.xscal(L.ind.ixg);
if L.isEvolutive
  % Ie is a state
  ixe = [L.ind.ixa,L.ind.ixu];
  ire = [L.ind.ira,L.ind.iru];
  xscale = L.xscal(ixe);
else
  % Ie is an input
  iue = [L.ind.iua,L.ind.iuu];
  xscale = 1;
end
% residual
resscalGS = L.resscal(L.ind.irGS);
% ag constraints are already rescaled
% cde resscal if needed
if L.np
  resscalp = L.resscal(L.ind.irp);
end
if L.isEvolutive
  resscalc = L.resscal(      ire);
end

%% Extract NL unknowns
xSI = L.xscal .* x;
if xHasIy
  Iy = reshape(xSI(L.ind.ixGS), L.nzy, L.nry);
else
  Fx = reshape(xSI(L.ind.ixGS), L.nzx, L.nrx);
end
ag = reshape(xSI(L.ind.ixg), L.ng , 1    );
if L.isEvolutive
  % Ie is a state
  Ia = xSI(L.ind.ixa);
  Iu = xSI(L.ind.ixu);
else
  % Ie is an input
  Ia = LX.Ia;
  Iu = LX.Iu;
end
Ie = [Ia;Iu];

if L.nD>1
  Ip = sum(LX.IpD);
else
  Ip = LX.Ip;
end

% Sign of Ip
sIp = sign(Ip);
rBt = LX.rBt;

%% smaller than Ipmin flag
assign_LXIy = abs(Ip) < L.P.Ipmin;
% If Iy is assigned then the jacobian is the one stored in L (corresponding
% to the vacuum case). Some steps in the computation of the jacobian can be skipped.
if assign_LXIy && L.np
  error('LX.Iy can only be assigned if no CDE is selected');
end

%% External currents contributions to meqFx
% For FGE or if aux(1:2) are not supplied, compute them
if L.isEvolutive || ~auxHasFbe
  Fbe = L.Mbe*Ie;
  Iyie = reshape(L.Tye*Ie,L.nzy,L.nry);
else
  Fbe  = aux{1};
  Iyie = aux{2};
end

%% Get missing Fx or Iy
if xHasIy
  if auxHasdz
    dz = aux{4};
    if L.nD>1 && ~isempty(aux{5})
      Opy = aux{5};
      IyD = zeros(L.nzy,L.nry,L.nD);
      IyD(:,:,1) = Iy.*(Opy <= int8(1));
      for iD = 2:L.nD
        IyD(:,:,iD) = Iy.*(Opy == int8(iD));
      end
    else
      IyD = zeros(L.nzy,L.nry,L.nD);
      IyD(:,:,1) = Iy;
    end
    Fx = meqFx(L,IyD,Ie,{Fbe,Iyie},dz,false,zeros(L.nh,1));
  else
    Fx = meqFx(L,Iy ,Ie,{Fbe,Iyie});
  end
else
  Iy = - reshape((L.dlst*Fx(:))./L.rhsf,L.nzy,L.nry) - Iyie;
end

%% Plasma domain and current distribution
if dojac && ~picardJac
  [rA,zA,FA,dr2FA,dz2FA,drzFA,rX,zX,FX,dr2FX,dz2FX,drzFX,...
    rB,zB,FB,lB,lX,Opy,F0,F1,pdomstat,msg,id,dF0dFx,dF1dFx,ixI] = ...
    meqpdom(Fx,sIp,L.P.isaddl,L);
else
  [rA,zA,FA,dr2FA,dz2FA,drzFA,rX,zX,FX,dr2FX,dz2FX,drzFX,...
    rB,zB,FB,lB,lX,Opy,F0,F1,pdomstat,msg,id] = ...
    meqpdom(Fx,sIp,L.P.isaddl,L);
  if dojac && picardJac
    % Dummy values as derivatives wrt Fx are not used
    if L.P.icsint
      dF0dFx = zeros(L.nD,L.nx);
    else
      dF0dFx = sparse([],[],[],L.nD,L.nx);
    end
    dF1dFx = dF0dFx;
    ixI = repmat(L.nzx+1,L.nD,1); % First y point
  end
end
% Enforce Opy value (freeze domains in fbtt/liut)
if auxHasOpy, Opy = aux{3}; end
nA = numel(FA); % number of axis domains
nB = numel(FB); % number of boundaries = number of active domains

%% Handle meqpdom failures
if ~pdomstat
  meqmsge('w',mfilename,L.P.tokamak,LX.t,0,LX.shot,msg,id);
  res(:)=NaN;
  if opts.doplot || opts.dopost
    resy = NaN; resC = NaN; resp = NaN; rese = NaN; resFx = NaN;

    LY = fgepost(L,LX,LYp,ag,...
      Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lB,lX, ...
      rX,zX,FX,dr2FX,dz2FX,drzFX, ...
      rBt,Ia,Iu,Iy,Opy,F0,F1,resy,resC,resp,rese,resFx);
    LY.isconverged = false;

    if opts.doplot, fgedebugplot(L,LY,LYp,res,x); end
  elseif any(strcmp(L.code,{'fbt','liu'}))
    % Return Opy
    LY = struct('shot',LX.shot,'t',LX.t,'Opy',Opy);
  end

  % Value and Jacobian of NL measurements
  if strcmp(L.code,'liu')
    % DML measurement
    if L.P.idml
      % Synthetic value
      LY.Xt = NaN;
    end
    % Regularization constraints
    if any(L.P.wreg) && L.nq
      % Synthetic value
      LY.Xq = NaN(L.nq,1);
    end
    % Inequality constraints
    if L.P.ipm && L.nc
      % Synthetic value
      LY.Xc = NaN(L.nc,1);
    end
    % Vertical stabilization parameter
    if L.P.stabz
      LY.Xr = NaN(L.nr,1);
    end
  end
  return
end

%% Plasma current distribution from basis function coefficients
if ~assign_LXIy
  [Tyg,TpDg,ITpDg] = L.bfct(1,L.bfp,Fx,F0,F1,Opy,L.ry,L.iry);
  Iy1 = reshape(Tyg*ag,L.nzy,L.nry);
else
  Iy1 = LX.Iy; % Prescribe Iy from LX
  Tyg = zeros(L.ny,L.ng);
  TpDg = zeros(L.nD,L.ng);
  ITpDg = zeros(L.nD,L.ng);
end

%% Assemble dIy/dFx
if dojacstat
  if ~assign_LXIy && ~picardJac
    % Iy depends non-linearly on Fx

    % Evaluates basis function gradient value on Fx, masked by Opy.
    % the bf mode 11 implements jacobians for Tyg, ITyg,
    % note that dITygdF = Tyg, which is why we dont need it explicitely
    [dTygdFy,dTygdF0,dTygdF1,dITygdF0,dITygdF1] = L.bfct(11,L.bfp,Fx,F0,F1,Opy,L.ry,L.iry);

    % Domains where there is some current
    mask = false(L.ny,1);
    active_domains = find(any(L.TDg.*ag.',2));
    for iD = active_domains.'
      mask = mask | (Opy(:) == int8(iD));
    end

    % dIypdFx is needed for Jx computation except for algoNL=all-nl with CS
    % interpolation or for Ju computation of static cases when
    % algoNL=all-nl and no CS interpolation
    hasdIypdFx = (opts.dojacx && ~(xHasIy && L.P.icsint)) || (opts.dojacu && ~L.isEvolutive && xHasIy && ~L.P.icsint);

    % Derivatives of Iyp
    if hasdIypdFx
      [dIypdFy,dIypdF0,dIypdF1,dIypdFx] = meqIyJac(L,ag,mask,dTygdFy,dTygdF0,dTygdF1,dF0dFx,dF1dFx);
    else
      [dIypdFy,dIypdF0,dIypdF1        ] = meqIyJac(L,ag,mask,dTygdFy,dTygdF0,dTygdF1);
    end

    if xHasIy
      % Assemble dIydIy/dIydIe when needed
      hasdIydIy = opts.dojacx;
      hasdIydIe = ((opts.dojacx && L.isEvolutive) || (opts.dojacu && ~L.isEvolutive));
      if hasdIydIy || hasdIydIe
        maskx = logical(L.Txy*mask);
      end

      % dIydIy
      if hasdIydIy
        if hasdIypdFx
          dIydIy = dIypdFx*L.Mxy;
        else
          dIydIy = zeros(L.ny,L.ny);
          dIydIy(mask,:) = dIypdFy.*L.Mxy(maskx,:) + dIypdF0*(dF0dFx*L.Mxy) + dIypdF1*(dF1dFx*L.Mxy);
        end
      end

      % dIydIe
      if hasdIydIe
        if hasdIypdFx
          dIydIe = dIypdFx*L.Mxe;
        else
          dIydIe = zeros(L.ny,L.ne);
          dIydIe(mask,:) = dIypdFy.*L.Mxe(maskx,:) + dIypdF0*(dF0dFx*L.Mxe) + dIypdF1*(dF1dFx*L.Mxe);
        end
      end
    end
  elseif assign_LXIy
    % Iy is independent of Fx

    % Iyp derivatives only needed for function handle
    if opts.dojacF
      mask = false(L.ny,1);
      nyp = 0;
      dIypdFy = zeros(nyp,1);
      dIypdF0 = zeros(nyp,L.nD);
      dIypdF1 = zeros(nyp,L.nD);
    end
  else
    % Neglecting dependence of Iy on Fx
    mask = false(L.ny,1);
    nyp = 0;
    dIypdFy = zeros(nyp,1);
    dIypdF0 = zeros(nyp,L.nD);
    dIypdF1 = zeros(nyp,L.nD);
    dITygdF0 = zeros(L.ny,L.ng,L.nD);
    dITygdF1 = dITygdF0;
  end
end

%% GS residuals
if xHasIy
  % all-nl: residual = Iy-Iy(previous)
  residualGS = resscalGS.*(Iy1(:) - Iy(:));
elseif rHasFx
  % all-nl-Fx: residual = Fx-Fx(previous)
  Fx1 = meqFx(L,Iy1,Ie,{Fbe,Iyie});
  residualGS = resscalGS.*(Fx1(:) - Fx(:));
else
  % Newton-GS: residual = [Fb-Fb(previous),Iy-Iy(previous)]

  % init Fx background residual array (computations of residual on y grid
  % and on the boundary differ)
  residualGS = zeros(L.nx,1);

  % compute residual for inner grid Fx (basically the residual of GS eq.)
  residualGS(L.lxy) = L.rhsf .* (Iy1(:) - Iy(:));

  % compute boundary residual for Fx; contribution of Iy is computed
  % using Mby instead of meqfbp. This is because the jacobian computation
  % also uses Mby and it is faster than meqfbp on TCV data
  residualGS(~L.lxy) = Fx(~L.lxy) - meqfbp(Iy1,L) - Fbe;

  % Rescale
  residualGS = resscalGS.*(residualGS);
end

%% ag constraint residuals
if dojacstat
  % Evaluate meqagcon residuals and all their derivatives with respect to
  % its inputs, dresdag and dresdCo are already returned here
  [residualag, dres_dF0,dres_dF1,dresdag,dres_dFx,dres_dTpDg,dres_dITpDg,...
      dres_drA, dres_ddr2FA, dres_ddz2FA, dres_ddrzFA, dresdCo] = ...
      meqagcon(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg);

  if ~picardJac
    % Derivatives of flux hessian at magnetic axis
    d = -1; % default: no derivatives needed
    if any(dres_ddr2FA(:)) || any(dres_ddrzFA(:)) || any(dres_ddz2FA(:)), d = 2;
    elseif any(dres_drA(:)),                                              d = 1;
    end
    if d>0 && L.P.icsint
      [ ~, ~, drAdFx, ddz2FAdFx, ddr2FAdFx, ddrzFAdFx] = ...
        asxycsJac(Fx,zA,rA,dz2FA,dr2FA,drzFA,L,d);
    elseif d>0
      [ ~, ~, drAdFx, ddz2FAdFx, ddr2FAdFx, ddrzFAdFx] = ...
        asxyJac(Fx, L.dzx, L.drx, L.idzx, L.idrx, ixI(1:nA),d);
    end

    % Group Fx derivatives
    dresdFx = dres_dFx;
    % F0
    if any(   dres_dF0(:))
      dresdFx = dresdFx +    dres_dF0 *    dF0dFx;
    end
    % F1
    if any(   dres_dF1(:))
      dresdFx = dresdFx +    dres_dF1 *    dF1dFx;
    end
    % TpDg
    if any( dres_dTpDg(:))
      [~,~,~, dTpDgdFx] = meqbfct1Jac(L,nB,Opy,dTygdFy, dTygdF0, dTygdF1,dF0dFx,dF1dFx);
      dresdFx = dresdFx +  dres_dTpDg *  dTpDgdFx;
    end
    % ITpDg
    if any(dres_dITpDg(:))
      [~,~,~,dITpDgdFx] = meqbfct1Jac(L,nB,Opy, Tyg   ,dITygdF0,dITygdF1,dF0dFx,dF1dFx);
      dresdFx = dresdFx + dres_dITpDg * dITpDgdFx;
    end
    % rA
    if d>0
      dresdFx = dresdFx +  dres_drA   *   drAdFx;
    end
    % dr2FA, drzFA, dz2FA
    if d>1
      dresdFx = dresdFx + (dres_ddr2FA*ddr2FAdFx + ...
        dres_ddrzFA*ddrzFAdFx + ...
        dres_ddz2FA*ddz2FAdFx);
    end
  else
    dresdFx = zeros(L.nC,L.nx);
  end
else
  residualag = ...
    meqagcon(L,LX,F0,F1,rA,dr2FA,dz2FA,drzFA,ag,Fx,Opy,TpDg,ITpDg);
end

%% CDE constraint residuals
if dojac
  % meqcde returns empty residuals and jacobians if np==0
  [residualcde,dcdedFx,dcdedag,dcdedIy,dcdedF0,dcdedF1,dcdedIe,dcdedIni,...
    dcdedFxdot,dcdedagdot,dcdedIydot,dcdedF0dot,dcdedF1dot,dcdedIedot] = ...
    meqcde(L,LX,LYp,Fx,ag,Iy,F0,F1,Ie,Opy);

  % Rescaling
  if L.np
    residualcde = resscalp.*residualcde;
  end
else
  residualcde = meqcde(L,LX,LYp,Fx,ag,Iy,F0,F1,Ie,Opy);
end

%% Circuit equation residuals
if L.isEvolutive
  % Extract controller actions
  Ve = zeros(L.ne,1);
  Ve(1:L.G.na) = LX.Va;

  Fedot = (L.Mee*(Ie-[LYp.Ia;LYp.Iu]) + L.Mey*(Iy(:)-LYp.Iy(:)))*idt;
  residualcirc = (Fedot + L.Re.*Ie - Ve); % residual for conductor equation

  % Rescaling
  residualcirc = resscalc.*residualcirc;

  if dojac
    % Jacobians
    dcircdIy = L.Mey*idt;
    dcircdIe = L.Mee*idt + diag(L.Re);
  end
else
  residualcirc = zeros(0,1);
end

%% Collect residuals
res = [residualGS(:);residualag(:);residualcde(:);residualcirc(:)];

%% Assemble Jx jacobian
if opts.dojacx && ~assign_LXIy
  % %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % GS residual
  if xHasIy % 'all-nl'
    % Subtract identity matrix to get jacobian of residual
    dIydIy(1:L.ny+1:end) = dIydIy(1:L.ny+1:end)-1;
    Jx(L.ind.irGS,L.ind.ixGS) = dIydIy; % No rescaling necessary since resscalGS.*xscalGS.' = ones(ny)

    Jx(L.ind.irGS,L.ind.ixg) = (resscalGS.* Tyg  ).*xscalg.';

    if L.isEvolutive
      Jx(L.ind.irGS,ixe) = (resscalGS.*dIydIe).*xscale.';
    end
  elseif rHasFx % 'all-nl-Fx'
    % Compute dFxdFx from dIydFx
    if L.P.icsint
      % dIypdFx is dense but has only rows where Opy>0
      dFxdFx = L.Mxy(:,mask)*dIypdFx;
    else
      % dIypdFx is a sparse matrix of size [ny,nx]
      dFxdFx = L.Mxy*dIypdFx;
    end
    dFxdag = L.Mxy*Tyg;

    % Subtract identity matrix to get jacobian of residual
    dFxdFx(1:L.nx+1:end) = dFxdFx(1:L.nx+1:end)-1;
    Jx(L.ind.irGS,L.ind.ixGS) = dFxdFx; % No rescaling necessary since resscalGS.*xscalGS.' = ones(nx)

    Jx(L.ind.irGS,L.ind.ixg ) = (resscalGS.*dFxdag).*xscalg.';

    % Dependence over Ie matches the vacuum case
  else % 'Newton-GS'
    % Compute dIy_dFx
    if L.P.icsint
      % dIypdFx is dense but has only rows where Opy>0
      dIy_dFx = zeros(L.ny,L.nx);
      dIy_dFx(mask,:) = dIypdFx; % Check if this can be done faster
    else
      % dIypdFx is a sparse matrix of size [ny,nx]
      dIy_dFx = dIypdFx;
    end

    % compute reference boundary jacobians with Mby (for Rb = Fb - Fb_ref)
    % keep the Iy Jacobians in sparse format although Mby is dense
    dFb_dFx = meqfbp(dIy_dFx,L);
    dFb_dag = meqfbp(Tyg,L);

    % this is the fastest way to apply the rhs factor to the Iy Jacobian
    drhs_Iy_dFx = spdiags(L.rhsf, 0, L.ny, L.ny) * dIy_dFx;

    % inner part of dRx_dFx is given by dlst contribution and
    % rhs_factor * dIy_dFx contribution
    dRx_dFx = L.Txy * drhs_Iy_dFx;

    % boundary part of dRx_dFx is given by the jacobian which dictates
    % how reference boundary changes with values on inner grid
    dRx_dFx = dRx_dFx - L.Txb * dFb_dFx;

    % inner part of dRx_dag is given by how ag changes Iy which changes the
    % rhs of the GS equation
    dRx_dag = L.Txy * (L.rhsf .* Tyg);

    % the boundary part of dRx_dag is given by how Fb_ref changes with
    % Iy times how Iy changes with ag
    dRx_dag = dRx_dag - L.Txb * dFb_dag;

    % Add to Jx
    Jx(L.ind.irGS,L.ind.ixGS) = Jx(L.ind.irGS,L.ind.ixGS) + (resscalGS.*dRx_dFx).*xscalGS.';
    Jx(L.ind.irGS,L.ind.ixg ) =                             (resscalGS.*dRx_dag).*xscalg.';
    
    % Dependence over Ie matches the vacuum case
  end

  % %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % ag residual
  if any(dresdFx(:)) && xHasIy
    % assemble full residual jacobian w.r.t. Iy
    Jx(L.ind.irC,L.ind.ixGS) = (dresdFx * L.Mxy).*xscalGS.';

    % assemble full residual jacobian w.r.t. Ie
    if L.isEvolutive
      Jx(L.ind.irC,ixe) = (dresdFx * L.Mxe).*xscale.';
    end
  elseif any(dresdFx(:))
    Jx(L.ind.irC,L.ind.ixGS) = dresdFx.*xscalGS.';

    % No direct dependence on Ie
  end

  % Residuals are already rescaled
  Jx(L.ind.irC,L.ind.ixg) = dresdag.*xscalg.';

  % %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % CDE residual
  if L.np
    % static part
    % Group Fx derivatives
    dcdedFx = dcdedFx + (dcdedF0*dF0dFx + dcdedF1*dF1dFx);
    if xHasIy
      drescdedGS = dcdedIy + dcdedFx * L.Mxy;
      drescdedIe = dcdedIe + dcdedFx * L.Mxe;
    else
      drescdedGS = dcdedFx - (dcdedIy./L.rhsf.') * L.dlst;
      drescdedIe = dcdedIe - (dcdedIy) * L.Tye;
    end
    drescdedag = dcdedag;

    % Jx
    Jx(L.ind.irp,L.ind.ixGS) = (resscalp.*drescdedGS).*xscalGS.';
    Jx(L.ind.irp,L.ind.ixg ) = (resscalp.*drescdedag).*xscalg.';
    
    % evolutive part
    if L.isEvolutive
      Jx(L.ind.irp,      ixe ) = (resscalp.*drescdedIe).*xscale.';
    end
  end

  % %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % Circuit equations residual
  if L.isEvolutive
    % Linear but misses part proportional to 1/dt
    irDe = ire - (L.nN-L.nrD);
    Jx(ire,:) = Jx(ire,:) + idt*L.Jxdot(irDe,:);
  end
end

%% Ju
if opts.dojacu && ~assign_LXIy
  % Only derivatives of non-linear terms need be updated
  % For FGS:
  %  - dIydIe [all-nl only]
  %  - dCodIe [all-nl only]
  %  - dCodCo
  % For FGE:
  %  - dCodCo
  %  - dcdedIni
  
  % GS residual
  if xHasIy && ~L.isEvolutive % FGS, all-nl
    Ju(L.ind.irGS,iue) = (resscalGS.*dIydIe).*xscale.';
  end
  
  % ag residual
  if xHasIy && ~L.isEvolutive % FGS, all-nl
    Ju(L.ind.irC ,iue) = (dresdFx*L.Mxe).*xscale.';
  end
  Ju(L.ind.irC,L.ind.iuC) = dresdCo;
  
  % CDE residual
  if L.np && L.isEvolutive % FGE, with CDE
    % CDE residual
    drescdedIni = dcdedIni;
    % Ini
    Ju(L.ind.irp,L.ind.iuni) = resscalp.*drescdedIni;
  end
  
  % Circuit equation residuals
  % Matches vacuum case, nothing to do
end

%% Jxdot
if opts.dojacxdot && ~assign_LXIy && L.isEvolutive
  % Only derivatives of non-linear terms need be updated
  %  - dcdedxdot
  
  % CDE residuals
  if L.np
    % Group Fx derivatives
    dcdedFxdot = dcdedFxdot + (dcdedF0dot*dF0dFx + dcdedF1dot*dF1dFx);
    if xHasIy
      drescdedGSdot = dcdedIydot + dcdedFxdot * L.Mxy;
      drescdedIedot = dcdedIedot + dcdedFxdot * L.Mxe;
    else
      drescdedGSdot = dcdedFxdot - (dcdedIydot./L.rhsf.') * L.dlst;
      drescdedIedot = dcdedIedot - (dcdedIydot) * L.Tye;
    end
    drescdedagdot = dcdedagdot;
    % xdot
    irDp = L.ind.irp - (L.nN-L.nrD);
    Jxdot(irDp,L.ind.ixGS) = (resscalp.*drescdedGSdot).*xscalGS.';
    Jxdot(irDp,L.ind.ixg ) = (resscalp.*drescdedagdot).*xscalg.';
    Jxdot(irDp,      ixe ) = (resscalp.*drescdedIedot).*xscale.';
  end
  
  % Circuit equation residuals
  % Matches vacuum case, nothing to do
end

%% Jacobian function
if opts.dojacF
  if ~L.isEvolutive
    Jx = @(dx) fgeFJac(dx,L,xHasIy,rHasFx,...
                       ... % meqpdom jacobians
                       dF0dFx,dF1dFx,...
                       ... % Iy jacobians
                       mask,dIypdFy,dIypdF0,dIypdF1,Tyg,...
                       ... % meqagcon jacobians
                       dresdFx,dresdag,...
                       ... % meqcde jacobians
                       dcdedIy,dcdedIe,dcdedFx,dcdedF0,dcdedF1,dcdedag);

  else
    Jx = @(dx) fgeFJac(dx,L,xHasIy,rHasFx,...
                       ... % meqpdom jacobians
                       dF0dFx,dF1dFx,...
                       ... % Iy jacobians
                       mask,dIypdFy,dIypdF0,dIypdF1,Tyg,...
                       ... % meqagcon jacobians
                       dresdFx,dresdag,...
                       ... % meqcde jacobians
                       dcdedIy,dcdedIe,dcdedFx,dcdedF0,dcdedF1,dcdedag,...
                       ... % circuit equation jacobians
                       dcircdIy,dcircdIe);
  end
end

%% Mask for rows of diagonal matrix
if assign_LXIy
  % Assigned Iy
  if xHasIy || rHasFx % all-nl, all-nl-Fx
    rowmask(L.ind.irGS) = -1;
  end
  rowmask([L.ind.irC,L.ind.irp]) = 1; % [resC,resp] = ag./ag0
else
  % GS equation
  if xHasIy % algoNL=all-nl
    % Identify grid points where there cannot be any Iy (outside domains with bfs)
    ymask = (Opy(:) == 0);
    for iD = 1:L.nD
      if any(L.TDg(iD,:)), continue; end
      ymask = ymask | (Opy(:) == int8(iD));
    end
    rowmask(L.ind.irGS) = -double(ymask);
  end
  % Direct constraint for ag(i) which is also the i-th constraint (not necessarily the case with CDEs)
  rowmask(L.ind.irC) = strcmp(L.agconc(:,3),'ag') & ([L.agconc{:,4}] == 1:L.nC).';
end
if L.isEvolutive && (dt==0)
  % If dt==0, circuit equations are diagonal Re*Ie - Ve = 0
  rowmask(      ire) = (resscalc.*L.Re.*xscale);
end

%% Add outputs
if opts.dopost || opts.doplot || opts.dodisp
  % collect residuals (TODO: add normalization or not)
  resy  = norm(Iy1 - Iy);
  resC  = norm(residualag);
  resp  = norm(residualcde);
  rese  = norm(residualcirc);
  if ~rHasFx
    Fx1 = meqFx(L,Iy1,Ie,{Fbe,Iyie}); % Poloidal flux residual w.r.t. new Iy
  end
  resFx = norm(Fx1 - Fx);

  LY = fgepost(L,LX,LYp,ag,...
    Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lB,lX, ...
    rX,zX,FX,dr2FX,dz2FX,drzFX, ...
    rBt,Ia,Iu,Iy,Opy,F0,F1,resy,resC,resp,rese,resFx);
elseif any(strcmp(L.code,{'fbt','liu'}))
  % Return Opy
  LY = struct('shot',LX.shot,'t',LX.t,'Opy',Opy);
end

% Value and Jacobian of NL measurements
if strcmp(L.code,'liu')
  if ~L.ndz
    IyD = [];
    dz = [];
  end
  if dojacstat && ~assign_LXIy
    % General case with jacobian
    LY = liuNLmeas(L,LX,LY,Iy,ag,F0,F1,rA,rB,nA,nB,ITpDg,IyD,Ie,dz,...
      dojacstat,picardJac,xHasIy, ...
      Fx,zA,zB,dr2FA,dz2FA,drzFA,dr2FX,dz2FX,drzFX,ixI,dF0dFx,dF1dFx,...
      assign_LXIy,Opy,Tyg,dITygdF0,dITygdF1);
  elseif dojacstat
    % |Ip|<Ipmin with jacobian - Avoids allocating and passing unused derivatives
    LY = liuNLmeas(L,LX,LY,Iy,ag,F0,F1,rA,rB,nA,nB,ITpDg,IyD,Ie,dz,...
      dojacstat,picardJac,xHasIy, ...
      Fx,zA,zB,dr2FA,dz2FA,drzFA,dr2FX,dz2FX,drzFX,ixI,dF0dFx,dF1dFx);
  else
    % General case no jacobian
    LY = liuNLmeas(L,LX,LY,Iy,ag,F0,F1,rA,rB,nA,nB,ITpDg,IyD,Ie,dz,false);
  end
end

%% disp and plot
if opts.dodisp
  fprintf(' fgeF residual details\n')
  fprintf('%10s,%10s,%10s,%10s,%10s\n','resy','resC','resp','rese','resFx')
  fprintf('%10.5e,%10.5e,%10.5e,%10.5e,%10.5e\n',resy,resC,resp,rese,resFx)
end

if opts.doplot
  fgedebugplot(L,LY,LYp,res,x)
end
end

function [LX,LYp] = fgeF_u2LXLY(u,xdot,L,LX,LYp,xHasIy)
% Update LX and LYp based on u and xdot input for fgeF
%
% This is a helper function for computing the derivatives of fgeF
% with respect to u = [Va/Co/IniD] or xdot using finite differences

if L.isEvolutive
  % u = [Va;Co;Ini]
  Va   = u(L.ind.iua);
  Co   = u(L.ind.iuC);
  Ini  = u(L.ind.iuni);
  % x = [GS;ag;Ia;Iu]
  dt = LX.t - LYp.t;
  if any(xdot) && (dt == 0), error('fgeF with xdot as argument requires dt>0'); end
  dxSI = xdot.*(L.xscal*dt);
  if xHasIy
    dIy = dxSI(L.ind.ixGS);
  else
    dFx = dxSI(L.ind.ixGS);
  end
  dag = dxSI(L.ind.ixg);
  dIa = dxSI(L.ind.ixa);
  dIu = dxSI(L.ind.ixu);
else
  % u = [Ia;Iu;Co]
  Ia    = u(L.ind.iua);
  Iu    = u(L.ind.iuu);
  Co    = u(L.ind.iuC);
end

% Update LX
LX = Co2LX(L,LX,Co);

if L.isEvolutive
  % Update LX
  LX.Va = Va;
  LX.IniD = Ini;
  LX.Ini = sum(LX.IniD);
  % Update LYp to compute dFdxdot = -dt*dFdx0 (x0 is the previous state)
  if xHasIy
    LYp.Iy(:) = LYp.Iy(:) - dIy;
  else
    LYp.Fx(:) = LYp.Fx(:) - dFx;
  end
  LYp.ag = LYp.ag - dag;
  LYp.Ia = LYp.Ia - dIa;
  LYp.Iu = LYp.Iu - dIu;
  % Update Iy for circuit equations
  if ~xHasIy
    LYp.Iy = -reshape(L.dlst*LYp.Fx(:)./L.rhsf + L.Tye*[LYp.Ia;LYp.Iu],L.nzy,L.nry);
  end
  % Update fields necessary for meqcde time derivatives
  if L.np
    if xHasIy
      LYp.Fx = meqFx(L,LYp.Iy,[LYp.Ia;LYp.Iu]);
    end
    for iD=1:L.nD
      LYp.IpD(iD) = sum(LYp.Iy(:) .* (LYp.Opy(:) == iD));
    end
    [meqpdom_out{1:20}] = meqpdom(LYp.Fx,LYp.Ip,L.P.isaddl,L);
    LYp.FA = meqpdom_out{ 3};
    LYp.FB = meqpdom_out{15};
    LYp.F0 = meqpdom_out{19};
    LYp.F1 = meqpdom_out{20};
  end
else
  % Update LX
  LX.Ia = Ia;
  LX.Iu = Iu;
end
end

function  LY = fgepost(L,LX,LYp,ag,...
  Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lB,lX, ...
  rX,zX,FX,dr2FX,dz2FX,drzFX, ...
  rBt,Ia,Iu,Iy,Opy,F0,F1,resy,resC,resp,rese,resFx)

LY = struct('shot',LX.shot,'t',LX.t,'aq',LYp.aq,'aW',LYp.aW);
LY = meqpost(L,LY,ag,...
  Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lB,lX, ...
  rX,zX,FX,dr2FX,dz2FX,drzFX, ...
  rBt,Ia,Iu,Iy,Opy,F0,F1);

% Evolutive equation inputs
if L.isEvolutive
  Va = LX.Va;
  IniD = LX.IniD;
else
  Va = zeros(L.G.na,1);
  IniD = zeros(L.nD,1);
end

LY = meqlarg(LY,resy,resC,resp,rese,resFx,Va,IniD);

% Additional output arguments
for k = 1:numel(L.argoutc)
  if strcmp(L.argoutc{k}.file,mfilename)
    LY.(L.argoutc{k}.fld) = evalin('caller',L.argoutc{k}.exp); % Evaluate in fgeF
  end
end

end

function fgedebugplot(L,LY,LYp,res,x)
clf;

% plot flux
ax=subplot(131);
if LY.lB
  FFB = repmat(LY.FB,1,max(3-numel(LY.FB),1)); % fix me
  contourf(ax,L.rx,L.zx,LY.Fx-LY.FB(1),21); hold on;
  contour(ax,L.rx,L.zx,LY.Fx,FFB,'linewidth',2,'linecolor','k');
else
  contourf(ax,L.rx,L.zx,LY.Fx,21); hold on;
end
plot(ax,LY.rA,LY.zA,'oc',LY.rX,LY.zX,'xc');
axis(ax,'equal','tight')
title(ax,sprintf('Ip=%2.3f',LY.Ip));

% plot Iy
ax=subplot(132);
Iy = LY.Iy; Iy(LY.Opy==0) = -(max(Iy(:)));
imagesc(ax,L.ry,L.zy,Iy); hold(ax,'on');
contour(ax,L.ry,L.zy,LY.Iy,11,'w');
axis(ax,'xy','equal','tight');
title(ax,sprintf('Ip=%8.6fkA',LY.Ip/1e3));

ax=subplot(5,3,3);
bar(ax,categorical({'y','g','p','e'}),[LY.resy;LY.resC;LY.resp;LY.rese]);
title('res');

ax=subplot(5,3,6);
plot(ax,res,'.');
title(ax,sprintf('max(|res|)=%4.2e',max(abs(res))));

ax=subplot(5,3,9);
plot(ax,x);
title(ax,'xnl');

ax=subplot(5,3,12);
plot(ax,1:L.G.na,LYp.Ia,'ob',1:L.G.na,LY.Ia,'xr');
set(ax,'XTick',1:L.G.na);
set(ax,'XTickLabel',L.G.dima)
title(ax,'Ia');

ax=subplot(5,3,15);
plot(ax,1:L.G.nu,[LYp.Iu,LY.Iu]);
title(ax,'Iu');

drawnow;
end
