function LX = fbtxinterp(LX,t,method,cpts)
%FBTXINTERP Interpolate FBT input data structure in time
% LX = fbtxinterp(LX0,t,method,cpts)
%   LX0    : FBT input structure
%   t      : new time base on which to interpolate
%   method : Interpolation method can be 'nan', 'nearest','previous' 'next', or 'linear' (default)
%   cpts   : Setting this to 'true' attempts to interpolate control points. 
%            Otherwise they are set to NaN. Default: false
%
% Linear interpolation of control points is only allowed when the slices between which
% interpolation is done have compatible weights and control points.
%
% See also MEQINTERP, FBT, FBTX
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% Checks
if nargin<3 || isempty(method), method='linear'; end % default
if nargin<4 || isempty(cpts),   cpts  = false;   end
assert(~strcmp(method,'none'),'fbtxinterp:NotAllowed','method ''none'' so interpolation not allowed')

assert(isfield(LX,'gpr'),'fbtxinterp:NotFBTLX','Field LX.gpr not found, is this an FBT LX structure?');
assert(issorted(LX.t),'fbtxinterp:LXTimeNotSorted','t is not sorted, can not interpolate')
assert(issorted(t),'fbtxinterp:TimeNotSorted','t is not sorted, can not interpolate')

% Interpolate
switch method
  case 'linear'
    LX = fbtxinterp_linear(LX,t,method,cpts);
  case {'previous','next','nearest'}
    LX = meqinterp(LX,t,method);  % Cases handled within meqinterp by just slicing
  otherwise
    error('fbtxinterp:UnknownMethod','Unknown interpolation method %s',method)
end
end

function LXi = fbtxinterp_linear(LX,t,method,cpts)
% Linear interpolation

% For linear interpolation perform some checks
zi1 = -1; % init
for it=1:numel(t)
  if any( abs(LX.t-t(it)) < 100*eps(LX.t(end)-LX.t(1))) % time slice contained in LX, skip check
    continue
  end
  % Find times that will be interpolated
  assert(t(it)>=LX.t(1)  ,'fbtxinterp:OutOfTimeRange','requested time %f before first time %f',t(it),LX.t(1)  );
  assert(t(it)<=LX.t(end),'fbtxinterp:OutOfTimeRange','requested time %f after last time %f'  ,t(it),LX.t(end));
  i1 = ifloor(LX.t,t(it));

  if it>1 && (zi1 == i1), continue; end % already checked this interval, continue

  if cpts
    % Require gpr,gpz structure to match
    fbtxgpinterpcheck(LX,'gpr',i1,i1+1,t(it))
    fbtxgpinterpcheck(LX,'gpz',i1,i1+1,t(it))

    % Check that these have the same weights
    for ider = [0,1,2] % for all derivative weights
      switch ider
        case 0, gp = 'gp';
        case 1, gp = 'g1';
        case 2, gp = 'g2';
      end
      for eqgrp = {'f','i','b','c','v','u','d','a'}
        for suffix = {'e','d'}
          fld = [gp,eqgrp{:},suffix{:}]; % construct field name
          if ~isfield(LX,fld), continue; end
          fbtxgpinterpcheck(LX,fld,i1,i1+1,t(it))
        end
      end
    end
  end
  zi1 = i1; % store previous
end

allowInf = true;
LXi = meqinterp(LX,t,method,allowInf);

if ~cpts  % disable control points
  LXi = disable_control_points(LXi,LX.t);
end
end


function fbtxgpinterpcheck(LX,fld,i1,i2,tt)
% function to check field compatibility

errmsg1 = @(errtype) sprintf('LX time slices %d and %d (t=%4.3f,%4.3f) have different %s pattern in %s, can not interpolate to t=%4.3f',...
    i1,i2,LX.t(i1),LX.t(i2),errtype,fld,tt); % anonymous function for error message

errmsg2 = @(type1,type2) sprintf('LX time slices %d and %d (t=%4.3f,%4.3f) have an %s entry followed by %s entry in %s, can not interpolate to t=%4.3f',...
    i1,i2,LX.t(i1),LX.t(i2),type1,type2,fld,tt); % anonymous function for error message
  
inan1 = isnan(LX.(fld)(:,i1)); % nan entries in first slice.
inan2 = isnan(LX.(fld)(:,i2)); % nan entries in second slice

iinf1 = isinf(LX.(fld)(:,i1)); % Inf entries in first slice.
iinf2 = isinf(LX.(fld)(:,i2)); % Inf entries in second slice.

izero1 = (LX.(fld)(:,i1) == 0);
izero2 = (LX.(fld)(:,i2) == 0);

if ~isequal(inan1,inan2) % need NaN pattern to match
  errid = 'fbtxinterp:NaNMismatch';
  error(errid,errmsg1('NaN'));
end
end

function LX = disable_control_points(LX,t_orig)
% Disable control points on interpolated slices.
% For entries LX.t which are not contained in t_orig, 
% * set gpr, gpz to NaN 
% * set gpb to false
% * set all the fa,fb,br,bz,ba,cr,cz,ca,vrr,vrz,vzz to NaN
% * set all the fe,be,ce,ve to Inf
% NB: the weights on currents, dipoles, etc are not touched

t = LX.t;
for ider = [0,1,2] % for all derivative weights
  switch ider
    case 0, gp = 'gp';
    case 1, gp = 'g1';
    case 2, gp = 'g2';
  end

  for it = 1:numel(t)
    if ~any( abs(t(it) - t_orig ) < 10*eps ) % an interpolated time slice
      if isfield(LX,[gp,'r'])
        LX.([gp,'r'])(:,it) = NaN; % gpr to NaN
        LX.([gp,'z'])(:,it) = NaN; % gpz to NaN
        LX.([gp,'b'])(:,it) = false; % gpb to false
      end

      for fld = {'fa','br','bz','ba','cr','cz','vrr','vrz','vzz'}
        field = [gp,fld];
        if ~isfield(LX,field), continue; end
        LX.(field)(:,it) = NaN; % value fields to 0
      end
      for fld = {'fe','fd','be','bd','ce','cd','ve','vd'}
        field = [gp,fld];
        if ~isfield(LX,field), continue; end
        LX.(field)(:,it) = Inf; % weight fields to inf
      end
      % g?fb special
      field = [gp,'fb'];
      if ~isfield(LX,field),continue; end
      LX.(field)(:,it) = false;
    end
  end
end
end