function varargout = fbtxdisp(L,LX,selection)
% function s = fbtxdisp(L,LX,selection)
%
% Textual display of FBT constraints.
% If called without output argument, displays in the command window.
% If passing output argument s, returns cell array with strings
%
% selection: optional argument to choose what to display among the following categories:
%            'currents','dipole','limits','geometry'. By default all are selected.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

assert(nargin>=2,'must provide at least two input arguments L,LX')

assert(numel(LX.t)==1,'supported for 1 timeslice only, use meqxk() to slice')
if nargout>0, s={}; else, s=[]; end % init cell array for storing messages

selection_all = {'currents','dipoles','limits','geometry','voltages','profiles'};
if nargin<3, selection = selection_all; % default
else
  if ~iscell(selection), selection={selection}; end
  assert(all(cellfun(@(x) ismember(x,selection_all),selection)),'invalid selection, allowed: %s',selection_all)
end
if isempty(selection); return; end

%% Header
s=vprintf(s,'\nFBTX cost function / constraints display\n');
s=vprintf(s,'  Tokamak: %s, shot#%d, t=%3.3f\n',L.P.tokamak,LX.shot(1),LX.t(1));

% display equations associated with LX.gp*/g1*/g2* variables
s = gp_disp(L,LX,selection,s);

% display profile/ag constraints
if ismember('profiles',selection)
  s = disp_profile_constraints(L,LX,s);
end

if nargout>0
  varargout{1} = s;
end

end

function s = gp_disp(L,LX,selection,s)

% extract LX into GP for each time derivative
GP = cell(3,1);
for derorder = 0:2
  GP{derorder+1} = fbtLX2GP(LX,derorder);
end

if ismember('currents',selection)
  s = forall_GP(@disp_currents,s,GP,L);
end

if ismember('dipoles',selection)
  s = forall_GP(@disp_dipoles,s,GP,L);
end

if ismember('limits',selection)
  s = disp_limits(s,L);
end

% Passive currents
if ismember('currents',selection)
  s = forall_GP(@disp_passives,s,GP,L);
end

% Voltages 
if ismember('voltages',selection)
  s = forall_GP(@disp_voltages,s,GP,L);
end

% display per shape point
if ismember('geometry',selection)
  s = forall_GP(@disp_shape_points,s,GP);
end
end

function s = disp_currents(GP,dsuf,dtext,dunit,s,L)
%% current constraints
iWa = GP.gpid.*GP.gpie;
iWo = GP.gpod.*GP.gpoe;

% Coil currents
s=vprintf(s,'\nCoil current %s cost / equality constraints:\n',dtext);
isela = find(isfinite(iWa));
% cost function or equality constraint per coil current
for ia = isela'
  s = vprintf(s,'%-18s Current:  Ia%s = %+8.2e[A]                               err=%8.2e[A%s]\n',...
    L.G.dima{ia},dsuf,GP.gpia(ia),iWa(ia),dunit);
end
if isempty(ia); s=vprintf(s,'   none\n'); end

% Coil current combinations
s=vprintf(s,'\nCoil current %s combination cost / equality constraints:\n',dtext);
iselo = find(isfinite(iWo));
% cost function or equality constraint per coil current
for io = iselo'
  expr = constraints_expr(L.G.dima,dsuf,L.G.Coa(io,:),[],[],GP.gpoa(io),GP.gpod*GP.gpoe(io),sprintf('[A]',dunit));
  s = vprintf(s,'%s\n',expr); % print
end
if isempty(io); s=vprintf(s,'   none\n'); end
end

function s = disp_dipoles(GP,dsuf,dtext,dunit,s,L)
iWd = GP.gpdd.*GP.gpde;
s=vprintf(s,'\nDipole %s cost / equality constraints:\n',dtext);
% Dipoles
for id = find(isfinite(iWd))'
  expr = constraints_expr(L.G.dima,dsuf,L.G.Dda(id,:),[],[],GP.gpda(id),GP.gpdd*GP.gpde(id),sprintf('[A%s]',dunit));
  s = vprintf(s,'%s\n',expr); % print
end
if isempty(id); s=vprintf(s,'   none\n'); end
end

function s = disp_limits(s,L)
s=vprintf(s,'\nCoil current limits:\n');
if all(isinf([L.P.limu(:);L.P.liml(:)]))
  s=vprintf(s,'   none\n');
else
  % Inequality constraints: liml*limm < limc*Ia < limu*limm
  limu = L.P.limu.*L.P.limm; % apply margin
  liml = L.P.liml.*L.P.limm;
  for ic = 1:size(L.P.limc,1)
    limc = L.P.limc(ic,:);
    if all(isinf([limu(ic),liml(ic)]))||~any(limc), continue; end
    expr = constraints_expr(L.G.dima,'',limc,limu(ic),liml(ic),[],[],'[A]');
    s = vprintf(s,'%s\n',expr); % print
  end
end
end

function s = disp_voltages(GP,dsuf,dtext,dunit,s,L)
%% Voltages and voltage limits
  iWv = GP.gpad.*GP.gpae;
  % Coil voltages
  s=vprintf(s,'\nCoil voltage %s cost / equality constraints:\n',dtext);
  isela = find(isfinite(iWv));
  % cost function or equality constraint per coil current
  for ia = isela'
    s = vprintf(s,'%-18s Voltage:  Va%s = %+8.2e[V%s]                               err=%8.2e[V%s]\n',...
      L.G.dima{ia},dsuf,GP.gpaa(ia),dunit,iWv(ia),dunit);
  end
  if isempty(ia); s=vprintf(s,'   none\n'); end

  s=vprintf(s,'\nCoil voltage limits:\n');
  if ~L.P.voltlim || all(isinf([L.G.Vamax(:);L.G.Vamin(:)]))
    s=vprintf(s,'   none\n');
  else
    limc = eye(L.G.na);
    for ia = 1:L.G.na
      if all(isinf([L.G.Vamin(ia),L.G.Vamax(ia)])), continue; end
      expr = constraints_expr(L.G.dima,dsuf,limc(ia,:),L.G.Vamax(ia),L.G.Vamin(ia),[],[],sprintf('[V%s]',dunit));
      s = vprintf(s,'%s\n',expr); % print
    end
  end
end

function s = disp_passives(GP,dsuf,dtext,dunit,s,L)
%% Passive currents
iWu = GP.gpud.*GP.gpue;
s=vprintf(s,'\nPassive currents cost / equality constraints:\n');

iselu = find(isfinite(iWu));
% cost function or equality constraint per coil current
for iu = iselu'
  s = vprintf(s,'%-10s Passive current:  Iu%s = %+8.2e[A%s]                               err=%8.2e[A%s]\n',...
    L.G.dimu{iu},dsuf,GP.gpua(iu),dunit,iWu(iu),dunit);
end
if isempty(iu); s=vprintf(s,'   none\n'); end
end

function s = disp_shape_points(GP,dsuf,dtext,dunit,s)
if isempty(fieldnames(GP)), return; end
ind0 = 30; % indentation
ind1 = ind0-18; % smaller indentation

%% display per shape point
iWf = GP.gpfd.*GP.gpfe;
rr=NaN; zz=NaN; % init

s=vprintf(s,'\nMagnetic geometry %s cost / equality constraints:\n',dtext);
if isfield(GP,'gpr')
  nS = numel(GP.gpr); %number of points
else
  nS = 0;
end

for iS=1:nS
  if any(isnan([GP.gpr(iS),GP.gpz(iS),GP.gpb(iS)])) % not a valid R,Z point
    continue;
  end

  if ~(GP.gpr(iS)==rr && GP.gpz(iS)==zz) % point not seen yet
    rr = GP.gpr(iS); zz=GP.gpz(iS); % track previously displayed point
    s = vprintf(s,'r=%+4.3f z=%+4.3f ',GP.gpr(iS),GP.gpz(iS)); % display R,Z
    nx = false; % reset
  end

  if ~any(isnan([iWf(iS),GP.gpfa(iS),GP.gpfb(iS)])) % flux constraint
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    if GP.gpb(iS), sB=sprintf(' B ');  else,sB=sprintf('   '); end    % indicate that point is on LCFS
    if isempty(dsuf)
      s=vprintf(s,'%*s:   MSa*Ia +  MSy*Iy = %-+7.3g + %4.2g*Fb                 err=%8.2e[Wb%s]\n',...
        ind,[sB,'Psi'],GP.gpfa(iS),GP.gpfb(iS),iWf(iS),dunit);
    else
      s=vprintf(s,'%*s:   MSa*Ia%s +  MSy*Iy%s = %-+7.3g                           err=%8.2e[Wb%s]\n',...
        ind,[sB,'Psi',dsuf],dsuf,dsuf,GP.gpfa(iS),iWf(iS),dunit);
    end
  end

  % BR,BZ constraints
  iWB = GP.gpbd.*GP.gpbe;
  if ~any(isnan([iWB(iS),GP.gpbr(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s:  BrSa*Ia%s + BrSy*Iy%s = %-+7.3g                           err=%8.2e[T%s]\n',...
      ind,['Br',dsuf],dsuf,dsuf,GP.gpbr(iS),            iWB(iS),dunit);
  end
  if ~any(isnan([iWB(iS),GP.gpbz(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s:  BzSa*Ia%s + BzSy*Iy%s = %-+7.3g                           err=%8.2e[T%s]\n',...
      ind,['Bz',dsuf],dsuf,dsuf,GP.gpbz(iS),            iWB(iS),dunit);
  end

  % strike point angle constraint
  if ~any(isnan([iWB(iS),GP.gpba(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s: %4.1f*(BrSa*Ia%s+BrSy*Iy%s) + %4.1f*(BzSa*Ia%s+BzSy*Iy%s) = 0  err=%8.2e[T/s]\n',...
      ind,['BN',dsuf],sin(GP.gpba(iS)),dsuf,dsuf,cos(GP.gpba(iS)),dsuf,dsuf,iWB(iS),dunit);
  end

  % Second order flux derivative constraint
  iWC = GP.gpcd.*GP.gpce;
  if ~any(isnan([iWC(iS),GP.gpcr(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s: MrrSa*Ia%s +MrrSy*Iy%s = %-+7.3g                           err=%8.2e[Wb/m2%s]\n',...
      ind,['d2r psi',dsuf],dsuf,dsuf,GP.gpcr(iS),iWC(iS),dunit);
  end
  if ~any(isnan([iWC(iS),GP.gpcz(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s: MrzSa*Ia%s +MrzSy*Iy%s = %-+7.3g                           err=%8.2e[Wb/m2%s]\n',...
      ind,['drdz psi',dsuf],dsuf,dsuf,GP.gpcz(iS),iWC(iS),dunit);
  end

  % Hessian angle constraint
  if ~any(isnan([iWC(iS),GP.gpca(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s: %4.1f*(MzzSa*Ia%s+MzzSy*Iy%s) - %4.1f*(MrrSa*Ia%s+MrrSy*Iy%s) = 0  err=%8.2e[Wb/m2%s]\n',...
      ind,'CN',sin(GP.gpca(iS)),dsuf,dsuf,cos(GP.gpca(iS)),dsuf,dsuf,iWC(iS),dunit);
  end

  % Field gradient constraint
  iWV = GP.gpvd.*GP.gpve;
  if ~any(isnan([iWV(iS),GP.gpvrr(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s: BrrSa*Ia%s %*s = %-+7.3g                  err=%8.2e[T/m%s]\n',...
      ind,['dr B0r',dsuf],dsuf,numel(dsuf)+9,'',GP.gpvrr(iS),iWV(iS),dunit);
  end
  if ~any(isnan([iWV(iS),GP.gpvrz(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s: BrzSa*Ia%s %*s = %-+7.3g                  err=%8.2e[T/m%s]\n',...
      ind,['dz B0r',dsuf],dsuf,numel(dsuf)+9,'',GP.gpvrz(iS),iWV(iS),dunit);
  end
  if ~any(isnan([iWV(iS),GP.gpvzz(iS)]))
    if ~nx, ind = ind1; nx=true; else, ind=ind0; end
    s=vprintf(s,'%*s: BzzSa*Ia%s %*s = %-+7.3g                  err=%8.2e[T/m%s]\n',...
      ind,['dz Bz0',dsuf],dsuf,numel(dsuf)+9,'',GP.gpvzz(iS),iWV(iS),dunit);
  end
end
if isempty(iS), s=vprintf(s,'   none\n'); end
end

%% Profile constraints
function s = disp_profile_constraints(L,LX,s)
s=vprintf(s,'\nPlasma profile constraints:\n');
for iC = 1:L.nC
  agconc = L.agconc(iC,:);
  funname = func2str(agconc{1});
  if strcmp(funname,'fbtlegacy')
    s = vprintf(s,'%-10s iD=%1d\n',funname,agconc{2});
  else
    s = vprintf(s,'%-10s iD=%1d LX.(%s)(%02d)=%5.2g\n',funname,agconc{2},agconc{3},agconc{4},LX.(agconc{3})(agconc{4}));
  end
end
end

function expr = constraints_expr(dima,dsuf,c,ub,lb,eql,weql,units)
% function expr = constraints_expr(C,ub,lb,eql,weql,units)
% construct expression string to display equality/inequality consraints
%    lb<c*x<ub                   (upper/lower bound)
% or c*x = eq                    (equality constraint)
% or 1/weql*(c*x) ~= 1/weql*(eq) (weighted LSQ)
% non-existent constraints should have empty ub, lb or eq.
% c is a row vector, lb, ub, eq are scalars

if nargin<7, units = ''; end

assert(isrow(c),'c must be a row vector');
assert(isempty(ub)||isscalar(ub) , 'ub must be scalar or empty')
assert(isempty(lb)||isscalar(lb) , 'lb must be scalar or empty')
assert(isempty(eql)||isscalar(eql),'eql must be scalar or empty')

% for each row, find the coils involved
iinvolved = find(c~=0);
expr = '';
for iinv = iinvolved
  expr = sprintf('%s %+5.2g*%s',expr,c(iinv),dima{iinv});
end
if ~(isempty(lb)||isinf(lb))
  expr = sprintf('%+5.2e%s <= %s',lb,units,expr);    % lower limit
end
if ~(isempty(ub)||isinf(ub))
  expr = sprintf('%-30s <= %+5.2e%s',expr,ub,units); % upper limit
end
if ~(isempty(eql)||isinf(eql))
  if weql ==0
    expr = sprintf('%-30s =  %4.2g%s',expr,eql,units); % equality constraint
  else
    wstr = sprintf('err=%5.2e%s',weql,units);
    expr = sprintf('%-30s =  %4.2g %30s',expr,eql,wstr);
  end
end
end

function s=forall_GP(fun,s,GP,varargin)
% call function fun(s,GP,dsuf,dtext,,dunit,varargin{:})
% for cell arrays of GP, with time derivative suffix string dsuf
for id=1:numel(GP)
  myGP = GP{id};
  if isempty(myGP) || isempty(fieldnames(myGP))
    continue
  else
    [dsuf,dtext,dunit] = derivative_strings(id-1); % derivative suffix
    s = fun(myGP,dsuf,dtext,dunit,s,varargin{:});  % call function
  end
end
end

function [suffix,text,unit] = derivative_strings(derorder)
% build derivative suffix string for time-derivative constraints
switch derorder
  case 0
    suffix = ''; text = ''; unit = '';
  case 1
    suffix = 'dot'; text = 'time derivative'; unit = '/s';
  case 2
    suffix = 'dotdot'; text = 'second time derivative'; unit = '/s^2';
end
end

function s=vprintf(s,varargin)
% auxiliary function for collecting strings and then displaying all
if iscell(s)
  s{end+1} = sprintf(varargin{:});
else
  fprintf(varargin{:})
end
end
