%FBTT  FBT inverse equilibrium solver
% LY = FBTT(L,LX[,'PAR',VAL,...]) calculates equilibrium using the parameters
% in L as obtained by FBTC, optionally replacing or adding parameters with
% specified values. It returns a structure LY with the calculated
% equilibria.
%
% For help on output structure: see meqt.m
% FBT-specific outputs:
% 
% .Sp    Sum of surfaces of all grid points containing plasma [m^2] 
% .Ep    Plasma box elongation    
% .chi   Root-mean-square of cost function residual
% .chie  Root-mean-square of constraint residual 
% .werr  Weighted cost function error
% .uerr  Unweighted cost function error
% .ID    Equation id for each cost function term
% .Fb    Boundary flux (fitting parameter)
%
% The current version extends both the original FBT algorithm [Hofmann CPC
% 1988] as well as the GSPD algorithm [J. Wai et al, arXiv 2023].
% The optimization algorithm is described in MEQOPT.
%
% See also FBTLSQINIT FBTINIT MEQOPT
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function LY = fbtt(L,LX,varargin)

%% Arguments
for k = 1:2:numel(varargin)
  switch varargin{k}
    % Additional output arguments
    case 'argout', L.argoutc = meqargoutc(varargin{k+1},L.argoutc);
      % Additional parameters
    otherwise, L.P.(varargin{k}) = varargin{k+1};
  end
end

% Constraint tolerance for post-LSQ checks
ctol = 1e-6;

%% QP solver setup
[qpsolve,qpopts] = meqqpsetup(L.P);

%% Initialization
masku = (1:L.nuN).' < L.G.na+1; % Only Va/Ia are optimized for

% if solving multiple slices at the same time, failures should trigger an error.
% Otherwise, warn and attempt next time slice.
if L.P.circuit, msgtype = 'e'; else, msgtype = 'w'; end

%% Interval loop
if ~L.P.circuit
  % time-independent 'FBT' algorithm - solve each time slice separately
  n_intervals = numel(LX.t);
  nts = 1;
else
  % time-dependent 'GSPD' algorithm - solve time-dependent problem
  % one interval for whole pulse (for now).
  n_intervals = 1;
  nts = numel(LX.t);
end

kts = 0; % init
for kint = 1:n_intervals % loop over number of intervals
  kts = kts(end) + (1:nts); % time indices in this interval

  LXs = meqxk(LX,kts);

  %% Prepare Least-Squares problem terms
  [Raxc,Rauc,Rabc,Ra0c,Rexc,Reuc,Rebc,Re0c,Rixc,Riuc,Ribc,Ri0c,...
    IDs,ws,bbs,TusU,TFbsU,d1dt,d2dt] = fbtlsqinit(L,LXs);

  % Scale
  if L.P.circuit
    Uscale = [repmat(L.Va0,nts,1);repmat(L.Fx0,nts,1)];
  else
    Uscale = [repmat(L.Ia0,nts,1);repmat(L.Fx0,nts,1)];
  end
  
  %% Assign vessel currents for static case
  LXs.Iu = LXs.gpua;

  %% Initialize Ia and Va in LX structure
  LXs.Ia = LXs.gpia;
  LXs.Va = LXs.gpaa;

  %% Prepare for Picard iterations
  % dxdu = -Jx\Ju, dxdxp = (Jx\Jxdot)*1/dt
  dts  = diff(LXs.t([1,1:end]));
  idts = 1./dts; idts(dts==0) = 0;
  [picard_Precs,dxdus,dxdxps] = fgeFJac_inverse_picard_approx(L,idts);
  if ~L.P.circuit
    % Static case, dxdu is dxdIe and we need only dxdIa
    dxdus{1} = dxdus{1}(:,1:L.G.na);
  end

  %% Initial guess
  if L.P.itert && ~L.P.circuit && kts>1
    % Use state (xs) inputs (us) and fluxes (Fbs) from previous time step
    s = []; % init status
    init_response_matrix = false;
  else
    % Initialize
    resFs = cell(nts,1);
    dxs_  = resFs; % Initial guess for dx0/dxdu
    
    [xs,us,Fbs,s,RaUt,ReUt,RiUt] = fbtinit(L,LXs,...
      Raxc,Rauc,Rabc,Ra0c,Rexc,Reuc,Rebc,Re0c,Rixc,Riuc,Ribc,Ri0c,...
      TusU,TFbsU,d1dt,d2dt,dxdus,dxdxps,bbs,Uscale,qpsolve,qpopts,ctol);
    init_response_matrix = true;
  end

  %% NL iterations
  [xs,us,Fbs,LYs,dxs_,conv] = meqopt(L,LXs,...
    xs,us,Fbs,s,init_response_matrix,RaUt,ReUt,RiUt,...
    Raxc,Rauc,Rabc,Ra0c,Rexc,Reuc,Rebc,Re0c,Rixc,Riuc,Ribc,Ri0c,...
    IDs,ws,TusU,TFbsU,d1dt,d2dt,...
    idts,picard_Precs,dxdus,dxdxps,dxs_,Uscale,masku,...
    msgtype,qpsolve,qpopts,ctol);

  %% Post-processing
  if conv || L.P.LYall || L.P.itert
    for its = 1:nts
      LYt = LYs{its};
      % FBT specific outputs
      bbox = bboxmex(logical(LYt.Opy),L.zy,L.ry);
      LYt.Sp = sum(LYt.Opy(:)) * L.dsx;
      LYt.Ep = (bbox(3) - bbox(1)) / (bbox(4) - bbox(2));
      LYt.Fb = Fbs{its};
      
      % time derivatives
      if kts(its)==1
        LYt = meqLYdot(LYt,LYt);
      else
        LYt = meqLYdot(LYt,LY(kts(its)-1));
      end
      LY(kts(its)) = LYt; %#ok<AGROW>
    end
  end

end % kt

if exist('LY','var') % There is at least one converged time slice
  LY = meqlpack(LY);
elseif nargout > 0
  LY = struct([]);
end
 
end

