%FBTPTCV  TCV FBT configuration parameters
% P = FBTPTCV(SHOT,'PAR',VAL,...) returns a structure P with configuration
% parameters from the PCS tree for SHOT, optionally replacing or adding
% parameters with specified values. See also FBTP.
% Most parameters are inherited from MGAMS. For some documentation see:
%  https://spcwiki.epfl.ch/wiki/Mgams_help
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function P = fbtptcv(shot,varargin)
 P = meqptcv(shot);

 % Defaults
 P.bfct    = @bffbt;    % FBT basis function set
 P.sela    = {'E','F'}; % No OH or G coils in FBTE
 P.selu    = 'n';       % no vessel in FBTE
 P.selx    = 'XF';      % Historical FBTE grid
 P.circuit = false;     % No passive structures by default
 P.dohzero = true;      % Constrain DOH (=OH1-OH2) =0 (if sela includes 'OH')
 P.fbtagcon = {'Ip','qA','fbtlegacy'}; % Legacy scheme for ag constraints
 P.useSQP  = false;     % Use Picard iterations as default method over Newton
 
 assert(~isempty(which('mgp')),'mgp.m not found - can''t load shot MGAMS programmed data for FBT-TCV');
 % Load from PCS tree
 mdsconnect(P.mdsserver); % make sure to connect to tcvdata because mgp does not necessarily do it
 Pmgp = mgp(shot);

 % Shots prior to 56469 did not have gp* variables and these will not have the correct number of columns
 if (shot > 0 && shot < 56469)
   Pmgp.fix;
 end

 % process fields from mgp object
 for ifield=(Pmgp.NAME).'
   if startsWith('debug_p',ifield{1}), continue; end
   P.(ifield{1}) = Pmgp.(ifield{1});
 end
 
 % Process additional arguments from debug_p1
 LXagconfields = {'Ip','IpD','qA','li','liD','bp','bpD','bt','btD','WkD','Wk'};
 if ~isprop(Pmgp,'debug_p1')
  warning('FBTE:fbtptcv:debug_p1_missing','debug_p1 is missing from mgp object, check the version of shotdesign used');
 else
  P1 = Pmgp.debug_struct(1);
  for ifield = fieldnames(P1).'
   % Skip LX agcon fields as they should not be passed via parameters
   if ismember(ifield{1},LXagconfields)
    continue;
   end
   P.(ifield{1}) = P1.(ifield{1});
  end
 end
 
 % Define shot,t convert kA to A, Wb/rad to Wb
 P.shot    = mdsdata('$SHOT');
 P.placu1  = 1000*P.placu1;
 P.dissi   = 1e-6*P.dissi ;
 P.strki   = 1e-6*P.strki ;
 P.t       = 0.01+P.timefac*(P.timeeq-P.timeeq(1));

 % New parameter for asymmetric grids
 if ~isfield(P,'zl'), P.zl = -P.zu; end

 % adds/overwrites parameters
 for k = 1:2:nargin-1
  P.(varargin{k}) = varargin{k+1};
 end
 % Check illegal passed parameters - to be unified in fbtp.m when parts of
 % this are moved to fbtxtcv
 legacy_fields = LXagconfields;
 ioffending = isfield(P,legacy_fields);
 assert(~any(ioffending),...
   'FBT:legacyP','Passing fields P.(%s) is deprecated, must pass directly through LX',strjoin(legacy_fields(ioffending),','))
 
 % Cannot use a different static tree when preparing the MODEL (-1) shot
 assert(shot ~= -1 || P.static == -1,'Cannot specify a custom static tree when preparing the MODEL (-1) shot');
 
 if ~all(isfield(P,{'gpvrr','gpvrz','gpvzz','gpve','gpvd'}))
  if any(isfield(P,{'gpvrr','gpvrz','gpvzz','gpve','gpvd'}))
   msg = {'Some of the gpv* variables are missing, replacing all with NaNs.',...
          'Please provide ''gpvrr'', ''gpvrz'', ''gpvzz'', ''gpve'' and ''gpvd''.'};
   warning('fbtptcv:gpv_incomplete',strjoin(msg,'\n'));
  end
  P.gpvrr = NaN(size(P.gpr));
  P.gpvrz = NaN(size(P.gpr));
  P.gpvzz = NaN(size(P.gpr));
  P.gpve  = Inf(size(P.gpr));
  P.gpvd  = Inf(size(P.gpbd));
 end
 
 % Treat Ip and B0 signs
 % NOTE: We assume here that in MGAMS every GP variable is set for Ip,B0>0
 assert(all(P.placu1>0) && all(P.bzero>0),'placu1,bzero should be >0 from MGAMS')
 P.placu1  = P.iohfb*P.placu1;
 P.gpia    = P.iohfb*P.gpia;
 P.gpfa    = P.iohfb*P.gpfa;
 P.gpbr    = P.iohfb*P.gpbr;
 P.gpbz    = P.iohfb*P.gpbz;
 P.gpba    = (1-sign(P.iohfb))*pi/2+P.gpba; % 0 for Ip>0, pi for Ip<0
 P.gpcr    = P.iohfb*P.gpcr;
 P.gpcz    = P.iohfb*P.gpcz;
 P.gpvrr   = P.iohfb*P.gpvrr;
 P.gpvrz   = P.iohfb*P.gpvrz;
 P.gpvzz   = P.iohfb*P.gpvzz;
 P.bzero   = P.if36fb*P.bzero;
 P.qzero   = P.iohfb*P.if36fb*P.qzero;
 
 % Use unified parameters
 P = fbtptcv_update(P);
 
 dima = {};
 for k = 1:numel(P.sela)
  switch P.sela{k}
   case 'E',   l = 1:8;
   case 'F',   l = 1:8;
   case 'G',   l = 1;
   case 'OH',  l = 1:2;
   case 'TOR', l = 1;
   otherwise, error('unknown coil %s',P.sela{k});
  end
  dima = [dima; cellstr(num2str(l(:),[P.sela{k} '_%03d']))]; %#ok<AGROW>
 end
 na = numel(dima);

 nt = P.nruns;

 % translate mvloop=8 to specific doublet parameters
 if ~isfield(P,'idoublet')
   P.idoublet = (P.mvloop == 8);
 end
 % Use multiple domains for TCV doublets as default
 if P.idoublet && ~any(strcmp(varargin(1:2:end),'isaddl'))
   P.isaddl = 1;
 end
 
 % Remember which coil current constraints were not specified from input parameters
 mask = ~isnan(P.gpie.*P.gpid);
 
 % Expected residual errors
 if ~isfield(P,'gpfd') || any(isnan(P.gpfd)), P.gpfd = repmat(5e-3,1,nt); end
 if ~isfield(P,'gpbd') || any(isnan(P.gpbd)), P.gpbd = repmat(5e-3,1,nt); end
 if ~isfield(P,'gpcd') || any(isnan(P.gpcd)), P.gpcd = repmat(5e-3,1,nt); end
 if ~isfield(P,'gpid') || any(isnan(P.gpid)), P.gpid = 5e-3./(18*2*pi*sqrt(2*P.dissi)); end
 if ~isfield(P,'gpdd') || any(isnan(P.gpdd)), P.gpdd = 5e-3./(   2*pi*sqrt(  P.dipol)); end

 % Fixes to gp variables from MGAMS
 P = gpfix(P); % this turns NaN values in gp*e/gp*d into Inf

 % Check consistency of sizes between pre-existing GPs
 P = gpcheck(P);
 
 % Convert old shape parameters
 n = max(P.ilia(logical(P.iansha)));
 P.rlia1(end+1:n,:) = NaN;
 P.zlia1(end+1:n,:) = NaN;
 for k = find(P.iansha)
  n = P.ilia(k);
  x = 2*pi/n*(0:n-1)';
  P.rlia1(:,k) = NaN;
  P.zlia1(:,k) = NaN;
  P.rlia1(1:n,k) = (P.rmajo1(k)+P.rmino1(k)*cos(x+P.delta1(k)*sin(x)-P.hlamd1(k)*sin(2*x)));
  P.zlia1(1:n,k) = (P.zmajo1(k)+P.rmino1(k)*P.cappa1(k)*sin(x));
 end
 %   fbtgp(P,r      ,z      ,b,fa,fb,fe,br,bz,ba,be,cr,cz,ca,ce,vrr,vrz,vzz,ve)
 P = fbtgp(P,P.rlim1,P.zlim1,1, 0, 1, 0,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
 P = fbtgp(P,P.rlia1,P.zlia1,1, 0, 1, 1,[],[],[],[],[],[],[],[],[] ,[] ,[] ,[]);
 P = fbtgp(P,P.rbro ,P.zbro ,0,[],[],[], 0,[],[], 0,[],[],[],[],[] ,[] ,[] ,[]);
 P = fbtgp(P,P.rbzo ,P.zbzo ,0,[],[],[],[], 0,[], 0,[],[],[],[],[] ,[] ,[] ,[]);
 % Adjust selection for plasma initial guess
 P.gpb = gpb(P);
 
 %% Coil current errors
 % also handle conversion of node values with only E and F coils into general dima specification
 iE = find(contains(dima,'E'));
 iF = find(contains(dima,'F'));
 iTOR = find(contains(dima,'TOR'));
 
 dimEF = dima([iE;iF]); % dim of errors in MGAMS nodes / mgp values
 % indices into dimEF
 [iEc,iFc] = n2k(numel(iE),numel(iF));
 iEc = iEc(:); iFc = iFc(:); % Expecting column vectors

 % If all errors are not NaN, use values passed in input
 %   otherwise apply standard TCV scheme for E and F coils
 if ~all(mask(:))

   % Avoid conflicts between icoilon and gpia
   conflicts = mask & ~P.icoilon;
   if any(conflicts(:))
     % Prepare error message with lists of times (runs) and coils with conflicts
     runs_c = find(any(conflicts,1));
     msg = cell(1,numel(runs_c)+1);
     msg{1} = 'Conflicts between icoilon and gpia settings.';
     for irun = 1:numel(runs_c)
       msg{1+irun} = sprintf('  %d: %s',irun,strjoin(dima(conflicts(:,runs_c(irun)))));
     end
     error('fbtptcv:IaConflict',strjoin(msg,'\n'));
   end

   % Convert dissi+icoilon into gpia+gpie+gpid
   gpia_ = P.gpia; gpie_ = P.gpie;
   gpia_(~mask) = 0;                                           % if no active constraint, set value to 0
   gpie_(~mask) = 1;                                           % Default error
   gpie_(iEc,:) = gpie_(iEc,:)*sqrt(2)*18/17;                  % Increase default error for E coils
   gpie_(~mask & ~P.icoilon) = gpie_(~mask & ~P.icoilon)/1e3;  % Coils "deselected" have small errors for their current.

   P.gpia = zeros(na,nt); P.gpia([iE;iF],:) = gpia_([iEc;iFc],:);
   P.gpie =  ones(na,nt); P.gpie([iE;iF],:) = gpie_([iEc;iFc],:);

   if ~isempty(iTOR)
     P.gpia(iTOR,:) = 2*pi*P.rBt/(4e-7*pi)/(6*16); % mu0=4e-7*pi; 16 coils and 6 turns per coils
     P.gpie(iTOR,:) = 0; % equality constraint
   end
 end
 
 %% Dipole errors
 check = isfield(P,{'gpdw','gpde'});
 if ~any(check)
   % Convert old errors for PFC currents dipole
   % Wij = Wpsi*strki/dij^2*(w(i)*Ia(i) - w(j)*Ia(j))^2
   % Variables: dij -> dc, w(i) -> gpdw
   meqmdsopen(P.static,'STATIC',[],P.mdsserver);
   % Dipole term only on E,F coils, depending on their distance
   rEF = mdsdata('STATIC("R_C")[$1]',dimEF);
   zEF = mdsdata('STATIC("Z_C")[$1]',dimEF);
   wEF = mdsdata('STATIC("W_C")[$1]',dimEF);
   hEF = mdsdata('STATIC("H_C")[$1]',dimEF);
   % E coils have 4 vert. stacked filaments
   % F coils have 2 rad.  stacked filaments
   rff = [rEF(iEc);rEF(iFc)-wEF(iFc)/4];   % R of first coil filament
   rfl = [rEF(iEc);rEF(iFc)+wEF(iFc)/4];   % R of last  coil filament
   zff = [zEF(iEc)-hEF(iEc)*3/8;zEF(iFc)]; % Z of first coil filament
   zfl = [zEF(iEc)+hEF(iEc)*3/8;zEF(iFc)]; % Z of last  coil filament
   dc = sqrt((rff(2:end)-rfl(1:end-1)).^2 + (zff(2:end)-zfl(1:end-1)).^2); % Distance from the last filament to the first of the next coil
   dw([iEc,iFc],:) = [17/2*ones(numel(iEc),1);18*ones(numel(iFc),1)]; % Weights for each coil current (from current in each turn to current in each filament)
   gpdw = zeros(na,nt); gpdw([iE;iF],:) = repmat(dw,1,nt);
   P.gpdw = gpdw;
   P.gpde = repmat(dc,1,nt); % Weight is inversely proportional to distance.
 elseif ~all(check)
   error('fbtptcv:wdipole','Must specify all or none of gpdw,gpde in input')
 end
 
 % Remove deprecated parameters
 P = rmfield(P,{'dissi','dipol'});

 %% Coil limits
 if shot>0 && shot<1e5, base_shot = shot;
 elseif shot>1e6,       base_shot = floor(shot/1e3);
 else,                  base_shot = -1;
 end

 % -INOM_A  < Ia     < INOM_A  % coil current limits
 %  ILOW    < Ceq*Ia < IHIG    % Coil protection, first 8 equations only (linear inequality constraints)
 meqmdsopen(base_shot,'BASE',[],P.mdsserver);
 Ceq_ = mdsdata('\COIL_PROT:CEQ[1:8,[$1]]',dima);
 % no CEQ entry for TOR in tree, add it if necessary.
 iTOR = contains(dima,'TOR');
 if any(iTOR), Ceq(:,~iTOR) = Ceq_; Ceq(:,iTOR)=zeros(size(Ceq,1),1);
 else, Ceq = Ceq_; end 
 P.limc = [eye(na);Ceq];
 P.liml = NaN(size(P.limc,1),1); P.limu = P.liml;
 P.liml(na+1:end) = mdsdata('\COIL_PROT:ILOW[1:8]');
 P.limu(na+1:end) = mdsdata('\COIL_PROT:IHIG[1:8]');
 meqmdsopen(P.static,'STATIC',[],P.mdsserver);
 P.limu(1:na) = mdsdata('STATIC("INOM_A")[$1]',dima);
 P.liml(1:na) = -P.limu(1:na);
 meqmdsopen(shot,'PCS',[],P.mdsserver);
 
 %% Grid choice
 validatestring(P.selx,{'','X','XF'},mfilename,'P.selx');
end

function P = fbtptcv_update(P)
 % update TCV parameters into FBT/MEQ ones
 
 % quantities which come from mds tree for legacy reasons but
 % should be loaded in fbtxtcv
 P.Ip  = P.placu1;    
 P.rBt = P.bzero*0.88;
 if ~isfield(P,'qA')
   P.qA  = P.qzero;
 end
 
 % quantities which may be provided via fbt(...) calls
 % For these, if no user-provided values are given, write TCV tree values
 if ~isfield(P,'niter'),  P.niter  = P.itamax; end
 if ~isfield(P,'tol'),    P.tol    = P.testa;  end
 if ~isfield(P,'dipol'),  P.dipol  = P.strki;  end

 % Remove legacy field names for clarity
 P = rmfield(P,{'qzero','bzero','placu1','itamax','testa','strki'});
end

function P = gpfix(P)
% fixes to MGAMS gp* variables to replace NaN entries of gp*e, gp*d by Inf
% Might be fixed on MGAMS side later

eqlist = {'i','f','b','c','v'};
for eq=eqlist
  gpe = ['gp',eq{1},'e']; % gp*e field name
  gpd = ['gp',eq{1},'d']; % gp*d field name
  P.(gpe)(isnan(P.(gpe))) = Inf;
  P.(gpd)(isnan(P.(gpd))) = Inf;
end

end

function P = gpcheck(P)

 nruns = P.nruns;

 % Since the mgp class removes final rows with only NaNs in time-dependent
 % arrays, we need here to restore these so that each GP parameter has the
 % same number of rows.

 list = strcat('gp',{'r','z','b','fa','fb','fe','br','bz','ba','be','cr','cz','ca','ce'});
 nvar = numel(list);
 
 % Get number of rows for each variable and check number of columns matches nruns
 rows = NaN(nvar,1);
 for ii = 1:nvar
  sz = size(P.(list{ii}));
  assert(sz(2) == nruns,'FBTE:Inconsistent number of columns for %s. Expected %d found %d',list{ii},nruns,sz(2));
  rows(ii) = sz(1);
 end
 
 % Now make sure all of them have the same number of rows
 nrows = max(rows);
 for ii = 1:nvar
  myfield = list{ii};
  if any(strcmp(myfield(end),{'d','e'})) % errors
    default = Inf;
  else
    default = NaN;
  end
  P.(list{ii})(rows(ii)+1:nrows,:) = repmat(default,nrows - rows(ii),nruns);
 end
 
 % Check that we've made things right
 rows0 = size(P.(list{1}),1);
 for ii = 2:nvar
  assert(size(P.(list{ii}),1) == rows0, 'FBTE:Inconsistent number of rows between gp%s and gp%s',list{1},list{ii});
 end
 
 % Check dimensions
 assert(size(P.gpia,1)==16,'FBTE:gpia must have 16 rows')
end

function gpb = gpb(P)
 % Attempt to eliminate points defining strike points characteristics
 % from initial plasma guess (typically for old TCV shots).
 gpb = P.gpb;
 for k = 1:P.nruns
  % X-points
  ibro = P.gpbr(:,k)==0;
  ibzo = P.gpbz(:,k)==0;
  rzbro = [P.gpr(ibro,k),P.gpz(ibro,k)];
  rzbzo = [P.gpr(ibzo,k),P.gpz(ibzo,k)];
  iX = ismember(rzbro,rzbzo,'rows');
  if P.idoublet, iX = false & iX; end % Do not consider X-points for doublets
  rX = rzbro(iX,1);
  zX = rzbro(iX,2);
  % Initiate loop
  bo = gpb(:,k) == 1 & ~isnan(P.gpr(:,k)) & ~isnan(P.gpz(:,k));
  kk = 0;
  while kk < 10
   % Axis as center of bounding box of points with gpb=1
   bb = [min(P.gpr(bo,k)) max(P.gpr(bo,k)) min(P.gpz(bo,k)) max(P.gpr(bo,k))]*0.5;
   rA1 = bb(1) + bb(2); zA1 = bb(3) + bb(4);
   % Axis as barycenter of active control points
   no = sum(bo);
   rA2 = sum(P.gpr(bo,k))/no; zA2=sum(P.gpz(bo,k))/no;
   % Weight both answers (1 for bounding box, 3 for barycenter)
   rA = (rA1 + 3*rA2)/4; zA = (zA1 + 3*zA2)/4;
   % Set gpb=0 for points outside of so-defined X-point polygon
   b = bo & bavxmex(rX,zX,rA-rX,zA-zX,0,P.gpr(:,k),P.gpz(:,k));
   kk = kk+1;
   % If gpb did not change, stop
   if isequal(b,bo), break;end
   bo = b;
  end
  gpb(:,k) = b;
 end
end
