function [xs,us,Fbs,s,RaUt,ReUt,RiUt] = fbtinit(L,LXs,...
  Raxc,Rauc,Rabc,Ra0c,Rexc,Reuc,Rebc,Re0c,Rixc,Riuc,Ribc,Ri0c,...
  TusU,TFbsU,d1dt,d2dt,dxdus,dxdxps,bbs,Uscale,qpsolve,qpopts,ctol)
% FBTINIT Provide initial guess for FBT optimization
%
%   [xs,us,Fbs,dx0s,s,RaUt,ReUt,RiUt] = fbtinit(L,LXs,...
%     Raxc,Rauc,Rabc,Ra0c,Rexc,Reuc,Rebc,Re0c,Rixc,Riuc,Ribc,Ri0c,...
%     TusU,TFbsU,d1dt,d2dt,dxdus,dxdxps,bbs,Uscale,qpsolve,qpopts,ctol)
%
% A detailed description of the inputs and outputs is available in MEQOPT.
% Note that for FBT the optimization varaibles zs are the fluxes Fbs.
%
% This function provides an initial guess for the FBT optimisation problem by:
% 1. first fixing the plasma current distribution either from the LX structure or
% using a parabolic profile inside rectangular boxes containing the boundary
% control points (those with LX.gpb=1)
% 2. Performing a single Newton step for the optimization problem at fixed plasma
% profiles (Iy,ag) to get a first estimate of the external currents Ie and fluxes
% Fb.
%
% See also FBTT FBTOPT MEQOPT FBTLSQINIT MEQLSQPREP MEQLSQSOLVE
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

% Initialize
nts = numel(LXs.t);
xs = cell(nts,1);

xHasIy = strcmp(L.P.algoNL,'all-nl');

% Plasma current distribution
if isfield(LXs,'Iy')
  if size(LXs.Iy,3)<nts
    error('To use a custom initial guess LX.Iy should contain 1 time slice if itert>0 or numel(LX.t) otherwise')
  end
  Iy = LXs.Iy; % Iy from LX
else
  if L.nD == 1; IpD = LXs.Ip;
  else,         IpD = LXs.IpD;
  end
  Iy = meqinitIy(L,LXs.t,IpD,bbs,LXs.capaj);
end

% Initial x
ag_ = zeros(L.ng,1);
Ie_ = zeros(L.ne,1);
for its = 1:nts
  % Get GS part of state
  Iy_ = Iy(:,:,its);
  if xHasIy % all-nl
    xGS_ = Iy_;
  else % all-nl-Fx, Newton-GS
    xGS_ = meqFx(L,Iy_,Ie_);
  end
  % Assign full state
  if L.isEvolutive
    xs{its} = [xGS_(:);ag_;Ie_]./L.xscal;
  else
    xs{its} = [xGS_(:);ag_]./L.xscal;
  end
end

% Initial u, Fb
us     = repmat({zeros(L.G.na,1)},nts,1);
Fbs    = repmat({0}              ,nts,1);
dx0s   = repmat({zeros(L.nN,1)}  ,nts,1);

% Auxiliaries
mask = false(L.nN,1); % Mask of rows where dxdu is 0 (static problem only)
skip_response_matrix = false; % Always compute RaUt/ReUt/RiUt for the initialization
no_ineq = ~L.P.initineq; % If true, inequalities are removed from the optimization

% Perform single Newton iteration at fixed Iy,ag=0
% Evaluate residuals
[Ra,Re,Ri] = meqlsqeval(L,...
  Raxc,Rauc,Rabc,Ra0c,Rexc,Reuc,Rebc,Re0c,Rixc,Riuc,Ribc,Ri0c,...
  d1dt,d2dt,xs,us,Fbs);
% Form reduced optimization problem
[Rat,RaUt,Ret,ReUt,Rit,RiUt] = meqlsqprep(L,...
  Raxc,Rauc,Rabc,Ra,Rexc,Reuc,Rebc,Re,Rixc,Riuc,Ribc,Ri,...
  TusU,TFbsU,d1dt,d2dt,xs,us,Fbs,dx0s,dxdus,dxdxps,mask,skip_response_matrix);
% Solve for full u/Fb (since initial guess is 0)
[dxs,dus,dFbs,lambda,~,s] = meqlsqsolve(L,qpsolve,qpopts,...
  Rat,RaUt,Ret,ReUt,Rit,RiUt,...
  dx0s,dxdus,dxdxps,TusU,TFbsU,...
  Uscale,no_ineq,ctol);
% Update Ia/Iu in xs
for its = 1:nts
  xs {its} = xs {its} + dxs {its};
  us {its} = us {its} + dus {its};
  Fbs{its} = Fbs{its} + dFbs{its};
end

end