function MG = fbt2pcs(L,LX)
% FBT2PCS converts L and LX FBT structures to MGAMS parameters
%   MG = fbt2pcs(L,LX)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

assert(isscalar(LX.t),'fbt2pcs only works on a single slice');

MG = mgp();

% Remove legacy MGAMS shape parameters
% Set non-analytic shape
MG.iansha = 0;
% rlia1/zlia1
MG.rlia1 = zeros(0,1);
MG.zlia1 = zeros(0,1);
% rlim1/zlim1
MG.rlim1 = zeros(0,1);
MG.zlim1 = zeros(0,1);
% rbro/zbro
MG.rbro = zeros(0,1);
MG.zbro = zeros(0,1);
% rbzo/zbzo
MG.rbzo = zeros(0,1);
MG.zbzo = zeros(0,1);
% icoilon
MG.icoilon = ones(16,1);

% mvloop
MG.mvloop = int8(LX.idoublet);

% time
MG.timeeq = 0; % let this just be the first equilibrium

% Set sign of Ip, Bt
MG.iohfb = sign(LX.Ip);
MG.if36fb = sign(LX.rBt);
assert(LX.Ip>0 && LX.rBt>0, 'fbt2pcs:IpBtsigns','fbt2pcs only works for Ip>0 and Bt>0 cases');

% Fields automatically filled
ignored = {'tokamak','shot','t','aq','aW','Iy'};

% Add in GPS variables from LX
LXfields = fieldnames(LX);
MGprops = MG.name;
nf = numel(LXfields);
debug_p1 = cell(1,nf); mask = false(1,nf);
for ii = 1:nf
  name = LXfields{ii};
  val = LX.(name);
  switch name
    case ignored
      continue
    case MGprops
      MG.(name) = val;
    % First some renaming
    case 'tol'
      MG.testa = val;
    case 'niter'
      MG.itamax = val;
    case 'rBt'
      MG.bzero = val/0.88;
    case 'Ip'
      MG.placu1 = val/1e3;
    otherwise
      if strcmp(name,'bp') && L.P.idoublet
        continue;
      end
      debug_p1{ii} = sprintf('''%s'',%s',name,mat2str(val));
      mask(ii) = true;
  end  
end

prec = 15;

% bfct
bfct = L.P.bfct;
if isa(L.P.bfct,'function_handle')
  bfct = func2str(L.P.bfct);
end
bfct_str = ['''bfct'',@',bfct];

% bfp
switch bfct
  case 'bfgenD'
    bfp = L.P.bfp;
    % cell2str does not convert function handles
    for ii = 1:size(bfp,1)
      if isa(bfp{ii,1},'function_handle')
        bfp{ii,1} = ['@',func2str(bfp{ii,1})];
      end
    end
    bfp_str = cell2str(bfp,prec);
    % Remove continuation lines and newlines
    bfp_str = strrep(bfp_str,['...',newline],'');
    bfp_str = strrep(bfp_str,newline,'');
    % Restore function handles
    bfp_str = regexprep(bfp_str,'''(@\w+)''','$1');
    bfp_str = ['''bfp'',{',bfp_str,'}'];
  case 'bfabmex'
    bfp = L.P.bfp;
    bfp_str = sprintf('''bfp'',[%s]',num2str(bfp));
  otherwise
    error('bfct=%s is not yet supported in fbt2pcs',bfct);
end

% agcon
fbtagcon_str = ['''fbtagcon'',{',cell2str(L.P.fbtagcon,0),'}'];

% selx
selx_str = ['''selx'',''',L.P.selx,''''];

% Assign debug node string
MG.debug_p1 = strjoin([debug_p1(mask),{bfct_str,bfp_str,fbtagcon_str,selx_str}],',');
% Check string (method will throw error when unable to convert to struct)
MG.debug_struct(1);

end
