function SC = fbt2SC(L,LX,varargin)
% Get and classify shape control points from FBT definition
%
% Inputs:
% L,LX: fbt results from [L,LX] = fbt(...)
% nc: if the number of control points is less than nc, pad them to reach nc
%
% Outputs:
% SC with fields:
%   - t: time array
%   - nc: number of control points
%   - rc,zc: coordinates of control points
%   - dimc: labels for control points
%   - Control point are classified into:
%     + Contour points     (lcC  = 1)
%     + Limiter points     (lcL  = 1)
%     + Strike points      (lcS  = 1)
%     + Primary X points   (lcX1 = 1)
%     + Secondary X points (lcX2 = 1)
%     + Separatrix points  (lcD  = 1)
%     + Ignored points     (lcI  = 1)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Parse inputs
p = inputParser;

% Validation functions
checkL  = @(L)  isstruct(L);
checkLX = @(LX) isstruct(LX);
checknc = @(nc) isnumeric(nc);

% Add inputs
addRequired(p,'L' ,checkL);
addRequired(p,'LX',checkLX);
addOptional(p,'nc',size(LX.gpr,1),checknc);

% Get parser results
parse(p,L,LX,varargin{:});
nc = p.Results.nc;

%% Classify control points
nt = numel(LX.t); % Number of equilibria
SC.t = LX.t; % Store time array
SC.nc = nc; % Number of control points
% Initialize fields
[SC.rc,SC.zc] = deal(zeros(nc,nt));
SC.dimc = cell(nc,nt);
[SC.lcC,SC.lcL,SC.lcS,SC.lcX1,SC.lcX2,SC.lcD,SC.lcI] = deal(false(SC.nc,nt));
% Iterate over time slices
for it = 1:nt
  % Init fields
  [gpr,gpz] = deal(zeros(nc,1));
  [gpb,gpbr,gpbz,gpfe] = deal(NaN(nc,1));
  % Get current time slice of LX
  LXt = meqxk(LX,it);
  % Group control points within distance of 1e-2*max(abs([LXt.gpr;LXt.gpz]))
  [~,ia,ic] = uniquetol([LXt.gpr,LXt.gpz],1e-2,'ByRows',true);
  % Iterate over control points groups
  jj = 0;
  for ii = 1:numel(ia)
    ioccur = find(ic == ii); % Indices of control points within that group
    % Check if it is a valid point
    if any(isnan([LXt.gpr(ioccur(1)),LXt.gpz(ioccur(1))])); continue; end
    % Store point coordinates
    jj = jj+1;
    if jj > nc
      error('Found more countor points than the desired total number');
    end
    gpr(jj) = LXt.gpr(ioccur(1)); gpz(jj) = LXt.gpz(ioccur(1));
    % Get gpb
    if any(LXt.gpb(ioccur)==1);  gpb(jj)  = 1; else; gpb(jj)  = 0;   end
    % Get gpbr
    if any(LXt.gpbr(ioccur)==0); gpbr(jj) = 0; else; gpbr(jj) = NaN; end
    % Get gpbz
    if any(LXt.gpbz(ioccur)==0); gpbz(jj) = 0; else; gpbz(jj) = NaN; end
    % Get gpbz
    if any(LXt.gpfe(ioccur)==0); gpfe(jj) = 0; else; gpfe(jj) = 1;   end
  end
  % Compute the distance between each point and the wall
  tol = 2e-2;
  dist_limiter = Inf(SC.nc,1);
  rl = [L.G.rl;L.G.rl(1)]; zl = [L.G.zl;L.G.zl(1)];
  for ic = 1:SC.nc
    for il = 1:numel(rl)-1
      d = dist_line_point(rl(il),zl(il),rl(il+1),zl(il+1),gpr(ic),gpz(ic));
      if d<dist_limiter(ic); dist_limiter(ic) = d; end
    end
  end
  % Ignored points
  lcI = isnan(gpb);
  % Primary X-points
  lcX1 = (gpbr==0) & (gpbz==0) & (gpb==1);
  % Secondary X-points
  lcX2 = (gpbr==0) & (gpbz==0) & (gpb~=1);
  % Limiter points
  lcL = false(SC.nc,1);
  if ~any(lcX1)
    [d,iL] = min(dist_limiter(gpb==1));
    if d<tol
      lcL(iL) = true;
    else
      % Search for a primary X-point in the secondary ones
      if any(lcX2)
        LYt = fbtt(L,LXt); % Compute fbt equilibrium
        if LYt.lX
          icX2 = find(lcX2);
          [~,ii] = min((gpr(icX2)-LYt.rX(1)).^2 + (gpz(icX2)-LYt.zX(1)).^2);
          lcX1(icX2(ii)) = true;
          lcX2(icX2(ii)) = false;
        end
      else
        warning('Could not find a primary X-point or limiter point');
      end
    end
  end
  % Strike points
  lcS = (dist_limiter<=tol) & (gpb~=1) & (gpfe==0);
  % Contour points
  lcC = ~(lcL|lcS|lcX1|lcX2|lcI);
  % Separatrix point
  % Drsep point to track the radial distance between the flux surfaces
  lcD = false(nc,1);
  if any(lcX2)
    % Find the outermost radial contour point
    icC = find(lcC);
    [~,ii] = max(gpr(icC));
    lcD(icC(ii)) = true;
  end
  % Store data
  SC.rc(  :,it) = gpr;
  SC.zc(  :,it) = gpz;
  SC.lcC( :,it) = lcC;
  SC.lcL( :,it) = lcL;
  SC.lcS( :,it) = lcS;
  SC.lcX1(:,it) = lcX1;
  SC.lcX2(:,it) = lcX2;
  SC.lcD( :,it) = lcD;
  SC.lcI( :,it) = lcI;
  % Label points
  ii = SC.lcC( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Surf%02d'));
  ii = SC.lcL( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Limi%02d'));
  ii = SC.lcS( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Stri%02d'));
  ii = SC.lcX1(:,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Xpri%02d'));
  ii = SC.lcX2(:,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Xsec%02d'));
  ii = SC.lcD( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Dsep%02d'));
  ii = SC.lcI( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Ignr%02d'));
end

end

function d = dist_line_point(r1,z1,r2,z2,r,z)
% Calculate the distance between the line [r1,z1]--[r2,z2] and the point
% [r,z]
a1 = r1-r2; a2 = z1-z2;
b1 = r-r2;  b2 = z-z2;
len = sqrt(a1^2+a2^2);
if len <= 1e-3
  d = sqrt(b1^2+b2^2);
else
  u1 = a1/len; u2 = a2/len;
  v1 = b1/len; v2 = b2/len;
  t = max(0,min(1,u1*v1+u2*v2));
  w1 = a1*t; w2 = a2*t;
  d = sqrt((w1-b1)^2+(w2-b2)^2);
end
end
