function M = dlstc(nzx,nrx,drx,dzx,rry)
% DLSTC assembles the sparse operator matrix of the GS operator
% function M = dlstc(drx,dzx,rry)
% Arguments:
%   nzx: int, number of x-grid elements along Z-axis
%   nrx: int, number of x-grid elements along R-axis
%   drx: float, R-axis grid spacing
%   dzx: float, Z-axis grid spacing
%   rry: Matrix, Matrix containing R values for y-grid
% returns:
%   M:   Sparse Matrix, (ny, nx)-matrix representing the dlst operator
%
% For details, see: [J-M.Moret et al. Fus.Eng.Des 2015] Section 4.2
%              and ﻿[F. Hofmann, Comput. Phys. Commun. 48(2) 1988].
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

A = (dzx/drx)^2*rry./(rry + drx/2);
B = (dzx/drx)^2*rry./(rry - drx/2);
C = 2 + A + B;

% compute additional index quantities
nzy = nzx-2; nry = nrx-2;
nx = nzx*nrx; ny = nzy*nry;

% getting all y grid indices and adding one to make them the x grid indices
iy = 1:ny;
[izy, iry] = ind2sub([nzy, nry], iy);
ix = sub2ind([nzx,nrx], izy+1, iry+1);

% combining all indices as well as the coeffs of dlst into big vectors
inds_in    = [ix   , ix+1       , ix-1       , ix+nzx, ix-nzx];
inds_out   = [iy   , iy         , iy         , iy    , iy    ];
vals_diags = [-C(:); ones(ny, 1); ones(ny, 1); A(:)  ; B(:)  ];

% constructing the sparse operator
M = sparse(inds_out, inds_in, vals_diags, ny, nx);
end
