%BFPSP  Parameters for B-spline basis functions
%
% Constructs necessary data for evaluation of B-splines of order 5 with 0
% value at the edge and natural or not-a-knot boundary conditions for the
% first derivative.
% Assumes same knots distribution for p' and TT'.
%
% P = BFPSP(tP, tT, [,EC]) where tP and tT are the knot distributions for
% the discretization of p' and TT', respecively. The knots correspond to
% values of FN (between 0 and 1). EC sets the boundary condition for p' and
% TT'. Options for EC are:
%   'a' for not-a-knot. Implies a continuous f'''. Note that for this
%       option, the resulting g' basis functions are not necessarily normed at
%       the knots.
%   'n' for natural. Implies f''=0 at the boundary
%   'z' for zero bc. Implies f'=0 at the boundary. Equivalent to 'n' with
%       the degree of freedom for a non-zero value at the boundary removed
% The default value for EC is 'a'
%
% See also: BFSP, CSDEC, BSPSUM, BSPBASE
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function [bfp] = bfpsp(tP, tT, ec)
% arguments
if nargin <= 2
 ec = 'a';
end

[bfp.tauP, bfp.MP] = psp(tP, ec);
[bfp.tauT, bfp.MT] = psp(tT, ec);

end


function [tau, M] = psp(t, ec)
nt = numel(t);
if nt < 2, error('Give more than one knot.'); end

d = 4; % Degree of the polynomials for the integral bf

% augment knot sequence
dt = mean(diff(t));
tau = [(-d:-1)'*dt+t(1);t(:);t(nt)+(1:d)'*dt];

% With c[nt+d-1] the coefficient for each spline, and ag[nt] the
% coefficient of each basis function, we look for the matrix M[nt+d-1,nt]
% such that c = M*ag.
% The first set of constraints establishes that the ag correspond to f'
% values at the knot sequence t.
b = bspsum(tau, eye(nt+d-1), t, 1);
% M verifies b*M = eye(nt)

if nt < 4
  assert(ec == 'n' || ec == 'z', 'Only "n" and "z" modes are supported for < 4 knots');
end

% We need d-1 (3) other constraints, 2 boundary conditions for f' either
% natural or not-a-knot, plus f must be 0 at t(nt)
switch ec
  case {'n','z'}
    % natural: f'''(t(1)) = f'''(t(nt)) = 0
    T = bspsum(tau, eye(nt+d-1), t([1,nt]), 3) * dt^3;
  case 'a'
    % not-a-knot: f'''' equal at some mid-points
    T = [1 -1 0 0; 0 0 1 -1] * bspsum(tau, eye(nt+d-1), ...
      [t(1)+t(2) t(2)+t(3) t(nt-2)+t(nt-1) t(nt-1)+t(nt)]/2, 4) * dt^4;
  otherwise
    error('Invalid edge condition.')
end
% Add condition f(t(nt)) = 0
T = [bspsum(tau,eye(nt+d-1),t(nt));T];
% M verifies T*M = zeros(d-1,nt)

% Solve
M = [b;T] \ [eye(nt);zeros(d-1,nt)];

if ec == 'z'
  % Impose f'(t(nt)) = 0
  % This is equivalent to ag(nt)=0, so remove last column of M
  M = M(:,1:nt-1);
end

end
