%BFGEND generic multi-domain base function set
% BFGEND assembles single-domain basis function sets into one multi-domain
% basis function set.
% 
% The user-defined parameters P are defined as a cell array where each line
% is composed of the function handle for a single-domain basis function
% set, its parameters and finally a column vector indicating which domains
% it applies to (must be of size [nD,1]).
%   Example: {@bf3pmex,0,[1;0;0];@bf3pmex,0,[0;1;0];@bf3pmex,1,[1;1;1]}
% Use this syntax when specifying the MEQ parameter BFP.
%
% When called directly, the BFGEND function expects a consolidated version
% of these parameters through the BFPGEND function:
%   Pu = {@bf3pmex,0,[1;0;0];@bf3pmex,0,[0;1;0];@bf3pmex,1,[1;1;1]};
%   P  = bfpgenD(Pu,3); % For 3 domains
%   [...] = bfgenD(mode,P,...);
%
% Note: You can also use the consolidated structure when specifying the MEQ
% parameter BFP but no sanity checks will be performed.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function varargout = bfgenD(mode,P,F,F0,F1,q1,q2,q3,q4,q5)

nbf  = P.nbf;
bfct = P.bfct;
bfp  = P.bfp;
ngi  = P.ngi;
ng   = P.ng;
nD   = P.nD;
iDbf = P.iDbf;

% Check all domains are active, get axis and boundary flux for each sub-bf
% F0bf(ibf) and F1bf(ibf) are the minimum and maximum flux values over all
% the domains where the basis functions set ibf is active. If there is a
% single domain iD, then they are simply F0(iD) and F1(iD), other wise one
% has to determine the maximum/minimum of F0 and F1 over the domains on
% which the current basis function set is active.
if nargin >= 5
  Finc = all(F1>=F0);
  F0bf = zeros(nbf,1);
  F1bf = zeros(nbf,1);
  i0bf = zeros(nbf,1);
  i1bf = zeros(nbf,1);
  for ibf = 1:nbf
    iD = iDbf{ibf};
    if all(F1(iD) - F0(iD)) % All domains active
      if Finc
        [F0bf(ibf),im] = min(F0(iD)); i0bf(ibf) = iD(im);
        [F1bf(ibf),im] = max(F1(iD)); i1bf(ibf) = iD(im);
      else
        [F0bf(ibf),im] = max(F0(iD)); i0bf(ibf) = iD(im);
        [F1bf(ibf),im] = min(F1(iD)); i1bf(ibf) = iD(im);
      end
    end
  end
end

switch mode
  case 0  % [FPG,FTG,TDG          ] = BF( 0,P                            )
    varargout = {P.fPg,P.fTg,P.TDg};
  case 1  % [TYG,TPDG,ITPDG       ] = BF( 1,P, F,F0,F1,O ,RY,IRY         )
    ny = numel(q1);
    Tyg   = zeros(ny,ng);
    TpDg  = zeros(nD,ng);
    ITpDg = zeros(nD,ng);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      for iD = iDbf{ibf}
        [Tygi,TpDg(iD,ig),ITpDg(iD,ig)] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf),int8(q1==iD),q2,q3);
        Tyg(:,ig) = Tyg(:,ig) + Tygi;
      end
    end
    varargout = {Tyg,TpDg,ITpDg};
  case 2  % [GQDG,IGQDG           ] = BF( 2,P,FN,F0,F1                   )
    nQ = numel(F);
    gQDg  = zeros(nQ,nD,ng);
    IgQDg = zeros(nQ,nD,ng);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      idFbf = 1./(F1bf(ibf) - F0bf(ibf));
      % For bfs extending to multiple domain, the normalized flux in each
      % domain depends on the relative flux variation across all domains
      for iD = iDbf{ibf}
        FN = (F0(iD) + F*(F1(iD) - F0(iD)) - F0bf(ibf))*idFbf;
        [gQDg(:,iD,ig),IgQDg(:,iD,ig)] = bfct{ibf}(mode,bfp{ibf},FN,F0bf(ibf),F1bf(ibf));
      end
    end
    if nD == 1,  gQDg = reshape( gQDg,nQ,ng);
                IgQDg = reshape(IgQDg,nQ,ng);
    end
    varargout = {gQDg,IgQDg};
  case 3  % [APP,ATTP,AP,AHQT     ] = BF( 3,P, A,F0,F1,FP,FT,IDS         )
    aPpg  = zeros(ng,1);
    aTTpg = zeros(ng,1);
    aPg   = zeros(ng,1);
    ahqTg = zeros(ng,1);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      [aPpg(ig),aTTpg(ig),aPg(ig),ahqTg(ig)] = bfct{ibf}(mode,bfp{ibf},F(ig),F0bf(ibf),F1bf(ibf),q1(ig),q2(ig),q3);
    end
    varargout = {aPpg,aTTpg,aPg,ahqTg};
  case 4  % [TDG,TGY              ] = BF( 4,P, F,F0,F1,O ,KD, FD         )
    nr = size(q1,2);
    nd = numel(q2);
    Tdg = zeros(nd,ng);
    Tgy = zeros(ng,nr);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      q1i = q1.*int8(ismember(q1,iDbf{ibf})); % Opy for selected domains only
      [Tdg(:,ig),Tgy(ig,:)] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf),q1i,q2,q3);
    end
    varargout = {Tdg,Tgy};
  case 5  % [G0G,IG0G             ] = BF( 5,P, ~,F0,F1                   )
    % Because we want the value at all domain axes, if a bf set extends to
    % multiple domains (i.e. there is more than one true value in lD) we
    % will not evaluate each basis function only at F=F0(ibf) and so we
    % cannot use bfct{ibf}(5,...)
    g0g  = zeros(nD,ng); %  dg0gdF10
    Ig0g = zeros(nD,ng); % dIg0gdF10
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end
      mask = iDbf{ibf};
      [g0g(mask,ig),Ig0g(mask,ig)] = bfct{ibf}(91,bfp{ibf},F0(mask),F0bf(ibf),F1bf(ibf));
    end
    varargout = {g0g,Ig0g};
  case 6  % [QQG,XQ               ] = BF( 6,P, ~,F0,F1,R0,IR0,IDS        )
    Qqgi = cell(1,nbf);
    Xqi  = cell(1,nbf);
    for ibf = 1:nbf
      if i0bf(ibf) == 0 % Inactive bf
        [Qqgi{ibf},Xqi{ibf}] = bfct{ibf}(mode,bfp{ibf},F,0,1,1,1,q3); % Default value to not change size
      else
        [Qqgi{ibf},Xqi{ibf}] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf),q1(i0bf(ibf)),q2(i0bf(ibf)),q3);
      end
    end
    Qqg = blkdiag(Qqgi{:});
    Xq  = vertcat( Xqi{:});
    varargout = {Qqg,Xq};
  case 7  % [QCG,XC               ] = BF( 7,P, ~,F0,F1                   )
    Qcgi = cell(1,nbf);
    Xci  = cell(1,nbf);
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      if i0bf(ibf) == 0 % Inactive bf
        [Qcgi{ibf},Xci{ibf}] = bfct{ibf}(mode,bfp{ibf},F,0,1); % Default value to not change size
      else
        [Qcgi{ibf},Xci{ibf}] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf));
      end
    end
    Qcg = blkdiag(Qcgi{:});
    Xc  = vertcat( Xci{:});
    varargout = {Qcg,Xc};
  case 8  % [BTY                  ] = BF( 8,P, F,F0,F1,O ,  A,RBT,IDS,IRY)
    Ty = zeros(size(q1));
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      q1i = q1.*int8(ismember(q1,iDbf{ibf})); % Opy for selected domains only
      Ty = Ty + (bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf),q1i,q2(ig),q3,q4,q5))./q5.' - q3; % Ty = Bty./iry.'-rBt
    end
    Bty = (Ty+q3).*q5.';
    varargout = {Bty};
  case 11 % [DGY,DG0,DG1,DIG0,DIG1] = BF(11,P, F,F0,F1,O ,RY,IRY         )
    ny = numel(q1);
    dgy  = zeros(ny,ng);    %  dTygdFy
    dg0  = zeros(ny,ng,nD); %  dTygdF0
    dg1  = zeros(ny,ng,nD); %  dTygdF1
    dIg0 = zeros(ny,ng,nD); % dITygdF0
    dIg1 = zeros(ny,ng,nD); % dITygdF1
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      q1i = int8(ismember(q1,iDbf{ibf}));
      [ dgy(:,ig), dg0(:,ig,i0bf(ibf)), dg1(:,ig,i1bf(ibf)),...
                  dIg0(:,ig,i0bf(ibf)),dIg1(:,ig,i1bf(ibf))] = ...
        bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf),q1i,q2,q3);
    end
    varargout = {dgy,dg0,dg1,dIg0,dIg1};
  case 12 % [DGDF,...  ,DIGDF,... ] = BF(12,PAR, F,F0,F1                   )
    nQ = numel(F);
    dgQDgdFQ  = zeros(nQ,nD,ng);
    dgQDgdF0  = zeros(nQ,nD,ng,nD);
    dgQDgdF1  = zeros(nQ,nD,ng,nD);
    dIgQDgdFQ = zeros(nQ,nD,ng);
    dIgQDgdF0 = zeros(nQ,nD,ng,nD);
    dIgQDgdF1 = zeros(nQ,nD,ng,nD);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      idFbf = 1./(F1bf(ibf) - F0bf(ibf));
      % For bfs extending to multiple domain, the normalized flux in each
      % domain depends on the relative flux variation across all domains
      for iD = iDbf{ibf}
        FN = (F0(iD) + F*(F1(iD) - F0(iD)) - F0bf(ibf))*idFbf;
        % bfct{ibf} is a single-domain basis function set so independent of F0 or F1
        [dgNdFN,~,~,dIgNdFN,~,~] = ...
          bfct{ibf}(mode,bfp{ibf},FN,F0bf(ibf),F1bf(ibf));
        % g
        dgQDgdFQ(:,iD,ig)            =  dgNdFN.*(F1(iD)-F0(iD))*idFbf;
        dgQDgdF0(:,iD,ig,iD)         =  dgNdFN.*(1- F(:))*idFbf;
        dgQDgdF0(:,iD,ig,i0bf(ibf))  =  dgNdFN.*(FN(:)-1)*idFbf + reshape( dgQDgdF0(:,iD,ig,i0bf(ibf)),[nQ,ngi(ibf)]);
        dgQDgdF1(:,iD,ig,iD)         =  dgNdFN.*(   F(:))*idFbf;
        dgQDgdF1(:,iD,ig,i1bf(ibf))  =  dgNdFN.*( -FN(:))*idFbf + reshape( dgQDgdF1(:,iD,ig,i1bf(ibf)),[nQ,ngi(ibf)]);
        % Ig
        dIgQDgdFQ(:,iD,ig)           = dIgNdFN.*(F1(iD)-F0(iD))*idFbf;
        dIgQDgdF0(:,iD,ig,iD)        = dIgNdFN.*(1- F(:))*idFbf;
        dIgQDgdF0(:,iD,ig,i0bf(ibf)) = dIgNdFN.*(FN(:)-1)*idFbf + reshape(dIgQDgdF0(:,iD,ig,i0bf(ibf)),[nQ,ngi(ibf)]);
        dIgQDgdF1(:,iD,ig,iD)        = dIgNdFN.*(   F(:))*idFbf;
        dIgQDgdF1(:,iD,ig,i1bf(ibf)) = dIgNdFN.*( -FN(:))*idFbf + reshape(dIgQDgdF1(:,iD,ig,i1bf(ibf)),[nQ,ngi(ibf)]);
      end
    end
    if nD == 1,  dgQDgdFQ = reshape( dgQDgdFQ,nQ,ng);
                 dgQDgdF0 = reshape( dgQDgdF0,nQ,ng);
                 dgQDgdF1 = reshape( dgQDgdF1,nQ,ng);
                dIgQDgdFQ = reshape(dIgQDgdFQ,nQ,ng);
                dIgQDgdF0 = reshape(dIgQDgdF0,nQ,ng);
                dIgQDgdF1 = reshape(dIgQDgdF1,nQ,ng);
    end
    varargout = {dgQDgdFQ,dgQDgdF0,dgQDgdF1,dIgQDgdFQ,dIgQDgdF0,dIgQDgdF1};
  case 13 % [DAPPDF0,.. DAPPDF1,..] = BF(13,PAR, A,F0,F1,FP,FT,IDS         )
    daPpgdF0  = zeros(ng,nD);
    daTTpgdF0 = zeros(ng,nD);
    daPgdF0   = zeros(ng,nD);
    dahqTgdF0 = zeros(ng,nD);
    daPpgdF1  = zeros(ng,nD);
    daTTpgdF1 = zeros(ng,nD);
    daPgdF1   = zeros(ng,nD);
    dahqTgdF1 = zeros(ng,nD);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      [daPpgdF0(ig,i0bf(ibf)),daTTpgdF0(ig,i0bf(ibf)),daPgdF0(ig,i0bf(ibf)),dahqTgdF0(ig,i0bf(ibf)),...
        daPpgdF1(ig,i1bf(ibf)),daTTpgdF1(ig,i1bf(ibf)),daPgdF1(ig,i1bf(ibf)),dahqTgdF1(ig,i1bf(ibf))] = ...
        bfct{ibf}(mode,bfp{ibf},F(ig),F0bf(ibf),F1bf(ibf),q1(ig),q2(ig),q3);
    end
    varargout = {daPpgdF0,daTTpgdF0,daPgdF0,dahqTgdF0,daPpgdF1,daTTpgdF1,daPgdF1,dahqTgdF1};
  case 15 % [DG00,DG01,DIG00,DIG01] = BF(15,PAR, ~,F0,F1                   )
    % For the same reasons as for mode 5, we cannot use bfct{ibf}(15,...)
    dg0dF0  = zeros(nD,ng,nD);
    dg0dF1  = zeros(nD,ng,nD);
    dIg0dF0 = zeros(nD,ng,nD);
    dIg0dF1 = zeros(nD,ng,nD);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      % We only evaluate the basis functions on the axes of the domains
      % where they are active. So F0(mask) contains as many elements as
      % there are active domains for this set, while F0(ibf)/F1(ibf) are
      % always scalars.
      mask = iDbf{ibf}; % Select domains where this basis function is active
      [dg1,dg2,dg3,dIg1,dIg2,dIg3] = bfct{ibf}(92,bfp{ibf},F0(mask),F0bf(ibf),F1bf(ibf));
      for k = 1:numel(mask)
        dg0dF0(mask(k),ig,mask(k)) = dg1(k,:);
        dIg0dF0(mask(k),ig,mask(k)) = dIg1(k,:);
      end
      dg0dF0(mask,ig,i0bf(ibf)) = dg0dF0(mask,ig,i0bf(ibf)) + dg2;
      dIg0dF0(mask,ig,i0bf(ibf)) = dIg0dF0(mask,ig,i0bf(ibf)) + dIg2;
      dg0dF1(mask,ig,i1bf(ibf)) =  dg3;
      dIg0dF1(mask,ig,i1bf(ibf)) =  dIg3;
    end
    varargout = {dg0dF0,dg0dF1,dIg0dF0,dIg0dF1};
  case 16 % [DQGF0,DQGF1,DQGR0,...] = BF(16,PAR, ~,F0,F1,R0,IR0,IDS        )
    % First get matrices for individual bf sets
    dQqgdF0i = cell(1,nbf);
    dQqgdF1i = cell(1,nbf);
    dQqgdr0i = cell(1,nbf);
    dQqgdir0i = cell(1,nbf);
    nq = 0;
    for ibf = 1:nbf
      if i0bf(ibf) == 0 % Inactive bf
        [dQqgdF0i{ibf},dQqgdF1i{ibf},dQqgdr0i{ibf},dQqgdir0i{ibf}] = bfct{ibf}(mode,bfp{ibf},F,0,1,1,1,q3); % Default value to not change size
      else
        [dQqgdF0i{ibf},dQqgdF1i{ibf},dQqgdr0i{ibf},dQqgdir0i{ibf}] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf),q1(i0bf(ibf)),q2(i0bf(ibf)),q3);
      end
      nq = nq + size(dQqgdF0i{ibf},1);
    end
    % Now assemble full matrices
    dQqgdF0  = zeros(nq,ng,nD);
    dQqgdF1  = zeros(nq,ng,nD);
    dQqgdr0  = zeros(nq,ng,nD);
    dQqgdir0 = zeros(nq,ng,nD);
    iq = 0;
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      nQq = size(dQqgdF0i{ibf},1);
      if i0bf(ibf) == 0, continue; end % Inactive bf
      if nQq == 0, continue; end % no constraints for this bf
      iq = iq(end)+(1:nQq);
      dQqgdF0(iq,ig,i0bf(ibf)) = dQqgdF0i{ibf};
      dQqgdF1(iq,ig,i1bf(ibf)) = dQqgdF1i{ibf};
      dQqgdr0(iq,ig,i0bf(ibf)) = dQqgdr0i{ibf};
      dQqgdir0(iq,ig,i0bf(ibf)) = dQqgdir0i{ibf};
    end
    varargout = {dQqgdF0,dQqgdF1,dQqgdr0,dQqgdir0};
  case 17 % [DQCGDF0,DQCGDF1      ] = BF(17,PAR, ~,F0,F1                   )
    % First get matrices for individual bf sets
    dQcgdF0i = cell(1,nbf);
    dQcgdF1i = cell(1,nbf);
    nc = 0;
    for ibf = 1:nbf
      if i0bf(ibf) == 0 % Inactive bf
        [dQcgdF0i{ibf},dQcgdF1i{ibf}] = bfct{ibf}(mode,bfp{ibf},F,0,1); % Default value to not change size
      else
        [dQcgdF0i{ibf},dQcgdF1i{ibf}] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf));
      end
      nc = nc + size(dQcgdF0i{ibf},1);
    end
    % Now assemble full matrices
    dQcgdF0  = zeros(nc,ng,nD);
    dQcgdF1  = zeros(nc,ng,nD);
    ic = 0;
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      nQc = size(dQcgdF0i{ibf},1);
      if i0bf(ibf) == 0, continue; end % Inactive bf
      if nQc == 0, continue; end % no constraints for this bf
      ic = ic(end)+(1:nQc);
      dQcgdF0(ic,ig,i0bf(ibf)) = dQcgdF0i{ibf};
      dQcgdF1(ic,ig,i1bf(ibf)) = dQcgdF1i{ibf};
    end
    varargout = {dQcgdF0,dQcgdF1};
  case 91 % [G,IG                 ] = BF(91,PAR, F,F0,F1                   )
    nx = numel(F);
    g  = zeros(nx,ng);
    Ig = zeros(nx,ng);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      [g(:,ig),Ig(:,ig)] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf));
    end
    varargout = {g,Ig};
  case 92 % [DGX,DG0,DG1,DIGX,DIG0,DIG1] = BF(92,PAR, F,F0,F1              )
    nx = numel(F);
    dgdFx  = zeros(nx,ng);
    dgdF0  = zeros(nx,ng,nD);
    dgdF1  = zeros(nx,ng,nD);
    dIgdFx = zeros(nx,ng);
    dIgdF0 = zeros(nx,ng,nD);
    dIgdF1 = zeros(nx,ng,nD);
    ig = 0;
    for ibf = 1:nbf
      if ngi(ibf) == 0; continue; end % no bf coefficients for this set
      ig = ig(end)+(1:ngi(ibf));
      if i0bf(ibf) == 0, continue; end % Inactive bf
      [ dgdFx(:,ig), dgdF0(:,ig,i0bf(ibf)), dgdF1(:,ig,i1bf(ibf)),...
       dIgdFx(:,ig),dIgdF0(:,ig,i0bf(ibf)),dIgdF1(:,ig,i1bf(ibf))] = bfct{ibf}(mode,bfp{ibf},F,F0bf(ibf),F1bf(ibf));
    end
    varargout = {dgdFx,dgdF0,dgdF1,dIgdFx,dIgdF0,dIgdF1};
  otherwise
    error('bfgenD:unknownmode','Mode %d not implemented for %s',mode,mfilename);
end
end
