%BFFBT  FBT base functions for p',TT'
% BFFBT produces base functions pf*(pl+(1-pl)*FN)*(1-FN)^pe for p' and
% tf*(1-FN)^te and tf*(tl*(FN-1)-1)*FN*(1-FN)^te for TT'. 
% P = [pf pl pe tf tl te]. 
%
% For details see [MEQ-redbook]
%
% See also BFCT BFHELP and BFPFBT
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function varargout = bffbt(mode,P,varargin)

% Check shape and size of parameters
assert(isvector(P) && numel(P) == 6,'Parameters for bffbt must be in the form of a 6 element vector');
fPg = [1;0;0];
fTg = [0;1;1];

% Prepare structure for bfct
Pbfct.name = 'bffbt';
Pbfct.fPg  = fPg;
Pbfct.fTg  = fTg;
Pbfct.f    = [];
Pbfct.fN   = @(FQ) bffbt_fN(P,FQ);
Pbfct.fag  = [];
Pbfct.fA   = [];
Pbfct.df   = [];
Pbfct.dfN  = @(FQ) bffbt_dfN(P,FQ);
Pbfct.dfag = [];
Pbfct.dfA  = [];

% Call generic bfct
[varargout{1:nargout}] = bfct(mode,Pbfct,varargin{:});
end

function [gQ,IgQ] = bffbt_fN(P, FQ)
% BFFBT_FN Compute BFFBT normalized basis functions

% Compute normalized basis functions
pf=P(1);pl=P(2);pe=P(3);tf=P(4);tl=P(5);te=P(6);
F  = FQ(:);
f  = 1-F;
% For points at or outside the boundary, enforce gQ=IgQ=0
mask = f<=0;
f(mask) = 0;
x1 = pf*f.^pe;
x2 = tf*f.^te;
gQ = [(pl+(1-pl)*F).*x1,x2,-(tl*f+1).*F.*x2];
x1 = x1.*f;
x2 = x2.*f;
IgQ = [((-(pl+1/(pe+1))/(pe+2))+((pl-1)/(pe+2))*F).*x1,...
       (-1/(te+1))*x2,...
       ((tl/((te+2)*(te+3))+1/((te+1)*(te+2)))+((tl*(te+1)/((te+2)*(te+3))+1/(te+2))-(tl/(te+3))*F).*F).*x2];
end

function [dgQ,dIgQ] = bffbt_dfN(P, FQ)
% BFFBT_DFN Compute derivatives of BFFBT normalized basis functions

% Compute normalized basis functions
pf=P(1);pl=P(2);pe=P(3);tf=P(4);tl=P(5);te=P(6);
F  = FQ(:);
f  = 1-F;
% For points at or outside the boundary, enforce dgQ=dIgQ=0
mask = f<=0;
f(mask) = 0;
x1 = pf*f.^(pe-1);
x2 = tf*f.^(te-1);
if (pe-1<0 || te-1<0) && any(mask)
  % Avoid Inf/NaN values
  x1(mask) = 0;
  x2(mask) = 0;
end
dgQ = [(-pe*(pl+(1-pl)*F)+(1-pl)*f).*x1,-te.*x2,((te*F-f).*(tl*f+1)+tl*f.*F).*x2];
x1 = x1.*f;
x2 = x2.*f;
dIgQ = [(pl+(1-pl)*F).*x1,x2,-(tl*f+1).*F.*x2];
end
