%BFDOUBLET 3-domain base function set optimized for doublets with mantle
% BFDOUBLET assembles 3 single-domain basis function sets into one 3-domain
% basis function set.
%
% The user-defined parameters P are defined as a cell array where
% each line is composed of the function handle for a single-domain basis
% function set and its parameters.
% The first two basis function sets are used for the lobes. The final one
% is used on in the mantle and trivially continued inside the lobes to
% ensure consistency of integrated parameters.
% Concretely, derivative basis functions are continued as a constant inside
% the lobes and primitive basis functions as a linear function where the
% slope is given by the constant value of the corresponding derivative
% basis function.
%   Example: {@bfabmex,[1,2];@bfabmex,[1,2];@bfabmex,[1,1]}
% Use this syntax when specifying the MEQ parameter BFP.
%
% When called directly, the BFDOUBLET function expects a consolidated
% version of these parameters through the BFPDOUBLET function:
%   Pu = {@bfabmex,[1,2];@bfabmex,[1,2];@bfabmex,[1,1]};
%   P  = bfpdoublet(Pu);
%   [...] = bfdoublet(mode,P,...);
%
% Note: You can also use the consolidated structure when specifying the MEQ
% parameter BFP but no sanity checks will be performed.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function varargout = bfdoublet(mode, P, varargin)
  % Useful index sets
  igm = P.igm;
  % Consolidated parameters for bfgenD
  PbfgenD = P.PbfgenD;
  % Mantle bf characteristics
  fPg3 = P.fPg(igm);
  fTg3 = P.fTg(igm);
  bfct3 = P.PbfgenD.bfct{3};
  bfp3  = P.PbfgenD.bfp{3};
  
  % If there is no mantle, no reason to not just use bfgenD
  if nargin >= 5
    [F0, F1] = deal(varargin{2:3});
    if F0(3) == F1(3) % no mantle
      [varargout{1:nargout}] = bfgenD(mode,PbfgenD,varargin{:});
      return;
    end
  end
    
  % chinese menu
  switch mode
    case 0 % [FPG,FTG,TDG          ] = BF( 0,P                            )
      varargout = {P.fPg,P.fTg,P.TDg};
    case 1 % [TYG,TPDG,ITPDG       ] = BF( 1,P, F,F0,F1,O ,RY,IRY         )
      [Tyg,TpDg,ITpDg] = bfgenD(mode,PbfgenD,varargin{:});
      
      [Fx,F0,F1,Opy,ry,iry] = deal(varargin{:});
      Fy = Fx(2:end-1,2:end-1);
      [GA3, IGA3] = bfct3(5, bfp3, [], F0(3), F1(3));
      
      fac1 = reshape((Opy==1) .*  ry(:).',[],1).*fPg3.' + ...
             reshape((Opy==1) .* iry(:).',[],1).*fTg3.';
          
      fac2 = reshape((Opy==2) .*  ry(:).',[],1).*fPg3.' + ...
             reshape((Opy==2) .* iry(:).',[],1).*fTg3.';
          
      Tyg(:, igm)   = Tyg(:, igm) + (fac1 + fac2) .* GA3;
      TpDg(1, igm)  = TpDg(1, igm) + sum(fac1 .* GA3,1);
      TpDg(2, igm)  = TpDg(2, igm) + sum(fac2 .* GA3,1);
      
      ITpDg(1, igm)  = ITpDg(1, igm) + sum(fac1 .* (IGA3 + (Fy(:) - F0(3)) .* GA3), 1);
      ITpDg(2, igm)  = ITpDg(2, igm) + sum(fac2 .* (IGA3 + (Fy(:) - F0(3)) .* GA3), 1);
      
      varargout = {Tyg,TpDg,ITpDg};
    case 2 % [GQDG,IGQDG           ] = BF( 2,P,FN,F0,F1                   )
      [gQg,IgQg] = bfgenD(mode,PbfgenD,varargin{:});
      
      [gA3, IgA3] = bfct3(2, bfp3, 0, 0, 1);
      gA3  = reshape( gA3, 1, 1, []);
      IgA3 = reshape(IgA3, 1, 1, []);
      
      FN = varargin{1}(:);
      gQg( :, 1:2, igm) =  gQg(:, 1:2, igm) + gA3;
      IgQg(:, 1:2, igm) = IgQg(:, 1:2, igm) + IgA3;
      
      % we have to scale the added linear ramp in the first two domains by
      % (F0(3) - F0(1,2)) / (F1(3) - F0(3))
      IgQg(:, 1, igm) = IgQg(:, 1, igm) + (FN - 1) .* gA3 * (F0(3) - F0(1)) / (F1(3) - F0(3));
      IgQg(:, 2, igm) = IgQg(:, 2, igm) + (FN - 1) .* gA3 * (F0(3) - F0(2)) / (F1(3) - F0(3));
      
      varargout = {gQg,IgQg};
    case 3 % [APP,ATTP,AP,AHQT     ] = BF( 3,P, A,F0,F1,FP,FT,IDS         )
      [varargout{1:4}] = bfgenD(mode,PbfgenD,varargin{:});
    case 4 % [TDG,TGY              ] = BF( 4,P, F,F0,F1,O ,KD, FD         )
      [~,Tgy] = bfgenD(mode,PbfgenD,varargin{:});
      [~,F0,F1,Opy,kd,fd] = deal(varargin{:});
      
      [GA3, ~] = bfct3(5, bfp3, [], F0(3), F1(3));
      Opy = Opy==1 | Opy==2;
      
      Tgy(:, igm) = Tgy(:, igm) + sum(Opy .* GA3);
      Tdg = (Tgy(:,kd+1).*fd(1,:) + Tgy(:,kd+2).*fd(2,:)).';
      
      varargout = {Tdg, Tgy};
    case 5 % [G0G,IG0G             ] = BF( 5,P, ~,F0,F1                   )
      [GAg,IGAg] = bfgenD(mode,PbfgenD,varargin{:});
      
      GAg( 1:2, igm) =  GAg(1:2, igm) +  GAg(3, igm);
      IGAg(1  , igm) = IGAg(1  , igm) + IGAg(3, igm) + (F0(3) - F0(1)) * GAg(3, igm);
      IGAg(2  , igm) = IGAg(2  , igm) + IGAg(3, igm) + (F0(3) - F0(2)) * GAg(3, igm);
      
      varargout = {GAg, IGAg};
    case 6 % [QQG,XQ               ] = BF( 6,P, ~,F0,F1,R0,IR0,IDS        )
      [Qqg,Xq] = bfgenD(mode,PbfgenD,varargin{:});
      varargout = {Qqg,Xq};
    case 7 % [QCG,XC               ] = BF( 7,P, ~,F0,F1                   )
      [Qcg,Xc] = bfgenD(mode,PbfgenD,varargin{:});
      varargout = {Qcg, Xc};
    case 8 % [BTY                  ] = BF( 8,P, F,F0,F1,O ,  A,RBT,IDS,IRY)
      [Bty] = bfgenD(mode,PbfgenD,varargin{:});
      [Fx,F0,F1,Opy,ag,rBt,ids,iry] = deal(varargin{:});
      
      [GA3, IGA3] = bfct3(5, bfp3, [], F0(3), F1(3));
      G = GA3 * (ag(igm).*fTg3);
      IG = IGA3 * (ag(igm).*fTg3);
      
      Ty = Bty ./ iry.' - rBt;
      Ty = Ty + (ids*2e-7/rBt) * (IG + (Fx(2:end-1, 2:end-1) - F0(3)) * G) .* (Opy==1);
      Ty = Ty + (ids*2e-7/rBt) * (IG + (Fx(2:end-1, 2:end-1) - F0(3)) * G) .* (Opy==2);
      Bty = (Ty + rBt) .* iry.';
      
      varargout = {Bty};
    case 11 % [DGY,DG0,DG1,DIG0,DIG1] = BF(11,P, F,F0,F1,O ,RY,IRY        )
      [dTygdFy,dTygdF0,dTygdF1,dITygdF0,dITygdF1] = bfgenD(mode,PbfgenD,varargin{:});
      
      [Fx,F0,F1,Opy,ry,iry] = deal(varargin{:});
      Fy = Fx(2:end-1, 2:end-1);
      [GA3, ~] = bfct3(5, bfp3, [], F0(3), F1(3));
      [dGA3dF0, dGA3dF1, dIGA3dF0, dIGA3dF1] = bfct3(15, bfp3, [], F0(3), F1(3));
      
      fac1 = reshape((Opy==1) .*  ry(:).',[],1).*fPg3.' + ...
             reshape((Opy==1) .* iry(:).',[],1).*fTg3.';
          
      fac2 = reshape((Opy==2) .*  ry(:).',[],1).*fPg3.' + ...
             reshape((Opy==2) .* iry(:).',[],1).*fTg3.';
      
      dTygdF0(:, igm, 3) = dTygdF0(:, igm, 3) + (fac1 + fac2) .* dGA3dF0;
      dTygdF1(:, igm, 3) = dTygdF1(:, igm, 3) + (fac1 + fac2) .* dGA3dF1;
      
      dITygdF0(:, igm, 3) = dITygdF0(:, igm, 3) + ...
                            fac1 .* (dIGA3dF0 + (Fy(:) - F0(3)) .* dGA3dF0 - GA3) + ...
                            fac2 .* (dIGA3dF0 + (Fy(:) - F0(3)) .* dGA3dF0 - GA3);
      dITygdF1(:, igm, 3) = dITygdF1(:, igm, 3) + ...
                            fac1 .* (dIGA3dF1 + (Fy(:) - F0(3)) .* dGA3dF1) + ...
                            fac2 .* (dIGA3dF1 + (Fy(:) - F0(3)) .* dGA3dF1);
      
      varargout = {dTygdFy,dTygdF0,dTygdF1,dITygdF0,dITygdF1};
    case 12 % [DGDF,...  ,DIGDF,... ] = BF(12,PAR, F,F0,F1                )
      [dgQgdFN,dgQdF0,dgQdF1,dIgQgdFN,dIgQdF0,dIgQdF1] = bfgenD(mode,PbfgenD,varargin{:});
      
      FN = varargin{1}(:);
      [gA3, ~] = bfct3(2, bfp3, 0, 0, 1);
      [~,dgA3dF0,dgA3dF1,~,dIgA3dF0,dIgA3dF1] = bfct3(12, bfp3, 0, 0, 1);
      iFM = 1 / (F1(3) - F0(3)); % Inverse mantle F range
      
      % reshaping to fit 3d dimension (ig-dimension)
      gA3 = reshape(gA3, 1, 1, []);
      dgA3dF0 = reshape(dgA3dF0, 1, 1, []); dgA3dF1 = reshape(dgA3dF1, 1, 1, []);
      dIgA3dF0 = reshape(dIgA3dF0, 1, 1, []); dIgA3dF1 = reshape(dIgA3dF1, 1, 1, []);
      
      dgQdF0(:, 1:2, igm, 3) = dgQdF0(:, 1:2, igm, 3) + dgA3dF0;
      dgQdF1(:, 1:2, igm, 3) = dgQdF1(:, 1:2, igm, 3) + dgA3dF1;
      
      dIgQgdFN(:, 1, igm) = dIgQgdFN(:, 1, igm) + gA3 * (F0(3) - F0(1)) * iFM;
      dIgQgdFN(:, 2, igm) = dIgQgdFN(:, 2, igm) + gA3 * (F0(3) - F0(2)) * iFM;
      
      dIgQdF0( :, 1:2, igm, 3) = dIgQdF0( :, 1:2, igm, 3) + dIgA3dF0 + ...
                                 (FN - 1) .* reshape(F0(3) - F0(1:2), 1, []) .* ...
                                 (dgA3dF0 + gA3 * iFM * iFM) + ...
                                 (FN - 1) .* gA3 * iFM;
      dIgQdF0( :, 1, igm, 1)  = -(FN - 1) .* gA3 * iFM;
      dIgQdF0( :, 2, igm, 2)  = -(FN - 1) .* gA3 * iFM;
      
      dIgQdF1( :, 1:2, igm, 3) = dIgQdF1( :, 1:2, igm, 3) + dIgA3dF1 + ...
                                 (FN - 1) .* reshape(F0(3) - F0(1:2), 1, []) .* ...
                                 (dgA3dF1 - gA3 * iFM * iFM);
      
      varargout = {dgQgdFN,dgQdF0,dgQdF1,dIgQgdFN,dIgQdF0,dIgQdF1};
    case 13 % [DAPPDF0,.. DAPPDF1,..] = BF(13,PAR, A,F0,F1,FP,FT,IDS      )
      [varargout{1:8}] = bfgenD(mode,PbfgenD,varargin{:});
    case 15 % [DG00,DG01,DIG00,DIG01] = BF(15,PAR, ~,F0,F1                )
      [GAg,~] = bfgenD(5,PbfgenD,varargin{:});
      [dGAdF0,dGAdF1,dIGAdF0,dIGAdF1] = bfgenD(mode,PbfgenD,varargin{:});
      
      dGAdF0(1:2, igm, 3) = dGAdF0(1:2, igm, 3) + dGAdF0(3, igm, 3);
      dGAdF1(1:2, igm, 3) = dGAdF1(1:2, igm, 3) + dGAdF1(3, igm, 3);
      
      dIGAdF0(1:2, igm, 3) = dIGAdF0(1:2, igm, 3) + dIGAdF0(3, igm, 3) + (F0(3) - F0(1:2)) .* dGAdF0(3, igm, 3) + GAg(3, igm);
      dIGAdF1(1:2, igm, 3) = dIGAdF1(1:2, igm, 3) + dIGAdF1(3, igm, 3) + (F0(3) - F0(1:2)) .* dGAdF1(3, igm, 3);
      
      dIGAdF0(1, igm, 1) = dIGAdF0(1, igm, 1) - GAg(3, igm);
      dIGAdF0(2, igm, 2) = dIGAdF0(2, igm, 2) - GAg(3, igm);
      
      varargout = {dGAdF0,dGAdF1,dIGAdF0,dIGAdF1};
    case 16 % [DQGF0,DQGF1,DQGR0,...] = BF(16,PAR, ~,F0,F1,R0,IR0,IDS     )
      [varargout{1:4}] = bfgenD(mode,PbfgenD,varargin{:});
    case 17 % [DQCGDF0,DQCGDF1      ] = BF(17,PAR, ~,F0,F1                )
      [varargout{1:2}] = bfgenD(mode,PbfgenD,varargin{:});
    case 91 % [G,IG                 ] = BF(91,PAR, F,F0,F1                )
      [G,IG] = bfgenD(mode,PbfgenD,varargin{:});
      
      [F,F0,F1] = deal(varargin{1:3});
      [GA3, IGA3] = bfct3(5, bfp3, [], F0(3), F1(3));
      sIp = sign(F1(3) - F0(3));
      mask = sIp * F(:) < sIp * F0(3);
      
      G(:, igm) = G(:, igm) .* ~mask + GA3 .* mask;
      IG(:, igm) = IG(:, igm) .* ~mask + (IGA3 + (F(:) - F0(3)) .* GA3) .* mask;
      varargout = {G,IG};
    case 92 % [DGX,...   DIGX,...   ] = BF(92,PAR, F,F0,F1                   )
      [dGdFx,dGdF0,dGdF1,dIGdFx,dIGdF0,dIGdF1] = bfgenD(mode,PbfgenD,varargin{:});
      
      [F,F0,F1] = deal(varargin{1:3});
      [GA3, ~] = bfct3(5, bfp3, [], F0(3), F1(3));
      [dGA3dF0, dGA3dF1, dIGA3dF0, dIGA3dF1] = bfct3(15, bfp3, [], F0(3), F1(3));
      sIp = sign(F1(3) - F0(3));
      mask = sIp * F(:) < sIp * F0(3);
      
      dGdFx(:, igm   ) = dGdFx(:, igm   ) .* ~mask;
      dGdF0(:, igm, :) = dGdF0(:, igm, :) .* ~mask;
      dGdF0(:, igm, 3) = dGdF0(:, igm, 3) + dGA3dF0 .* mask;
      dGdF1(:, igm, :) = dGdF1(:, igm, :) .* ~mask;
      dGdF1(:, igm, 3) = dGdF1(:, igm, 3) + dGA3dF1 .* mask;
      
      dIGdFx(:, igm   ) = dIGdFx(:, igm   ) .* ~mask + GA3 .* mask;
      dIGdF0(:, igm, :) = dIGdF0(:, igm, :) .* ~mask;
      dIGdF0(:, igm, 3) = dIGdF0(:, igm, 3) + (dIGA3dF0 + (F(:) - F0(3)) .* dGA3dF0 - GA3).* mask;
      dIGdF1(:, igm, :) = dIGdF1(:, igm, :) .* ~mask;
      dIGdF1(:, igm, 3) = dIGdF1(:, igm, 3) + (dIGA3dF1 + (F(:) - F0(3)) .* dGA3dF1      ).* mask;
      
      varargout = {dGdFx,dGdF0,dGdF1,dIGdFx,dIGdF0,dIGdF1};
    otherwise
      [varargout{1:nargout}] = bfgenD(mode,PbfgenD,varargin{:});
  end
end
