%BFCT generic basis-function evaluation
%
% This function allows the definition of a basis function set for MEQ with
% a minimum set of requirements, namely a function handle to compute the
% values of each basis functions at a given set of values for the
% normalized poloidal flux and assigning each function to either P' or TT'.
%
%   VARARGOUT = BFCT(MODE,PAR,VARARGIN)
%
% The different values for MODE and the corresponding requirements for
% VARARGIN are as follows:
%   [FPG,FTG,TDG          ] = BF( 0,PAR                            )
%   [TYG,TPDG,ITPDG       ] = BF( 1,PAR, F,F0,F1,O ,RY,IRY         )
%   [GQDG,IGQDG           ] = BF( 2,PAR,FN,F0,F1                   )
%   [APP,ATTP,AP,AHQT     ] = BF( 3,PAR, A,F0,F1,FP,FT,IDS         )
%   [TDG,TGY              ] = BF( 4,PAR, F,F0,F1,O ,KD, FD         )
%   [G0G,IG0G             ] = BF( 5,PAR, ~,F0,F1                   )
%   [QQG,XQ               ] = BF( 6,PAR, ~,F0,F1,R0,IR0,IDS        )
%   [QCG,XC               ] = BF( 7,PAR, ~,F0,F1                   )
%   [BTY                  ] = BF( 8,PAR, F,F0,F1,O ,  A,RBT,IDS,IRY)
%   [DGY,DG0,DG1,DIG0,DIG1] = BF(11,PAR, F,F0,F1,O ,RY,IRY         )
%   [DGDF,...  ,DIGDF,... ] = BF(12,PAR, F,F0,F1                   )
%   [DAPPDF0,.. DAPPDF1,..] = BF(13,PAR, A,F0,F1,FP,FT,IDS         )
%   [DG00,DG01,DIG00,DIG01] = BF(15,PAR, ~,F0,F1                   )
%   [DQGF0,DQGF1,DQGR0,...] = BF(16,PAR, ~,F0,F1,R0,IR0,IDS        )
%   [DQCGF0,DQCGDF1       ] = BF(17,PAR, ~,F0,F1                   )
%   [G,IG                 ] = BF(91,PAR, F,F0,F1                   )
%   [DGX,...   ,DIGX,...  ] = BF(92,PAR, F,F0,F1                   )
% See BFHELP for a description of each MODE value
%
% PAR is a structure with the following fields:
%   - name: name of the basis function set
%   - fPg : vector of size ng equal to 0 or 1 if the basis function contributes to P'
%   - fTg : vector of size ng equal to 0 or 1 if the basis function contributes to TT'
%           (needs to be equal to 1-fPg)
%   - f   : if non-empty, handle to a function with 2 input arguments FxA=Fx-FA and
%           FBA=FB-FA and returning the values of the physical basis functions and
%           their primitives at the corresponding values of Fx, FA and FB.
%   - fN  : handle to a function (FN) that returns the values of the normalized
%           basis functions and their primitives at the corresponding values for the
%           normalized flux.
%           If FN has N elements then the function should return two
%           arguments of size [N,ng] each.
%   - fag : if non-empty, handle to a function (FBA=FB-FA) that returns the
%           conversion coefficients from normalized to physical basis
%           functions
%   - fA  : if non-empty, handle to a function (FBA=FB-FA) that returns the
%           values of the physical basis functions and their primitives at Fx=FA
%   - df  : if non-empty, handle to a function (FxA=Fx-FA, FBA=FB-FA) returning the
%           values of the derivatives of the physical basis functions and their
%           primitives with respect to FxA and FBA
%   - dfN : handle to a function (FN) that returns the values of the derivatives
%           of the normalized basis functions and their primitives with respect to FN
%   - dfag: if non-empty, handle to a function (FBA=FB-FA) that returns the values
%           of the derivatives of the conversion coefficents fag with respect to FBA
%   - dfA:  if non-empty, handle to a function (FBA=FB-FA) that returns the values
%           of the derivatives of the physical basis functions and their derivatives
%           at the magnetic axis (derivatives with respect to FBA)
%
% If f, fag, fA, df, dfag or dfA are empty, then some generic replacements are
% provided assuming that:
%   g(Fx,FA,FB) =  gN((Fx-FA)/(FB-FA))
%  Ig(Fx,FA,FB) = IgN((Fx-FA)/(FB-FA))*(FB-FA)
% Meaning that the conversion coefficients from normalized to physical
% basis functions are
%  alphapg = 1     for the basis functions
%  alphag  = FB-FA for their primitives
%
% BFSP, BFFBT and the MATLAB implementations of BFAB and BFEF are examples
% of basis function sets using BFCT.
%
% For details see: [MEQ-redbook]
%
% See also: BFHELP
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

function varargout = bfct(mode,P,varargin)

% Optional parameters (function handles)
%  Order is important since fag can be used in f and f can be used in fA
if isempty(P.fag)
  P.fag = @(FBA) bfct_fag(P,FBA);
end
if isempty(P.f)
  P.f = @(FxA,FBA) bfct_f(P,FxA,FBA);
end
if isempty(P.fA)
  P.fA = @(FBA) bfct_fA(P,FBA);
end
% Derivatives
if isempty(P.dfag)
  P.dfag = @(FBA) bfct_dfag(P,FBA);
end
if isempty(P.df)
  P.df = @(FxA,FBA) bfct_df(P,FxA,FBA);
end
if isempty(P.dfA)
  P.dfA = @(FBA) bfct_dfA(P,FBA);
end

% Extract parameters
name = P.name;
fPg  = P.fPg;
fTg  = P.fTg;
f    = P.f;
fN   = P.fN;
fag  = P.fag;
fA   = P.fA;
df   = P.df;
dfN  = P.dfN;
dfag = P.dfag;
dfA  = P.dfA;
% Total number of basis functions
ng   = numel(fPg);

% Check this is not used for multiple domains
if nargin>4
  assert(numel(varargin{2}) <= 1 && numel(varargin{3}) <= 1,'bfct:multiD',...
    'To use %s with multiple domains, please use bfgenD',name);
end

% Identifier and text for errors when number of arguments is incorrect
msgid  = 'bfct:nargin';
msgerr = 'bfct mode %d requires exactly %d arguments';

% Chinese menu
switch mode
  case 0
    % Assignment to profiles and domains
    % [FP,FT,TDG            ] = BF( 0,PAR                            )
    TDg = ones(1,ng);
    varargout = {fPg,fTg,TDg};
  case 1
    % Transfer matrix and integrals over plasma domain
    % [TYG,TPDG,ITPDG       ] = BF( 1,PAR, F,F0,F1,OY,RY,IRY         )

    % Handle inputs
    if nargin<8, error(msgid,msgerr,mode,8); end
    [Fx,FA,FB,Opy,ry,iry] = deal(varargin{:});

    % Convert Opy to logical
    Opy = logical(Opy);
    % Evaluate G and IG at each flux point
    [G,IG] = f(Fx(2:end-1,2:end-1)-FA,FB-FA);
    % Combine with r or 1/r
    fac = reshape(Opy.* ry(:).',[],1).*fPg.' + ...
          reshape(Opy.*iry(:).',[],1).*fTg.'; % fPg*r+fTg/r in plasma domain
    Tyg  = fac.* G;
    ITyg = fac.*IG;
    % Compute the integrals
    Tpg  = sum( Tyg,1);
    ITpg = sum(ITyg,1);

    % Assign outputs
    varargout = {Tyg,Tpg,ITpg};
  case 2
    % Normalized basis functions
    % [GDQG,IGDQG           ] = BF( 2,PAR,FN,F0,F1                   )

    % Handle inputs
    FQ = varargin{1};

    % Evaluate normalized basis functions at each normalized flux value
    [gQg,IgQg] = fN(FQ);

    % Assign outputs
    varargout = {gQg,IgQg};
  case 3
    % Coefficients to transform normalized basis functions into Pp/P/TTp/hqT
    % [APP,ATTP,AP,AHQT     ] = BF( 3,PAR, A,F0,F1,FP,FT,IDS         )

    % Handle inputs
    if nargin<8, error(msgid,msgerr,mode,8); end
    [ag,FA,FB,fPg,fTg,ids] = deal(varargin{:});

    % Conversion coefficients from normalized to physical basis functions
    [alphapg,alphag] = fag(FB-FA);
    cP = ids/2/pi;
    cT = cP*mu0;
    aPpg  = cP*alphapg.*ag.*fPg;
    aTTpg = cT*alphapg.*ag.*fTg;
    aPg   = cP*alphag .*ag.*fPg;
    ahqTg = cT*alphag .*ag.*fTg;

    % Assign outputs
    varargout = {aPpg,aTTpg,aPg,ahqTg};
  case 4
    % Vertical chord integrals
    % [TDG,TGY              ] = BF( 4,PAR, F,F0,F1,O ,KD, FD         )

    % Handle inputs
    if nargin<8, error(msgid,msgerr,mode,8); end
    [Fx,FA,FB,Opy,kd,fd] = deal(varargin{:});

    % Convert Opy to logical
    Opy = logical(Opy);
    % Evaluate G at each flux point
    G = f(Fx(2:end-1,2:end-1)-FA,FB-FA);
    % Compute vertical integrals
    [nzy,nry] = size(Opy); ng = size(G,2);
    Tgy = reshape(sum(reshape(G,nzy,nry,ng).*Opy,1),[nry,ng]).';
    % Interpolate
    Tdg = (Tgy(:,kd+1).*fd(1,:) + Tgy(:,kd+2).*fd(2,:)).';

    % Assign outputs
    varargout = {Tdg,Tgy};
  case 5
    % Physical basis functions at magnetic axis
    % [G0G,IG0G             ] = BF( 5,PAR, ~,F0,F1                   )

    % Handle inputs
    if nargin<5, error(msgid,msgerr,mode,5); end
    [~,FA,FB] = deal(varargin{1:3});

    % Evaluate G at the magnetic axis
    [gAg, IgAg] = fA(FB-FA);

    % Assign outputs
    varargout = {gAg,IgAg};
  case 6
    % Regularization constraints
    % [QQG,XQ               ] = BF( 6,PAR, ~,F0,F1,R0,IR0,IDS        )

    % No regularization constraints
    y1 = zeros(0,ng);
    y2 = zeros(0, 1);

    % Assign outputs
    varargout = {y1,y2};
  case 7
    % Inequality constraints
    % [QCG,XC               ] = BF( 7,PAR, ~,F0,F1                   )

    % No inequality constraints
    y1 = zeros(0,ng);
    y2 = zeros(0, 1);

    % Assign outputs
    varargout = {y1,y2};
  case 8
    % Toroidal magnetic field amplitude
    % [BTY                  ] = BF( 8,PAR, F,F0,F1,O ,  A,RBT,IDS,IRY)

    % Handle inputs
    if nargin<10, error(msgid,msgerr,mode,10); end
    [Fx,FA,FB,Opy,ag,rBt,ids,iry] = deal(varargin{:});

    % Convert Opy to logical
    Opy = logical(Opy);
    % Evaluate IG at each flux point
    [~, IG] = f(Fx(2:end-1,2:end-1)-FA,FB-FA);
    % Compute toroidal magnetic field in small diamagnetism approx.
    rBty = (rBt + (ids*2e-7/rBt)*sum(IG.*reshape(ag.*fTg,1,[]),2).*Opy(:));
    Bty = reshape(rBty,size(Opy)).*iry.';

    % Assign outputs
    varargout = {Bty};
  case 11
    % Derivatives of transfer matrices
    % [DGY,DG0,DG1,DIG0,DIG1] = BF(11,PAR, F,F0,F1,O ,RY,IRY         )

    % Handle inputs
    if nargin<8, error(msgid,msgerr,mode,8); end
    [Fx,FA,FB,Opy,ry,iry] = deal(varargin{:});

    % Convert Opy to logical
    Opy = logical(Opy);
    % Evaluate DG and DIG at each flux point
    [DGFXA,DGFBA,DIGFXA,DIGFBA] = df(Fx(2:end-1,2:end-1)-FA,FB-FA);
    % Combine with r or 1/r
    fac = reshape(Opy.* ry(:).',[],1).*fPg.' + ...
          reshape(Opy.*iry(:).',[],1).*fTg.'; % fPg*r+fTg/r in plasma domain
    % Also convert to derivatives with respect to Fy, F0 (FA) and F1 (FB)
    dTygdFy  =  fac.*  DGFXA; 
    dTygdF0  = -fac.*( DGFXA+ DGFBA);
    dTygdF1  =  fac.*  DGFBA;
    dITygdF0 = -fac.*(DIGFXA+DIGFBA);
    dITygdF1 =  fac.* DIGFBA;

    % Assign outputs
    varargout = {dTygdFy,dTygdF0,dTygdF1,dITygdF0,dITygdF1};
  case 12
    % Derivatives of normalized basis functions
    % [DGDF,...  ,DIGDF,... ] = BF(12,PAR, F,F0,F1                   )

    % Handle inputs
    FQ = varargin{1};
    nQ = numel(FQ);

    % Evaluate normalized basis functions at each normalized flux value
    % For single-domain basis functions this is independent of F0/F1
    [dgQgdFN,dIgQgdFN] = dfN(FQ);
    dgQdF0  = zeros(nQ,ng);
    dgQdF1  = zeros(nQ,ng);
    dIgQdF0 = zeros(nQ,ng);
    dIgQdF1 = zeros(nQ,ng);

    % Assign outputs
    varargout = {dgQgdFN,dgQdF0,dgQdF1,dIgQgdFN,dIgQdF0,dIgQdF1};
  case 13
    % Derivatives of coefficients to transform normalized basis functions into Pp/P/TTp/hqT
    % [DAPPDF0,.. DAPPDF1,..] = BF(13,PAR, A,F0,F1,FP,FT,IDS         )

    % Handle inputs
    if nargin<8, error(msgid,msgerr,mode,8); end
    [ag,FA,FB,fPg,fTg,ids] = deal(varargin{:});

    % Conversion coefficients from normalized to physical basis functions
    [dalphapg,dalphag] = dfag(FB-FA);
    cP = ids/2/pi;
    cT = cP*mu0;
    daPpg  = cP*dalphapg.*ag.*fPg;
    daTTpg = cT*dalphapg.*ag.*fTg;
    daPg   = cP*dalphag .*ag.*fPg;
    dahqTg = cT*dalphag .*ag.*fTg;

    % Assign outputs
    varargout = {-daPpg,-daTTpg,-daPg,-dahqTg,daPpg,daTTpg,daPg,dahqTg};
  case 15
    % Physical basis functions derivatives at magnetic axis
    % [DG00,DG01,DIG00,DIG01] = BF(15,PAR, ~,F0,F1                   )

    % Handle inputs
    if nargin<5, error(msgid,msgerr,mode,5); end
    [~,FA,FB] = deal(varargin{1:3});

    % Evaluate derivatives at the magnetic axis
    [dgAg, dIgAg] = dfA(FB-FA);
    % These are derivatives with respect to FBA, but we need derivatives
    % with respect to FA and FB

    % Assign outputs
    varargout = {-dgAg,dgAg,-dIgAg,dIgAg};
  case 16
    % Derivatives of regularization constraints
    % [DQGF0,DQGF1,DQGR0,...] = BF(16,PAR, ~,F0,F1,R0,IR0,IDS        )

    % No regularization constraints
    y1 = zeros(0,ng,1);
    y2 = zeros(0,ng,1);
    y3 = zeros(0,ng,1);
    y4 = zeros(0,ng,1);
    
    % Assign outputs
    varargout = {y1,y2,y3,y4};
  case 17
    % Derivatives of inequality constraints
    % [DQCGDF0,DQCGDF1      ] = BF(17,PAR, ~,F0,F1,                  )

    % No inequality constraints
    y1 = zeros(0,ng,1);
    y2 = zeros(0,ng,1);
    
    % Assign outputs
    varargout = {y1,y2};
  case 91
    % Physical basis functions as function of F,FA,FB
    % [G,IG                 ] = BF(91,PAR, F,F0,F1                   )

    % Handle inputs
    if nargin<5, error(msgid,msgerr,mode,5); end
    [F,FA,FB] = deal(varargin{1:3});

    % Evaluate physical basis functions
    [g,Ig] = f(F-FA,FB-FA);

    % Assign outputs
    varargout = {g,Ig};
  case 92
    % Physical basis functions derivatives as function of F,FA,FB
    % [DGXA,DGBA,DIGXA,DIGBA] = BF(92,PAR, F,F0,F1                   )

    % Handle inputs
    if nargin<5, error(msgid,msgerr,mode,5); end
    [F,FA,FB] = deal(varargin{1:3});

    % Evaluate physical basis functions
    [dgdFxA,dgdFBA,dIgdFxA,dIgdFBA] = df(F-FA,FB-FA);

    % Assign outputs
    varargout = {dgdFxA,-dgdFxA-dgdFBA,dgdFBA,dIgdFxA,-dIgdFxA-dIgdFBA,dIgdFBA};
  otherwise
    error([name,':unknownmode'],'Mode %d not implemented for %s',mode,name);
end
end

%% Default functions
% This functions are used for evaluation of physical basis functions when
% they are not provided in the parameter structure

function [alphapg,alphag] = bfct_fag(P,FBA)
% BFCT_FAG Conversion factors from normalized to physical basis functions

alphapg = ones(numel(P.fPg),1);
alphag  = alphapg*FBA;
end

function [g,Ig] = bfct_f(P, FxA, FBA)
% BFCT_F Compute physical basis functions using normalized basis functions

% Compute normalized basis functions
[gN,IgN] = P.fN((FxA(:))/FBA);
% Conversion to physical basis functions
[alphapg,alphag] = P.fag(FBA);
 g =  gN.*alphapg.';
Ig = IgN.*alphag.';
end

function [gAg,IgAg] = bfct_fA(P, FBA)
% BFCT_FA Compute physical basis functions at magnetic axis

% Use computation of physical basis functions at FxA=0
[gAg,IgAg] = P.f(0,FBA);
end

function [dalphapg,dalphag] = bfct_dfag(P,~)
% BFCT_DFAG Derivatives of conversion factors from normalized to physical basis functions

dalphapg = zeros(numel(P.fPg),1);
dalphag  =  ones(numel(P.fPg),1);
end

function [dgdFxA,dgdFBA,dIgdFxA,dIgdFBA] = bfct_df(P, FxA, FBA)
% BFCT_DF Compute derivatives of physical basis functions using normalized basis functions

FQ = FxA(:)/FBA;
% Compute normalized basis functions
[ gN, IgN] = P.fN(FQ);
% Compute normalized basis functions derivatives
[dgN,dIgN] = P.dfN(FQ);
% Conversion to physical basis functions
[ alphapg, alphag] = P.fag(FBA);
% Derivatives of conversion facts
[dalphapg,dalphag] = P.dfag(FBA);

% Combine to form derivatives of physical basis functions */
 dgdFxA = alphapg.'./FBA .*  dgN;
dIgdFxA = alphag .'./FBA .* dIgN; % Note: this is equal to alphapg * gN or simply g
 dgdFBA = dalphapg.' .*  gN - FQ .*  dgdFxA;
dIgdFBA = dalphag .' .* IgN - FQ .* dIgdFxA;
end

function [dgAg,dIgAg] = bfct_dfA(P, FBA)
% BFCT_DFA Compute derivatives of physical basis functions at magnetic axis

% Use computation of derivatives of physical basis functions at FxA=0
[~,dgAg,~,dIgAg] = P.df(0,FBA);
end

