function LY = bdet(L,LX,varargin)
% BDET Circuit equation evolution simulator 
%
% LY = bdet(L,LX,varargin)
%
% Assuming known time histories of LX.Ia and LX.Iy, 
% and zero initial condition for LX.Iu, solve the vessel circuit equation to 
% get the Iu time evolution as well as the Va evolution to maintain the Ia.
%
% Inputs: 
% L: structure typically from FGE (or FBT with 'circuit'=true)
%   containing Maa, Muu and other mutuals.
% LX: Input data with a time sequence of t,Ia,Iy,Ip,rBt
%
% Outputs an LY object with fluxes and fields inferred from the currents,
% but without attempting basis function fits, so ag contains NaNs.
%
% Useful for simulating expected vessel evolution following FBT trajectory
% planning or otherwise vacuum shots with measured currents.
%
% Could later be merged with a new mode of operation of FGE enabling to
% solve the plasma equilibrium instead of assuming a known Iy.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Arguments
for k = 1:2:numel(varargin)
  switch varargin{k}
    % Additional output arguments
    case 'argout', L.argoutc = meqargoutc(varargin{k+1},L.argoutc);
    % Additional parameters
    otherwise, L.P.(varargin{k}) = varargin{k+1};
  end
end

%% Debug options
dotext = L.P.debug;
dopost = L.P.debugplot>0;
doplot = L.P.debugplot>1;

dt_mean = round(mean(diff(LX.t)),6); % round mean to us
equal_dt = all(abs(diff(LX.t)-dt_mean)<1e-8);
if equal_dt
  % precompute some matrices
  MM = (L.G.Muu + dt_mean*diag(L.G.Ru));
  Suu = MM\L.G.Muu;
  Sua = MM\L.G.Mau';
  Suy = MM\L.G.Mxu(L.lxy(:),:)';
end
May = L.G.Mxa(L.lxy(:),:)';
Muy = L.G.Mxu(L.lxy(:),:)';

for it=1:numel(LX.t)

  LXt = meqxk(LX,it);
  Ia = LXt.Ia;
  Iy = LXt.Iy;

  if it==1
    % assume starting from stationary condition
    dt  = 1; % dummy
    dIu = zeros(L.G.nu,1); 
    dIa = zeros(L.G.na,1);
    dIy = zeros(L.ny,1);
    % initial conditions
    Iu = zeros(L.G.nu,1);
  else
    dIa = LXt.Ia - zIa;
    dIy =     Iy - zIy;
    if equal_dt
      dt = dt_mean;
      Iu = Suu*zIu - Suy*dIy(:) - Sua*dIa;
    else
      % varying dt, must recompute matrices each time
      dt = LXt.t - zt;
      MM = (L.G.Muu + dt*diag(L.G.Ru));
      % circuit eq. for Iu
      Iu = MM\(L.G.Muu*zIu - L.G.Mau'*dIa - Muy*dIy(:));
    end
    dIu = Iu - zIu;
  end

  Va = L.G.Ra.*Ia + (L.G.Maa*dIa + L.G.Mau*dIu + May*dIy(:))/dt;
  
  Fx = meqFx(L,Iy,[Ia;Iu]);
  
  sIp = sign(LXt.Ip);
  
  [rA,zA,FA,dr2FA,dz2FA,drzFA,rX,zX,FX,dr2FX,dz2FX,drzFX,...
    rB,zB,FB,lB,lX,Opy,F0,F1,pdomstat,msg] = ...
    meqpdom(Fx,sIp,L.P.isaddl,L);
  
  assert(pdomstat,'pdom error %s',msg)
  
  LYt.Ia = Ia;
  LYt.Iu = Iu;
  LYt.Iy = Iy;
  LYt.Va = Va;
  LYt.t = LXt.t;

  ag = NaN(L.ng,1);
  LYt = meqpost(L,LYt,ag,...
    Fx,FA,FB,rA,zA,dr2FA,dz2FA,drzFA,rB,zB,lB,lX, ...
    rX,zX,FX,dr2FX,dz2FX,drzFX, ...
    LXt.rBt,Ia,Iu,Iy,Opy,F0,F1);

  LYt.Ip = LXt.Ip; % force this as integral value is unreliable without GS equilibrium
  LYt.shot = LXt.shot;

  % Additional output arguments
  for k = 1:numel(L.argoutc)
    if isempty(L.argoutc{k}.file)
      LYt.(L.argoutc{k}.fld) = eval(L.argoutc{k}.exp);
    end
  end
 
  if it==1
    LYt = meqLYdot(LYt,LYt);
  else
    LYt = meqLYdot(LYt,LY(it-1));
  end
  LY(it) = LYt; %#ok<AGROW>
  
  zIa = Ia;
  zIu = Iu;
  zIy = Iy;
  zt  = LXt.t;
  
  % debugging displays
  debug(L,LYt,doplot,dotext)
end
LY = meqlpack(LY);

debug(L,LYt,dopost,false)

end

%% Debug
function debug(L,LYt,doplot,dotext)
if dotext
 fprintf('%s#%d %6.4fs Ip=%+.0fkA Iv=%+.0fkA\n',...
  L.P.tokamak,LYt.shot,LYt.t,LYt.Ip/1e3,(sum(L.G.Tvu,1)*LYt.Iu)/1e3);
end

if doplot
  ax = gca; cla;
  meqgplot(L.G,ax,'vl'); hold(ax,'on');
  contour(ax,L.rx,L.zx,LYt.Fx,31,'b'); 
  if L.P.ifield
    meqplotfield(L,LYt,'vacuum',L.P.ivacuum,'parent',ax)
  end
  drawnow;
end%
end
