function [dFs,dzs,drs,ddz2Fs,ddr2Fs,ddrzFs] = ...
  asxycsJac(Fx,zs,rs,dz2Fs,dr2Fs,drzFs,L,d)
% ASXYCSJAC Compute derivatives of asxycs outputs with respect to Fx
%
%   [dFs,dzs,drs,ddz2Fs,ddr2Fs,ddrzFs] = asxycsJac(Fx,zs,rs,dz2Fs,dr2Fs,drzFs,L,d)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if nargin < 8
  hasd0 = true;
  hasd1 = nargout>1;
  hasd2 = nargout>3;
else
  hasd0 = any(d>-1);
  hasd1 = any(d> 0);
  hasd2 = any(d> 1);
end

rk = L.taur;
zk = L.tauz;

ns = numel(rs);

irs = min(max(floor((rs-rk(1))*L.idrx)+1,4),L.nrx+3);
izs = min(max(floor((zs-zk(1))*L.idzx)+1,4),L.nzx+3);

drxs = (rs-rk(irs))*L.idrx;
dzxs = (zs-zk(izs))*L.idzx;

% Initialize derivatives of F
dFs = zeros(ns,L.nx);
drdFs = dFs;
dzdFs = dFs;
dz2dFs = dFs;
dr2dFs = dFs;
drzdFs = dFs;
% Initialize derivatives of position and hessian
drs = dFs;
dzs = dFs;
ddr2Fs = dFs;
ddrzFs = dFs;
ddz2Fs = dFs;

for is = 1:ns
  ir = irs(is);
  iz = izs(is);
  dr = drxs(is);
  dz = dzxs(is);
  
  % R-interpolation
  A = csint_helper(dr,0);
  Ar = A*L.Mr(ir-3:ir,:);

  % Z-interpolation
  B = csint_helper(dz,0);
  Bz = B*L.Mz(iz-3:iz,:);
  
  if hasd0
    dFs(is,:) = reshape(Bz.'*Ar,1,L.nx);
  end
  
  if hasd1
    % R-interpolation
    dA = csint_helper(dr,1)*L.idrx;
    dAr = dA*L.Mr(ir-3:ir,:);

    % Z-interpolation
    dB = csint_helper(dz,1)*L.idzx;
    dBz = dB*L.Mz(iz-3:iz,:);

    drdFs(is,:) = reshape( Bz.'*dAr,1,L.nx);
    dzdFs(is,:) = reshape(dBz.'* Ar,1,L.nx);
  end
  
  if hasd2
    % R-interpolation
    d2A = csint_helper(dr,2)*(L.idrx*L.idrx);
    d2Ar = d2A*L.Mr(ir-3:ir,:);

    % Z-interpolation
    d2B = csint_helper(dz,2)*(L.idzx*L.idzx);
    d2Bz = d2B*L.Mz(iz-3:iz,:);
    
    dr2dFs(is,:) = reshape(  Bz.'*d2Ar,1,L.nx);
    drzdFs(is,:) = reshape( dBz.'* dAr,1,L.nx);
    dz2dFs(is,:) = reshape(d2Bz.'*  Ar,1,L.nx);
  end
end

if hasd1
  % [drs,dzs] = - Hs\ddF(rs,zs)
  detHs = dr2Fs.*dz2Fs - drzFs.^2;
  imdetHs = zeros(ns,1);
  imdetHs(detHs~=0) = -1./detHs(detHs~=0);
  drs = imdetHs.*( dz2Fs.*drdFs - drzFs.*dzdFs);
  dzs = imdetHs.*(-drzFs.*drdFs + dr2Fs.*dzdFs);
end

if hasd2
  % [ddr2Fs,ddz2Fs,ddrzFs] = [dr2dFs,dz2dFs,drzdFs] - d3Fs*[drs;dzs]
  % Unperturbed coefficients (to compute 3rd order derivatives)
  c = L.Mr*(L.Mz*Fx).'; % [nrspl,nzspl]
  [~,~,~,~,~,~,d3rrrFs,d3rrzFs,d3rzzFs,d3zzzFs] = cs2devalmex(rk,zk,c,rs,zs,3);
  ddr2Fs = dr2dFs + d3rrrFs.*drs + d3rrzFs.*dzs;
  ddrzFs = drzdFs + d3rrzFs.*drs + d3rzzFs.*dzs;
  ddz2Fs = dz2dFs + d3rzzFs.*drs + d3zzzFs.*dzs;
end

end
