function [dFSdFx, dzSdFx, drSdFx, ddz2FSdFx, ddr2FSdFx, ddrzFSdFx] = asxyJac(f, dx, dy, idx, idy, ixS, d)
% ASXYJAC Computes critical point position, flux and hessian jacobians
%
%   [dzAdFx, drAdFx, dFAdFx, ddz2FAdFx, ddr2FAdFx, ddrzFAdFx] = asxyJac(f, dx, dy, idx, idy, ind, d)
%
% Arguments:
%   f:    Matrix, Flux field (references as Fx in most of meq)
%   dx:   float, z spacing in x grid
%   dy:   float, r spacing in x grid
%   idx:  float, inverse of z spacing in x grid
%   idy:  float, inverse of r spacing in x grid
%   ixS:  int, flattened index of cell containing critical point in x grid
%   d:    int, list of derivatives to produce (0 for flux, 1 for position,
%         2 for hessian)
% returns:
%   dFSdFx:     Matrix(nS,nX), Sparse matrix containing the derivatives of
%               the flux value at the critical point with respect to the
%               flux field.
%   dzSdFx:     Matrix(nS,nX), Sparse matrix containing the derivatives of zS
%   drSdFx:     Matrix(nS,nX), Sparse matrix containing the derivatives of rS
%   ddz2FSdFx:  Matrix(nS,nX), Sparse matrix containing the derivatives of dz2FS
%   ddr2FSdFx:  Matrix(nS,nX), Sparse matrix containing the derivatives of dr2FS 
%   ddrzFSdFx:  Matrix(nS,nX), Sparse matrix containing the derivatives of drzFS
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

if nargin < 7
  hasd0 = true;
  hasd1 = nargout>1;
  hasd2 = nargout>3;
else
  hasd0 = any(d>-1);
  hasd1 = any(d> 0);
  hasd2 = any(d> 1);
end

nS = numel(ixS);

% grid cell scaling quantities
qidx=idx*idx; qidy=idy*idy; idxy=idx*idy;

% get neighbouring values
nx  = numel(f);
nzx = size(f,1);
i0 = ixS   ;  d0 = f(i0);      % 3 4 5
i1 = i0-  1;  d1 = f(i1) - d0; % 2 0 6
i2 = i0-nzx;  d2 = f(i2) - d0; %   1
i3 = i2+  1;  d3 = f(i3) - d0;
i4 = i0+  1;  d4 = f(i4) - d0;
i6 = i0+nzx;  d6 = f(i6) - d0;

% Hessian estimation (see asxymex)
a = (d4 - d1) *     0.5;
b = (d6 - d2) *     0.5;
c =  d1 + d4           ;
d =  d2 + d6           ;
e = (d4 + d2 - d3)     ;
h = c.*d - e.*e;
mask = (h ~= 0);
h(mask) = 1./h(mask);
xe = (e.*b - a.*d) .* h; xmask = abs(xe)>1; xe(xmask) = sign(xe(xmask));
ye = (e.*a - b.*c) .* h; ymask = abs(ye)>1; ye(ymask) = sign(ye(ymask));

% compute gradient of the extremum/saddle
% dependence on xe, ye cancels as it is a zero gradient point
% the 6 point interpolant is: f0 + a x + b y + c/2 x^2 + d/2 y^2 + e xy

% Derivatives of f0,a,b,c,d,e (rows) coefficients with respect to Fx(i0,...i4,i6) values (columns)
dabcdf = [ 1.0,  0.0,  0.0,  0.0,  0.0,  0.0;
           0.0, -0.5,  0.0,  0.0,  0.5,  0.0;
           0.0,  0.0, -0.5,  0.0,  0.0,  0.5;
          -2.0,  1.0,  0.0,  0.0,  1.0,  0.0;
          -2.0,  0.0,  1.0,  0.0,  0.0,  1.0;
          -1.0,  0.0,  1.0, -1.0,  1.0,  0.0];

if hasd0
  % Flux derivatives
  dFSdFx = [ones(nS,1), xe, ye, 0.5*xe.*xe, 0.5*ye.*ye, xe.*ye] * dabcdf;

  % Convert to sparse matrices
  nstencil = 6;
  jj = double([i0,i1,i2,i3,i4,i6]);
  iS = repmat((1:nS).',1,nstencil);
  dFSdFx = sparse(iS,jj,dFSdFx   ,nS,nx);
else
  dFSdFx = sparse([],[],[],nS,nx);
end

if hasd1
  % r,z position estimation has a bit more complicated derivatives
  drSdFx = ([zeros(nS,1), e, -c, -(b+d.*ye), -c.*ye, (a+2.0*e.*ye)] .* h .* dy) * dabcdf;
  drSdFx(xmask,:) = 0;
  dzSdFx = ([zeros(nS,1), -d, e, -d.*xe, -(a+c.*xe), (b+2.0*e.*xe)] .* h .* dx) * dabcdf;
  dzSdFx(ymask,:) = 0;

  % Convert to sparse matrices
  dzSdFx    = sparse(iS,jj,dzSdFx   ,nS,nx);
  drSdFx    = sparse(iS,jj,drSdFx   ,nS,nx);
else
  drSdFx = sparse([],[],[],nS,nx);
  dzSdFx = sparse([],[],[],nS,nx);
end

if hasd2
  % hessian derivatives just depend linearly on estimation flux values
  ddz2FSdFx = repmat(dabcdf(4, :) * qidx,nS,1);
  ddr2FSdFx = repmat(dabcdf(5, :) * qidy,nS,1);
  ddrzFSdFx = repmat(dabcdf(6, :) * idxy,nS,1);

  % Convert to sparse matrices
  ddz2FSdFx = sparse(iS,jj,ddz2FSdFx,nS,nx);
  ddr2FSdFx = sparse(iS,jj,ddr2FSdFx,nS,nx);
  ddrzFSdFx = sparse(iS,jj,ddrzFSdFx,nS,nx);
else
  ddz2FSdFx = sparse([],[],[],nS,nx);
  ddr2FSdFx = sparse([],[],[],nS,nx);
  ddrzFSdFx = sparse([],[],[],nS,nx);
end

end
