function SCplot(SC,L,LY,varargin)
% SCplot(SC,[L=[]],[LY=[]],[color='k'],[verbose=false],[parent=gca])
% 
% Plotting for shape control struct
%
% Inputs:
% SC: shape control structure
% L: L structure
% LY: LY structure for LCFS (to plot the boundary)
% color: set the color for the plot
% verbose: if true, it displays the point labels
% parent: use parent axes instead of current axes (gca)
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.


%% Parse inputs
if nargin<2, L  = []; end
if nargin<3, LY = []; end
assert(isstruct(L) ||isempty(L),  'Unrecognized L argument.');
assert(isstruct(LY)||isempty(LY), 'Unrecognized LY argument.');

% optional parameters
params = struct('color','k','verbose',false,'parent',[]);
if nargin>3
  for k=1:2:numel(varargin)
    assert(ischar(varargin{k}), 'Optional arguments must come in name-value pairs.');
    assert(isfield(params,varargin{k}), sprintf('Unknown parameter %s.', varargin{k}));
    params.(varargin{k}) = varargin{k+1};
  end
end

if isempty(params.parent)
  params.parent = gca;
end

% extract for readability
parent  = params.parent;
color   = params.color; 
verbose = params.verbose;

% sanity check
assert(numel(SC.t)==1,'supported for 1 timeslice only, use meqxk() to slice')

%% Plots

% geometry
if ~isempty(L)
  meqgplot(L.G,parent,'l');
end

% separatrix
if ~isempty(L) && ~isempty(LY)
  contour(parent,L.rx,L.zx,LY.Fx-LY.FB,[0 0],'-','color',color,'linewidth',2);
  if any(SC.lcX2) && LY.nX>1
    for i = 2:LY.nX
      contour(parent,L.rx,L.zx,LY.Fx-LY.FX(i),[0 0],'--','color',color,'linewidth',2);
    end
  end
end

% points stored in SC, divided by type
hold on, axis equal
hC  = plot(parent,SC.rc(SC.lcC) ,SC.zc(SC.lcC) ,'o','linewidth',2,'color',color,'displayname','Contour points');
hL  = plot(parent,SC.rc(SC.lcL) ,SC.zc(SC.lcL) ,'*','linewidth',2,'color',color,'displayname','Limiter points');
hS  = plot(parent,SC.rc(SC.lcS) ,SC.zc(SC.lcS) ,'*','linewidth',2,'color',color,'displayname','Strike points');
hX1 = plot(parent,SC.rc(SC.lcX1),SC.zc(SC.lcX1),'x','linewidth',2,'color',color,'displayname','Primary X-points');
hX2 = plot(parent,SC.rc(SC.lcX2),SC.zc(SC.lcX2),'x','linewidth',2,'color',color,'displayname','Secondary X-points');

% drsep
if isfield(SC,'drsep') && isfield(SC,'dzsep')
  plot(parent,SC.rc(SC.lcD),SC.zc(SC.lcD),'o','linewidth',2,'color',color);
  hD  = plot(parent,SC.rc(SC.lcD)+SC.drsep,SC.zc(SC.lcD)+SC.dzsep,'+','color',color,'displayname','Separatrix points');
else
  hD  = plot(parent,SC.rc(SC.lcD),SC.zc(SC.lcD),'+','linewidth',2,'color',color,'displayname','Separatrix points');
end

% point displaced after applying shape control references
if isfield(SC,'drcdes') && isfield(SC,'dzcdes') && isfield(SC,'ref')
  des = SC.Mesys*SC.ref;
  rrc = SC.rc+SC.drcdes*des;
  zzc = SC.zc+SC.dzcdes*des;
  hD  = plot(parent,rrc(~SC.lcI),zzc(~SC.lcI),'.','markersize',20,'color',color,'displayname','Projected points');
end

% centroid
if isfield(SC,'rp') && isfield(SC,'zp')
  plot(parent,SC.rp,SC.zp,'+','linewidth',2,'color',color);
end

% legend and labels
if verbose
  for ic=1:SC.nc
    if ~SC.lcI(ic); text(parent,SC.rc(ic)+0.02,SC.zc(ic),SC.dimc{ic}); end
  end
  legend([hC;hL;hS;hX1;hX2;hD],'location','northeastoutside');
end

% figure title
if ~isempty(L)  
  title(sprintf('%s shot:%d t:%0.3f',L.P.tokamak,L.P.shot,SC.t));
else
  title(sprintf('t:%0.3f',SC.t));
end

end
