function SC = LY2SC(L,LY,varargin)
% SC = LY2SC(L,LY,[nc=20],[ncC=9])
% 
% Generate and classify shape control points from an equilibrium
%
% Inputs:
% L,LY: equilibrium structs
% nc: desired number of shape control points
% ncC: desired number of contour shape control points
%
% Outputs:
% SC with fields:
%   - t: time array
%   - nc: number of control points
%   - rc,zc: coordinates of control points
%   - dimc: labels for control points
%   - Control point are classified into:
%     + Contour points     (lcC  = 1)
%     + Limiter points     (lcL  = 1)
%     + Strike points      (lcS  = 1)
%     + Primary X points   (lcX1 = 1)
%     + Secondary X points (lcX2 = 1)
%     + Separatrix points  (lcD  = 1)
%     + Ignored points     (lcI  = 1)
% 
% NOTE: the fields in SC are padded with zeros up to nc.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Parse inputs
p = inputParser;

% Validation functions
checkL   = @(L)  isstruct(L);
checkLY  = @(LY) isstruct(LY);
checknc  = @(nc) isnumeric(nc);
checkncC = @(ncC) isnumeric(ncC);

% Add inputs
addRequired(p,'L' ,checkL);
addRequired(p,'LY',checkLY);
addOptional(p,'nc',20,checknc);
addOptional(p,'ncC',9,checkncC);

% Get parser results
parse(p,L,LY,varargin{:});
nc  = p.Results.nc;
ncC = p.Results.ncC;

% Make sure that ncC default value is consistent
if nc<ncC
  warning('nc and ncC are not consistent, overwrinting ncC...')
  ncC = nc-1; % at least one limiter or X-point
end

%% 
nt = numel(LY.t); % Number of equilibria
SC.t = LY.t; % Store time array
SC.nc = nc; % Number of control points

% Initialize SC structure fields
[SC.rc,SC.zc] = deal(zeros(SC.nc,nt));
SC.dimc = cell(SC.nc,nt);
[SC.lcC,SC.lcL,SC.lcS,SC.lcX1,SC.lcX2,SC.lcD] = deal(false(SC.nc,nt));

% Iterate over time slices
for it = 1:nt
  % Init fields
  k = 0;
  [gpr,gpz] = deal(zeros(SC.nc,1));
  [lcC,lcS,lcL,lcX1,lcX2,lcD] = deal(false(SC.nc,1));
  % Get current time slice of LY
  LYt = meqxk(LY,it);
  
  %% Main contour points
  [rc,zc] = contour_points(L,LYt,ncC+1,LYt.FB,false);
  ic = k+(1:ncC+1); k = k+ncC+1;
  gpr(ic) = rc; gpz(ic) = zc;
  lcC(ic(2:end)) = true;
  % Divertor or limiter plasma
  if LYt.lX
    lcX1(1) = true; % primary X-points
    ncX1 = 1; ncL = 0;
  else
    lcL(1) = true;  % limiter point
    ncX1 = 0; ncL = 1;
  end
  
  %% Other primary X-points  
  if k>=nc
    warning('maximum number of points reached, skipping additional primary X-points...')
  else  
    % Check other X-points...
    iXs = ~isnan(LYt.rX) & LYt.rX~=LYt.rB & LYt.zX~=LYt.zB;
    % ...on the LCFS
    iX1 = abs(LYt.FX-LYt.FB)/abs(LYt.FB-LYt.FA) <= 1e-4;
    rX1 = LYt.rX(iXs & iX1); zX1 = LYt.zX(iXs & iX1);
    ncX1 = numel(rX1); ic = k+(1:ncX1); k = k+ncX1;
    gpr(ic) = rX1; gpz(ic) = zX1;
    lcX1(ic) = true;  
  end
  
  %% Secondary X-points and dsep
  if k>=nc
    warning('maximum number of points reached, skipping secondary X-points...')
    ncX2 = 0;
  else 
    % ...close to the the LCFS
    rX2 = LYt.rX(iXs & ~iX1); zX2 = LYt.zX(iXs & ~iX1);
    ncX2 = numel(rX2); ic = k+(1:ncX2); k = k+ncX2;
    gpr(ic) = rX2; gpz(ic) = zX2;
    lcX2(ic) = true;
    
    % Dsep points
    if any(lcX2)
      icC = find(lcC);
      [~,ii] = max(gpr(icC));
      lcD(icC(ii)) = true;
    end
  end
  
  %% Strike points
  if k>=nc
    warning('maximum number of points reached, skipping strike points...')
    ncS = 0;
  else 
    % strike points (on the separatrix)
    if LYt.lX
      [rs,zs] = contour_points(L,LYt,nc,LYt.FB,false,true);
      if ~any(gpz(lcX1)>=LYt.zA)
        irm = zs>=LYt.zA;
        rs(irm) = []; zs(irm) = [];
      end
      if ~any(gpz(lcX1)<=LYt.zA)
        irm = zs<=LYt.zA;
        rs(irm) = []; zs(irm) = [];
      end
      ncS = numel(rs); ic = k+(1:ncS); k = k+ncS;
      gpr(ic) = rs; gpz(ic) = zs;
      lcS(ic) = true;
    else
      ncS = 0;
    end
  end
  
  %% Post-processing
  % Ignored points
  lcI = ~(lcC|lcL|lcS|lcX1|lcX2|lcD);
  % Store data
  SC.rc(  :,it) = gpr;
  SC.zc(  :,it) = gpz;
  SC.lcC( :,it) = lcC;
  SC.lcL( :,it) = lcL;
  SC.lcS( :,it) = lcS;
  SC.lcX1(:,it) = lcX1;
  SC.lcX2(:,it) = lcX2;
  SC.lcD( :,it) = lcD;
  SC.lcI( :,it) = lcI;
  % Label points
  ii = SC.lcC( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Surf%02d'));
  ii = SC.lcL( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Limi%02d'));
  ii = SC.lcS( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Stri%02d'));
  ii = SC.lcX1(:,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Xpri%02d'));
  ii = SC.lcX2(:,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Xsec%02d'));
  ii = SC.lcD( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Dsep%02d'));
  ii = SC.lcI( :,it); SC.dimc(ii,it) = cellstr(num2str((1:sum(ii))','Ignr%02d'));  
  
  % Final sanity check on number of points
  assert(nc>=ncC+ncL+ncX1+ncX2+ncS)
end

end

function [ri,zi,ni] = contour_points(L,LY,npts,FF,doplot,opencontours)
% [ri,zi,ni] = contour_points(L,LY,npts,FF,doplot,opencontours)
% returns r,z coordinates and contour index
% for npts points equally spaced along a contour of flux map LY.Fx
% on flux level surface FF.
% doplot optionally plots the result.
% opencontour= false searches the points on a closed contour:
% in this case, first point is boundary-defining point given by [LY.rB,LY.zB]
% opencontour= true searches for intersections between the open field lines and the vessel.
% whose cross section is contained in [L.G.rl L.G.zl]

if nargin<6, opencontours=false; end
if nargin<5, doplot=false; end
if nargin<4, FF = LY.FB(~isnan(LY.FB)); end % default, LCFS

assert(numel(LY.t)==1,'one LY time slice only, use meqxk to slice')

[ri,zi,ni] = deal([]);
FF=unique(FF); nfound=0;
for iF=1:numel(FF)
  
  F = FF(iF);
  attempt = 1;
  attmax = 100;  
  isclosed = false;
  while ~any(isclosed) % try attmax times to find a CFS, otherwise error
    C = contourc(L.rx,L.zx,LY.Fx,F*[1 1]);
    if doplot && ~opencontours
      contour(L.rx,L.zx,LY.Fx,F*[1 1]);
    end
    % scan it
    ii=1; n=0;
    
    while ii<=size(C,2)% for each (isoflux) contour line obtained with contourc
      ncpts = C(2,ii); % total # of pairs (R,z) per contour line (see contourc help)
      n=n+1;           % active # of contour line
      if doplot , fprintf('npts=%d\n',ncpts); end
      rrp{n} = C(1,ii+(1:ncpts)); %#ok<AGROW>
      zzp{n} = C(2,ii+(1:ncpts)); %#ok<AGROW>
      isclosed(n) = (rrp{n}(1)==rrp{n}(end) && zzp{n}(1)==zzp{n}(end));
      ii = ii+ncpts+1;
    end
    
    assert(attempt < attmax,'could not find a closed flux surface after %2.0d attempts', attempt);
    
    attempt = attempt+1;
    F = F + 0.001*(LY.FA-LY.FB); % change the flux in the right direction to find CFS
  end
  
  if ~opencontours
    % keep only closed domains
    isel =  isclosed;
    openstr = 'closed';
  else
    % keep only open domains
    isel = ~isclosed;
    openstr = 'open';
  end
  rrp = rrp(isel);
  zzp = zzp(isel);
  nnp = sum(isel);

  
  if doplot, fprintf('Found %2.0d %s domain(s) for F=%2.3f \n',nnp,openstr,F); end
  if nnp == 0
    [ri,zi,ni] = deal([]);
    return
  end

  if ~opencontours
    [rpt,zpt,nnn] = deal(cell(npts,nnp));
    for ii=1:nnp % for each found contour
      % arc length
      r = rrp{ii}; z = zzp{ii};
      
      if any(LY.lX) % diverted equilibrium
        % note: LY.FX not necessarily equal to F
        iX = (LY.FX==LY.FB);
        assert(sum(iX)==1,'only 1 X point on contour supported')
      end
      % add either x- or limiter point in the right place
      r0 = LY.rB;
      z0 = LY.zB;
      % for all adjacent points, put Bounding point (limiter or x-point)
      % in between them, use inner product to determine whether they are in
      % a row
      col=zeros(numel(r)-1,1);
      for icand = 1:(numel(r)-1)
        v1 = [r0-r(icand);z0-z(icand)];
        v2 = [r(icand+1)-r0;z(icand+1)-z0];
        % colinearity indicator: 1 means they are 'in a row'
        col(icand) = (v1'*v2)/(norm(v1)*norm(v2));
      end
      [~,imax]=max(col);
      r = [r0,r(imax+1:end),r(1:imax)];
      z = [z0,z(imax+1:end),z(1:imax)];
      
      % arc length
      s = [0,cumsum(sqrt(diff(z).^2+diff(r).^2))];
      spts = linspace(0,s(end),npts+1); spts=spts(1:end-1);
      for ipt = 1:npts
        jpt = find(s<=spts(ipt),1,'last');
        lam = (spts(ipt)-s(jpt))/(s(jpt+1)-s(jpt));
        rpt{ipt,ii} = lam*(r(jpt+1)-r(jpt))+r(jpt);
        zpt{ipt,ii} = lam*(z(jpt+1)-z(jpt))+z(jpt);
        nnn{ipt,ii} = nfound + ii;
      end
      if doplot
        hold on; plot([rpt{:,ii}],[zpt{:,ii}],'o'); axis equal;
      end
    end
    
    ri = [ri,rpt{:}]; %#ok<AGROW>
    zi = [zi,zpt{:}]; %#ok<AGROW>
    ni = [ni,nnn{:}]; %#ok<AGROW>
    
    nfound = ni(end);
  else % opencontour
    for ii=1:nnp
      r = rrp{ii}; z = zzp{ii};
      % for all curves, find intersection with limiter
      % seek intersection from either end
      for idir = [1 2]
        switch idir
          case 1, ipgrid = 1:numel(r);    % one direction
          case 2, ipgrid = numel(r):-1:1; % other direction
        end
        for ip = ipgrid
          isinside = inpolygon(r(ip),z(ip),L.G.rl,L.G.zl);
          if ip==1
            assert(~isinside,'first point is not outside limiter! Grid may not extend sufficiently outside limiter')
          else
            if isinside
              [rs,zs] = interpolate_strike_point(r(ip),z(ip),r(ip-1),z(ip-1),L);
              ri = [ri,rs]; zi = [zi,zs]; %#ok<AGROW> % add point to list
              if doplot
                hold on;
                plot(rs,zs,'x');
                plot(r(ip-1:ip+1),z(ip-1:ip+1),'.-')
              end
              break;
            end
          end
        end
      end
    end
    
  end
  
end

%% output
ri = ri';
zi = zi';
ni = ni';

end

function [rs,zs] =  interpolate_strike_point(rin,zin,rout,zout,L)
% interpolate the strike point at the limiter (whose geometry is contained
% in L.G.rl,L.G.zl) in the assumption that (rin,zin) is the first point
% within the limiter and the previous one is outside along an open flux surface

% angular coefficients of the line passing for (rin,zin) and (rout,zout)
mp = (zin-zout)/(rin-rout);

% angular coefficient of the line passing for the two points of the limiter
% closest to (rin,zin)
dist = abs((rin-L.G.rl).^2 + (zin-L.G.zl).^2);
[~,indc] = sort(dist);
rv1 = L.G.rl(indc(1));
zv1 = L.G.zl(indc(1));
rv2 = L.G.rl(indc(2));
zv2 = L.G.zl(indc(2));
mv = (zv1-zv2)/(rv1-rv2);

%solve for the solution of two lines passing by
if rv1~=rv2
  A = [-mp 1 ; -mv 1]; b = [-mp*rout+zout ; -mv*rv1+zv1];
  x = A\b;
else % vertical vessel
  A = [-mp 1 ; 1 0]; b = [-mp*rout+zout ; rv1];
  x = A\b;
end

rs = x(1);
zs = x(2);

end
