function G = wall2G(G,wall,P)
%WALL2G Complement G structure with information from wall IDS
%
% wall: wall IDS
% P: (struct) needs to contain the parameters of patch_inner_outer.m and outline2innerouterlines.m. See documentation.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Vessel
% Assumption: only 1 description of the vessel given.
[rv,zv,hv,wv,Rv] = deal([]);

if isfield(wall.description_2d{1},'vessel') %Vessel can be eventually missing or provided in pf_passive
  switch wall.description_2d{1}.vessel.type.index
    case 0 % annular
      % Loop over units
      for ii=1:numel(wall.description_2d{1}.vessel.unit)
        [ri,zi,ro,zo] = deal([]);
        
        % inner outline coordinates
        ri = wall.description_2d{1}.vessel.unit{ii}.annular.outline_inner.r;
        zi = wall.description_2d{1}.vessel.unit{ii}.annular.outline_inner.z;
        
        % outer outline coordinates
        ro = wall.description_2d{1}.vessel.unit{ii}.annular.outline_outer.r;
        zo = wall.description_2d{1}.vessel.unit{ii}.annular.outline_outer.z;
        
        resistivity = wall.description_2d{1}.vessel.unit{ii}.annular.resistivity;
        
        % Create filaments from inner and outer line.
        [rv_tmp,zv_tmp, hv_tmp, wv_tmp, Rv_tmp] = patch_inner_outer(ri,zi,ro,zo,P.rayr0,P.rayz0,resistivity,P);
        rv = [rv; rv_tmp]; zv = [zv; zv_tmp];
        hv = [hv; hv_tmp]; wv = [wv; wv_tmp];
        Rv = [Rv; Rv_tmp];
      end
    case 1 % outline
      % Assumption: each vessel unit is composed by a single closed region
      for ii=1:numel(wall.description_2d{1}.vessel.unit) % Loop over units
        [ri,zi,ro,zo] = deal([]);
        for jj=1:numel(wall.description_2d{1}.vessel.unit{ii}.element) % Loop over elements
          % Get component outline
          rl = wall.description_2d{1}.vessel.unit{ii}.element{jj}.outline.r;
          zl = wall.description_2d{1}.vessel.unit{ii}.element{jj}.outline.z;
          resistivity = wall.description_2d{1}.vessel.unit{ii}.element{jj}.resistivity;
          
          % Compute the outer and inner lines from outline of the element
          [ri_tmp,zi_tmp,ro_tmp,zo_tmp] = outline2innerouterlines(rl,zl,P.rayr0,P.rayz0,P);
          ri = [ri;ri_tmp]; zi = [zi;zi_tmp];
          ro = [ro;ro_tmp]; zo = [zo;zo_tmp];
        end
        
        % Create filaments from inner and outer line.
        [rv_tmp,zv_tmp, hv_tmp, wv_tmp, Rv_tmp] = patch_inner_outer(ri,zi,ro,zo,P.rayr0,P.rayz0,resistivity,P);
        rv = [rv; rv_tmp]; zv = [zv; zv_tmp];
        hv = [hv; hv_tmp]; wv = [wv; wv_tmp];
        Rv = [Rv; Rv_tmp];
      end
    case 2 % rectangle
      error('geometry_type rectangle not yet implemented')
    case 3 % oblique
      error('geometry_type oblique not yet implemented')
    case 4 % arcs circle
      error('geometry_type outline not implemented')
  end

  G.rv = rv(:); G.zv = zv(:); G.nv = numel(rv);
  G.wv = wv(:); G.hv = hv(:);
  G.Rv = Rv(:); 
  % TODO Subdivision into segments could be used to separate components. Not yet implemented.
  G.Tivs = ones(G.nv); G.Rs = G.Rv(:);
end

%% Tile apperture (limiter)
% Assumption: only 1 description is given. Pack together all the units if multiple ones available.
rl = []; zl = [];
if isfield(wall.description_2d{1},'limiter')
  for ii=1:numel(wall.description_2d{1}.limiter.unit)
    rl = [rl; wall.description_2d{1}.limiter.unit{ii}.outline.r];
    zl = [zl; wall.description_2d{1}.limiter.unit{ii}.outline.z];
  end
end
G.rl = rl; G.zl = zl;

if P.debugplot > 0
  figure
  meqgplot(G, gca, 'l')
end

end