function [rv,zv,hv,wv,Rv] =  patch_inner_outer(ri,zi,ro,zo,rr0,zz0,resistivity, P)
%PATCH_INNER_OUTER  Filamentary discretisation from inner line/ outer line of closed geometry
%
% The algorithm is suited to discretized into filaments thin shell like geometries.
%
% ri,zi: arrays with r,z coordinates of inner line of the geometry
% ro,zo: arrays with r,z coordinates of outer line of the geometry
% rr0,zz0: r,z coordinate of reference systeme center to compute poloidal angle of ri,zi,r0,z0
% resistivity: material resitivity [Ohm m]. The resistance of a filament is 2*pi*r_center*resistivity/area
% P.raydeltadist (optional): [m] curvilinear length of segments for resampling of inner and outer line.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Get parameters and overload defaults
D.debug = 0;
D.debugplot = 0;
D.raydeltadist = 0.03; % [m]
for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
end

% Center inner line wrt rr0,zz0.
% Ordered in theta from 0 to 2*pi counterclockwise
r_centered = ri - rr0;
z_centered = zi - zz0;
theta_segments = atan2(z_centered, r_centered);
[~, indices] = sort(theta_segments);
ri = ri(indices); zi = zi(indices);
ro = ro(indices); zo = zo(indices);

% Compute the cumulative inner line outer line length starting from min(theta)
dist_i = sqrt(diff(ri).^2 + diff(zi).^2);
length_i = [0;cumsum(dist_i)];
dist_o = sqrt(diff(ro).^2 + diff(zo).^2);
length_o = [0;cumsum(dist_o)];

% Original discretisation of ri,zi,ro,zo could be coarse and only focused on corners.
% Re-interpolating allows for a more uniform representation.
% Re-interpolating in theta, with fixed segment length, allows to obtain a discretisation
% which, after passing a coorner in the geometry, keeps the patch similar to rectangles.
npoints = ceil(max(length_i)/P.raydeltadist);
lq_i = linspace(0,max(length_i),npoints);
lq_o = linspace(0,max(length_o),npoints);
ri = interp1(length_i, ri, lq_i); zi = interp1(length_i, zi, lq_i);
ro = interp1(length_o, ro, lq_o); zo = interp1(length_o, zo, lq_o);

% Instantiate figure for debugging purposes
if P.debugplot >2
  figure
  hold on
  axis equal
end

% From the re-sampled inner and outer lines, extract patches of the closed geometry.
% Create a filament in the center of the trapezoidal patch.
[rv,zv,hv,wv,Rv] = deal([]);
for ii=1:numel(ri)-1
  r = [ri(ii),ro(ii),ro(ii+1),ri(ii+1)];
  z = [zi(ii),zo(ii),zo(ii+1),zi(ii+1)];
  
  rcenter = (max(r) +  min(r))/2;
  zcenter = (max(z) +  min(z))/2;
  
  rv = [rv; rcenter]; zv = [zv; zcenter];
  av = comp_area(r,z); hv = [hv; sqrt(av)];
  wv = [wv; sqrt(av)];
  Rv = [Rv; 2*pi*rcenter*resistivity/av];
  
  % Plot patches and discretization.
  if P.debugplot > 2
    patch(r,z,'b')
    plot([ri(ii),ri(ii+1)],[zi(ii),zi(ii+1)],'r.')
    plot([ro(ii),ro(ii+1)],[zo(ii),zo(ii+1)],'g.')
    plot(rcenter,zcenter,'y*')
  end
end
end

% Compute area of the trapezoidal patch
function area = comp_area(r,z)
n = numel(r); % Number of vertices
area = 0;
i = 1:n;
j = mod(i, n) + 1;
area = sum(r(i) .* z(j) - r(j) .* z(i));
area = abs(area) / 2;
end
