function [ri,zi,ro,zo, theta_out] = outline2innerouterlines(r,z,rr0,zz0, P)
% OUTLINE2INNEROUTERLINES Extract inner/outer line wrt rr0,zz0 location for closed contour provided in r,z
%
% The routine performs ray tracing from rr0,zz0 center to r,z array. The first intersection belongs to inner line the second to outer line.
%
% Inputs:
% r,z: (1darray) Coordinates of the closed contour describing the original geometry as in `outline` type in ids
% P.rayntheta: (optional) N of rays used to find inner and outer lines
% P.raythrcut: (optional) (float>0.) [m].
% - Some geometries can have sharp angles, often resulting from cutting a thin shell, like the vessel, into several blocks, at discrete poloidal angles.
%   When extracting the inner and outer lines for a given geometry, a ray intersects the countour 2 times, 1 for the inner
%   1 for the outer point. When distance of the corresponding points on the inner and outer contour is < P.raythrcut,
%   these points are not considered in the resulting extracted outer and inner line.
%
% Outputs:
% ri,zi: (1darrays) Coordinates of inner line wrt to rr0,zz0, extracted from r,z
% ro,zo: (1darrays) Coordinates of outer line wrt to rr0,zz0, extracted from r,z
% theta_out: (1darrays) Theta angles between (ri,zi)/(ro,zo) and rr0,zz0
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Set defaults and overload parameters
D.debug = 0;
D.debugplot = 0;
D.rayntheta = 100;
D.raythrcut = 0;
for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
end

% Enforce last point of closed surface to be equal to the first.
if (r(end) - r(1))^2 + (z(end) - z(1))^2 > 1e-3 % if distance > 1mm
  r(end+1) = r(1);
  z(end+1) = z(1);
end

% Compute theta coordinates of the contour.
% Assumption: r,z needs to be ordered clock/counter clockwise contour
r_centered = r - rr0; z_centered = z - zz0;
theta_segments = mod( atan2(z_centered, r_centered), 2*pi);

theta_min = min(theta_segments);
theta_max = max(theta_segments);

% Array of theta angles for ray tracing.
thetaarr = linspace(theta_min, theta_max, P.rayntheta);
theta_out = [];

% Inner line coords
ri= []; zi = [];
% Outer line coords
ro = []; zo = [];
% Coordinates of the ray intersecting point
rint = []; zint = [];

for theta = thetaarr
  % Get all ray intersection point.
  % This could be > 2
  [rint_tmp, zint_tmp] = ray_trace(r,z,rr0,zz0,theta);
  rint = [rint;rint_tmp];
  zint = [zint;zint_tmp];
  
  % Compute distances of the ray intersection to the ray origin
  distances = sqrt((rint_tmp - rr0).^2 + (zint_tmp - zz0).^2);
  [~, indices] = sort(distances);
  
  % Check if there are hidden regions from the ray. This happens for gemotries with angles.
  if numel(rint_tmp) > 2
    warning('loop region hidden from the ray. Consider moving the ray origin.Keep only first 2 intersection')
  end
  
  if numel(rint_tmp) > 1
    % Take only the first 2 intersecting points, the ones with the inner and outer line.
    % Do not consider points hidden from the ray center.
    rint_tmp = rint_tmp(indices(1:2));
    zint_tmp = zint_tmp(indices(1:2));
    
    % Compute the distance between the intersection points
    dist_intersection = sqrt(diff(rint_tmp).^2 + diff(zint_tmp).^2);
    
    % Remove points apart for a distance below a certain
    % threshold. This happens at the edge where the element can look
    % like a triangle
    if dist_intersection > P.raythrcut
      ri = [ri;rint_tmp(1)]; ro = [ro;rint_tmp(2)];
      zi = [zi;zint_tmp(1)]; zo = [zo;zint_tmp(2)];
      theta_out = [theta_out; theta];
    end
  end
end

%% Debugging plot
if P.debugplot >0
  figure
  hold on
  plot(r,z, 'DisplayName', 'outline original')
  plot(rr0,zz0, 'o', 'DisplayName', 'ray origin')
  plot(ri, zi, 'r.', 'DisplayName', 'inner line')
  plot(ro, zo, 'b.', 'DisplayName', 'outer line')
  plot(rint, zint, 'o', 'DisplayName', 'ray intersection')
  for theta = thetaarr
    length = max( sqrt((r - rr0).^2 + (z - zz0).^2) );
    lineLength = linspace(0,max(length),10);
    % Convert polar coordinates to Cartesian coordinates
    rr = lineLength * cos(theta) + rr0;
    zr = lineLength * sin(theta) + zz0;
    plot(rr, zr, 'g')
  end
  axis('equal')
end

end
