function  [rout, zout, aw, ww, hw] = gen_filament(vvdata, nturns, type_coil_description,varargin)
% GEN_FILAMENT generate filamentary discretisation of rectangle/oblique components.
%
% Inputs:
% type_coil_description: 2 -> rectangle description
% vvdata(1) = R coordinate of the center [m]
% vvdata(2) = Z coordinate of the center [m]
% vvdata(3) = dR total length. [m]
% vvdata(4) = dz total length. [m]
% nturns = number of turns. Could be fractional.
%
% type_coil_description: 3 -> oblique description
% 2----3
% |    |
% 1----4
% vvdata(1) = x coordinate of 1
% vvdata(2) = y coordinate of 1
% vvdata(3) = length 4-1
% vvdata(4) = length 2-1
% vvdata(5) = angle between 4-1 and x axis [rad]. Called alpha in the following. alpha increases counterclowise
% vvdata(6) = angle between 2-1 and y axis [rad]. Called beta in the following. beta increaase counterclockwise
%
% nturns: requested number of filaments. In case of windings, often taken from the number of turns.
%
% Outputs:
% rout,zout: (1darray) filament discretisation
% aw, ww, hw: see meqg documentation.
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

%% Get parameters
P = struct;
P.debug = 0;
P.debugplot = 0;
P.discrtype = 'rectangles'; % 'rectangles','squares'. Discretize the coil using either squares or rectangles.
for k = 1:2:length(varargin), P.(varargin{k}) = varargin{k+1}; end

% Initialize inputs
R = vvdata(1); Z = vvdata(2);
dR = vvdata(3); dZ = vvdata(4);
if type_coil_description==3
  alpha = vvdata(5); beta = vvdata(6);
end

% Initialize the extrema of the trapezoid
pointr = zeros(4,1); pointz = zeros(4,1);
rout = []; zout = [];

% Find 4 extrema of the parallelogram. Working both for rectangular description and oblique description
% 2----3
% |    |
% 1----4
switch type_coil_description
  case 2 % Rectangle
    pointr(1) = R  - dR/2;
    pointr(2) = R  - dR/2;
    pointr(3) = R  + dR/2;
    pointr(4) = R  + dR/2;
    pointz(1) = Z - dZ/2;
    pointz(2) = Z + dZ/2;
    pointz(3) = Z + dZ/2;
    pointz(4) = Z - dZ/2;
    alpha = 0; 
    beta = 0;
  case 3 % Oblique
    beta = -beta; % Convert beta to clockwise
    pointr(1) = R;
    pointr(2) = R + dZ*sin(beta);
    pointr(3) = R + dR*cos(alpha) + dZ*sin(beta);
    pointr(4) = R + dR*cos(alpha);
    pointz(1) = Z;
    pointz(2) = Z + dZ*cos(beta);
    pointz(3) = Z + dR*sin(alpha) + dZ*cos(beta);
    pointz(4) = Z + dR*sin(alpha);
end
% The following generate the filaments in square. It then deform the
% square into a parallelogramm

% Define basis vector of the 2D parallelogram, with origin in 1
ex1 = [pointr(4)-pointr(1), pointz(4)-pointz(1)];
ey1 = [pointr(2)-pointr(1), pointz(2)-pointz(1)];

% Define basis vectors of the 2D square unit [0;1]x[0;1] space with origin in point 1.
% Uniform grid will be defined in this unit space and then deformed with affine transformation to match the parallelogram.
ex = [1,0];
ey = [0,1];

% Rotation matrix containing the scalar product of the deformed space
% and orthogonal space
S = ([ex*ex1', ex*ey1'; ey*ex1', ey*ey1']);

% Try to use a number of filaments that respect the shape of the parallelogram
ratio = sqrt((ex1*ex1'))/sqrt((ey1*ey1'));
if nturns < 1 || ((abs(nturns-1)<1e-2) && (abs(ratio-1)<1e-3))
  % If turn <1, force 1 single filament in the middle.
  % If close to 1 turn and close to square, use only 1 filament
  num1 = 1; num2 = 1;
else
  % If turn > 1, match the parallelogram shape.
  num1 = ceil(sqrt(abs(nturns)*ratio));
  num2 = ceil(num1/ratio);
end

% Compute area of the parallelogram.
Area = eval_area(pointr(:), pointz(:));

% The oblique parallelogram will be discretized into rectangles, oriented on the x,y axis with the same area.
pointx = zeros(num1*num2, 1);
pointy = zeros(num1*num2, 1);
for kk=1:num1
  for jj=1:num2
    pointx((kk-1)*num1+jj) = (2*kk-1)/(2*num1); %uniform grid in the [0;1]x[0;1] space
    pointy((kk-1)*num2+jj) = (2*jj-1)/(2*num2); %uniform grid in the [0;1]x[0;1] space
    tmp = S*[pointx((kk-1)*(num1)+jj), pointy((kk-1)*(num2)+jj)]'; % evaluate component in parallelogram space.
    rout(end+1) = pointr(1) + tmp(1); % translate to the original central location
    zout(end+1) = pointz(1) + tmp(2);  % translate to the original central location
  end
end

% Compute the area of each rectangles
nw = num1*num2;
aw = ones(nw,1)*Area/nw; % Assume each rectangle to have the same area

switch P.discrtype
  case 'rectangles'
    % Length x,y projection of parallelogram onto x,y axis for each parallelogram.
    xproj = ex*ex1'/num1;
    yproj = ey*ey1'/num2;
    % Compute an equivalent width and height for each parallelogram, to match the total area.
    % Assumption are:
    % aw = hw*ww
    % xproj/yproj = ww/hw
    hw = (aw.*yproj/xproj).^0.5;
    ww = hw*xproj/yproj;
case 'squares'
    % Assumes each fialement to be square of the same length
    ww = aw.^0.5;
    hw = ww;
end

% Cast into column vector.
rout = rout(:); zout = zout(:); 

%% Plot the coils discretization
if P.debugplot >2
  figure
  hold on
  axis equal
  patch(pointr(:),pointz(:),'red');
  for ii=1:numel(rout)
    patchr = rout(ii) + ww(ii)/2*[-1;-1;1;1];
    patchz = zout(ii) + hw(ii)/2*[-1;1;1;-1];
    patch(patchr,patchz,'blue', 'EdgeColor', 'green');
  end
  scatter(rout, zout, 'g.');
end

end

function area = eval_area(r, z)
area = abs(0.5 * (r(1) * z(2) + r(2) * z(3) + r(3) * z(4) + r(4) * z(1) - r(2) * z(1) - r(3) * z(2) - r(4) * z(3) - r(1) * z(4)));
end
