function [ids] = G2ids(G, P)
%G2ids Convert G structure to IDS device description.
%
% This function is mainly develop for test purposes and do not generate all possible IDS option
% for each field
%
% [+MEQ MatlabEQuilibrium Toolbox+]

%    Copyright 2022-2025 Swiss Plasma Center EPFL
%
%   Licensed under the Apache License, Version 2.0 (the "License");
%   you may not use this file except in compliance with the License.
%   You may obtain a copy of the License at
%
%       http://www.apache.org/licenses/LICENSE-2.0
%
%   Unless required by applicable law or agreed to in writing, software
%   distributed under the License is distributed on an "AS IS" BASIS,
%   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
%   See the License for the specific language governing permissions and
%   limitations under the License.

D = struct();
D.coil_geom_type = 2; % 2 = rectangle
D.wall_type = 'annular'; % annular
D.debugplot = 0;
D.debug = 0;
D.vessel_2_ids_key = 'wall'; % 'wall','pf_passive'. Select in which ids structure v is placed.
for k = fieldnames(D)'
  if ~isfield(P,k{1}), P.(k{1}) = D.(k{1}); end
end

ids = struct;
ids = G2pf_active(ids,G, P);
ids = G2wall(ids,G, P);
ids = G2magnetics(ids,G, P);
ids = G2pf_passive(ids,G,P);

end

function [ids] = G2pf_active(ids,G, P)
  % Convert G structure to pf_active
  pf_active = struct;

  % All coil are described as single element
  for ii=1:numel(G.dimc)
    s = struct();

    switch P.coil_geom_type
      case 2 % rectangle description
        s.geometry.geometry_type = 2; 
        s.geometry.rectangle.r = G.rc(ii);
        s.geometry.rectangle.z = G.zc(ii);
        % Check if circular description. 
        icirc = isnan(G.wc(ii)) & isnan(G.hc(ii)); % not rectangular windings
        assert(~icirc || sum(G.Twc(:,ii)) == 1, 'Only circular coils with single winding are supported')
        if icirc
          wc = sqrt(pi*G.dw(ii)^2);
          hc = wc;
        else
          wc = G.wc(ii);
          hc = G.hc(ii);
        end
        s.geometry.rectangle.width = wc;
        s.geometry.rectangle.height = hc;
        s.turns_with_sign = sum(abs(G.Twc(:,ii)))*sign(sum(G.Tca(ii,:)));
        pf_active.coil{ii}.element = {s};
        pf_active.coil{ii}.name = G.dimc{ii};
        pf_active.coil{ii}.resistance = G.Rc(ii);
      otherwise
        error('Geometry type %d not supported', P.coil_geom_type)
    end
  end

  % Get circuit coils connectivity
  connections = G2connections(G);

  % Add names of power supplies
  for ia=1:G.na
    pf_active.supply{ia}.name = G.dima{ia}; % Name of supply equal to name of the coil
    pf_active.circuit{ia}.connections = connections{ia};
    pf_active.circuit{ia}.name = G.dima{ia};
  end
  ids.pf_active = pf_active;
end

function [ids] = G2wall(ids,G,P)
  if strcmp(P.vessel_2_ids_key, 'wall')
    switch P.wall_type
      case 'annular'
        % Generated an outline description of the vessel from the filament description
        wall = struct();
        s = struct();
        rcenter = (max(G.rv)+min(G.rv))/2;
        zcenter = 0;
        lcenter = sqrt((G.zv - zcenter).^2 + (G.rv - rcenter).^2);
        theta = atan2(G.zv-zcenter, G.rv-rcenter);
        lin = lcenter + cos(pi/4).*G.wv;
        lout = lcenter - cos(pi/4).*G.wv;
        rin = rcenter + lin.*cos(theta);
        zin = zcenter +  lin.*sin(theta);
        zout = zcenter + lout.*sin(theta);
        rout = rcenter + lout.*cos(theta);

        if P.debugplot>2
          figure 
          hold on
          plot(rin, zin ,'.', 'DisplayName', 'Inner outline generated')
          plot(rout,zout, '.', 'DisplayName', 'Outer outline generated')
          plot(rcenter, zcenter, 'x', 'DisplayName', 'Center')
          plot(G.rv,G.zv, 'o', 'DisplayName', 'Vessel original')
          legend('show')
          axis equal
        end
        s.annular.outline_inner.r =  rin;
        s.annular.outline_inner.z = zin;
        s.annular.outline_outer.r = rout;
        s.annular.outline_outer.z = zout;

        % Compute and add resistivity
        res = (G.Rv.*G.wv.*G.hv/2/pi./G.rv);
        s.annular.resistivity = res(1);

        wall.description_2d{1}.vessel.type.index = 0;
        wall.description_2d{1}.vessel.unit{1} = s;

    otherwise
      error('Wall type %s not supported', wall_type)
    end
  end
  wall.description_2d{1}.limiter.unit{1}.outline.r = G.rl;
  wall.description_2d{1}.limiter.unit{1}.outline.z = G.zl;
  ids.wall = wall;
end

function [ids]=G2magnetics(ids, G, P)
  magnetics = struct();
  for ii=1:G.nf
    magnetics.flux_loop{ii}.position{1}.r = G.rf(ii);
    magnetics.flux_loop{ii}.position{1}.z = G.zf(ii);
    magnetics.flux_loop{ii}.name = G.dimf(ii);
  end

  for ii=1:G.nm
    magnetics.b_field_pol_probe{ii}.position.r = G.rm(ii);
    magnetics.b_field_pol_probe{ii}.position.z = G.zm(ii);
    magnetics.b_field_pol_probe{ii}.name = G.dimm(ii);
    magnetics.b_field_pol_probe{ii}.poloidal_angle = -G.am(ii); % - sign needed for MEQ to IDS conversion
  end
  ids.magnetics = magnetics;
end

function ids = G2pf_passive(ids,G,P)
  % Always initialize the empty structure
  ids.pf_passive.loop{1}.element = {};
  if strcmp(P.vessel_2_ids_key, 'pf_passive')
    for ii=1:G.nv
      ids.pf_passive.loop{1}.element{ii}.geometry.geometry_type = 3;
      ids.pf_passive.loop{1}.element{ii}.geometry.oblique.r = G.rv(ii)-G.wv(ii)/2;
      ids.pf_passive.loop{1}.element{ii}.geometry.oblique.z = G.zv(ii)-G.hv(ii)/2;
      ids.pf_passive.loop{1}.element{ii}.geometry.oblique.length_alpha = G.wv(ii);
      ids.pf_passive.loop{1}.element{ii}.geometry.oblique.length_beta = G.hv(ii);
      ids.pf_passive.loop{1}.element{ii}.geometry.oblique.alpha = 0;
      ids.pf_passive.loop{1}.element{ii}.geometry.oblique.beta = 0;
      ids.pf_passive.loop{1}.element{ii}.turns_with_sign = 1;
      ids.pf_passive.loop{1}.resistivity =  G.Rv(ii)/2/pi/G.rv(ii)*G.wv(ii)*G.hv(ii);
    end
  end

end
