classdef test_wave_compress < matlab.unittest.TestCase
  % test function for test_wave_compress.m
  
  properties
    verbose = 1;
  end
  
  properties (TestParameter)
    slope = struct('positive',1,'negative',-1)
  end
  
  methods(Test, TestTags={'Unit'})
    function test_scalar(testCase)
      % test for a scalar trace    
      t = (-0.1:5e-2:1)';
      y = zeros(size(t));
      y(t< 0.3) = t(t< 0.3);
      y(t>=0.3) = t(t>=0.3).^2;
      y(t>=0.7) = t(t>=0.7).^1;
      [tc,yc] = wave_compress(t,y);
      
      if testCase.verbose
        plot(t,y,'.-',tc,yc,'ok--');
        drawnow;
        shg
      end
      
      testCase.verifyLessThan(numel(tc),numel(t),'no compression?');
      testCase.verifyTrue(size(yc,1)==numel(tc),'output element size mismatch');
      testCase.verifyTrue(iscolumn(tc),'tc should be a column');
      testCase.verifyEqual(sort(tc),tc,'tc should be sorted');
      testCase.verifyEqual(yc([1,end],:),y([1,end],:),'first and last points should be included')
      testCase.verifyEqual(tc([1,end],:),t([1,end],:),'first and last points should be included')
    end
    
    function test_vector(testCase)
      % test for a vector trace
      t = (-0.3:5e-2:1)';
      
      n=3;
      y = zeros(numel(t),n);
      y(t< 0.3,:) = t(t< 0.3)*rand(1,n);
      y(t>=0.3,:) = t(t>=0.3).^2*ones(1,n);
      y(t>=0.7,:) = rand(sum(t>=0.7),n);
      
      [tc,yc] = wave_compress(t,y);
      if testCase.verbose
        plot(t,y,'.-',tc,yc,'o');
        drawnow;
        shg
      end
      testCase.verifyLessThan(numel(tc),numel(t),'no compression?');
      testCase.verifyTrue(size(yc,1)==numel(tc),'output element size mismatch');
      testCase.verifyTrue(iscolumn(tc),'tc should be a column');
      testCase.verifyEqual(sort(tc),tc,'tc should be sorted');
      testCase.verifyEqual(yc([1,end],:),y([1,end],:),'first and last points should be included')
      testCase.verifyEqual(tc([1,end],:),t([1,end],:),'first and last points should be included')
    end
    
    function test_straight_line(testCase,slope)
      % test for a straight line trace    
      t = (-0.1:5e-2:1)';
      y = slope*t;
      [tc,yc] = wave_compress(t,y);
      
      if testCase.verbose
        plot(t,y,'.-',tc,yc,'ok--');
        drawnow;
        shg
      end
      
      testCase.verifyEqual(yc,y([1,end],:),'Output value should be first and last point only')
      testCase.verifyEqual(tc,t([1,end],:),'Output time  should be first and last point only')
    end
    
    function test_errors(testCase)
      % test known errors
      t = linspace(0,1,3)';
      y = rand(numel(t)+1,3);
      testCase.verifyError(@() wave_compress(t,y),'WAVECOMPRESS:invalid_y_size')
      y = rand(3,10,5);
      testCase.verifyError(@() wave_compress(t,y),'WAVECOMPRESS:y_matrix')
      testCase.verifyError(@() wave_compress(y,t),'WAVECOMPRESS:t_size')
      
      t = [1 2 2]'; y = [1 2 3]';
      testCase.verifyError(@() wave_compress(t,y),'WAVECOMPRESS:t_sorted')      
    end
  end
end
