classdef tcv_temp_tree_fixture < matlab.unittest.fixtures.Fixture
  % Setup temporary storage of TCV PCS or RESULTS trees when writing MEQ test results
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties
    tree
    shot = -1;
    tree_path
    temp_path
  end

  methods
    function fixture = tcv_temp_tree_fixture(tree,shot)
      % Fixture constructor
      fixture.tree = validatestring(lower(tree),{'pcs','results'});
      if strcmp(fixture.tree,'results')
        if nargin<2, error('tcv_temp_tree_fixture with tree=results needs a 2nd argument "shot"'); end
        fixture.shot = shot;
      end
    end
  end

  methods (Access=protected)
    function tf = isCompatible(~,~)
      % Make fixtures incompatible by default
      tf = false;
    end
  end

  methods
    function setup(fixture)
      % Setup fixture

      % Establish fresh connection to tcvdata
      n = mdsremotelist;
      mdsconnect('tcvdata.epfl.ch');
      if (mdsremotelist == n) % Connection to tcvdata pre-existed
        % Ensure fresh connection to tcvdata (avoid filename cache issues)
        mdsdisconnect;
        mdsconnect('tcvdata.epfl.ch');
      end

      % Get current path
      fixture.tree_path = mdsvalue('getenv($1//"_path")',fixture.tree);

      % Create temporary tree storage
      [s,dir] = system('ssh tcvdata.epfl.ch mktemp --directory');
      if s, error('Could not create temporary directory'); end
      dir = deblank(dir);
      % Store info for teardown
      fixture.temp_path = dir;
      % Set permissions to all users to write there
      [s,~] = system(['ssh tcvdata.epfl.ch chmod 777 ',dir]);
      if s, error('Could not set appropriate permissions on temporary directory'); end

      % Add it to path
      mdsvalue('setenv($1//"_path="//$2//";"//$3)',fixture.tree,dir,fixture.tree_path);

      if strcmp(fixture.tree,'results')
        % Copy desired shot in temporary directory
        [~,st] = mdsopen('results',fixture.shot);
        if ~rem(st,2), error('Could not open results tree from reference shot'); end
        st = mdsvalue('TreeCreatePulseFile($1,0,[])',int32(fixture.shot));
        if ~rem(st,2), error('Could not create pulse file in temporary directory'); end
        mdsclose;
      end

      % Add description
      fixture.SetupDescription = sprintf('Created temporary tree storage under tcvdata::%s',dir);
    end

    function teardown(fixture)
      % Teardown fixture

      % Restore original path
      mdsvalue('setenv($1//"_path="//$2)',fixture.tree,fixture.tree_path);

      % Remove temporary tree storage
      [s,~] = system(['ssh tcvdata.epfl.ch rm -rf ',fixture.temp_path]);
      if s, error('Could not remove temporary directory %s',fixture.temp_path); end

      % Add description
      fixture.TeardownDescription = 'Restored original path and removed temporary directory on tcvdata';
    end
  end
end
