function G = fbtmeas(nvvel)
% G = fbtmeas
% Provides equivalent static tree data to mga.m from reading the file
% 'fbtmeas.dat' to provide data equivalent to the FORTRAN version. Now
% computes all data directly as in fbt.f
%
% Author: A. Merle (SPC-EPFL).

persistent cache

if nargin<1, nvvel=19;end

% Check cache
if ~isempty(cache)
  nv_list = [cache.nvvel];
  [~,ii] = ismember(nvvel,nv_list);
  if ii>0
    G = cache(ii(1)).G;
    return;
  end
end

usegreenem = false;

%%

% [nprob,nloop,nr,nz,ngroup,ri,ro,zu,...
%   rvf,zvf,hvf,wvf,tvf,kgroup,~,~,itype,...
%   rbee,zbee,thbee,rflu,zflu] = read_fbtmeas_;

[nprob,nloop,nr,nz,ngroup,ri,ro,zu,...
  rvf,zvf,hvf,wvf,tvf,kgroup,~,~,itype,...
  rbee,zbee,thbee,rflu,zflu] = fbtmeas_(0,nvvel);

%
ndigits = 5;
ri    = round_(ri   ,ndigits);
ro    = round_(ro   ,ndigits);
zu    = round_(zu   ,ndigits);
rvf   = round_(rvf  ,ndigits);
zvf   = round_(zvf  ,ndigits);
hvf   = round_(hvf  ,ndigits);
wvf   = round_(wvf  ,ndigits);
tvf   = round_(tvf  ,ndigits);
rbee  = round_(rbee ,ndigits);
zbee  = round_(zbee ,ndigits);
thbee = round_(thbee,ndigits);
rflu  = round_(rflu ,ndigits);
zflu  = round_(zflu ,ndigits);

%% Geometry

% Conductors
maskw = ismember(itype(kgroup),[1,2]);
G.nw = sum(maskw);
G.rw = rvf(maskw);
G.zw = zvf(maskw);
G.aw = wvf(maskw);
G.bw = hvf(maskw);

masku = ismember(itype(kgroup),4);
G.nu = sum(masku);
G.ru = rvf(masku);
G.zu = zvf(masku);
G.au = wvf(masku);
G.bu = hvf(masku);

G.na = sum(ismember(itype,[1,2]));
G.nv = sum(ismember(itype,   4 ));

% Magnetics
G.nm = nprob;
G.rm = rbee;
G.zm = zbee;
G.am = thbee*pi/180;

G.nf = nloop;
G.rf = rflu;
G.zf = zflu;

% Mesh
G.nrx = nr+1;
G.nzx = 2*nz+1;
G.nx  = G.nrx*G.nzx;
G.rx  = linspace( ri,ro,G.nrx);
G.zx  = linspace(-zu,zu,G.nzx);
G.ax  = (ro-ri)/nr;
G.bx  = (zu)/nz;
[G.rrx,G.zzx] = meshgrid(G.rx,G.zx);

%% Green's functions

G.Twa = zeros(G.nw,G.na,'single');
G.Tuv = zeros(G.nu,G.nv,'single');
ia = 0;
iv = 0;
for ii=1:ngroup
  if ismember(itype(ii),[1,2])
    ia = ia+1;
    G.Twa(kgroup(maskw) == ii,ia) = tvf(maskw & kgroup == ii);
  else
    iv = iv+1;
    G.Tuv(kgroup(masku) == ii,iv) = tvf(masku & kgroup == ii);
  end
end

%TODO: G.Txy

% Toroidal coil (only used for quadrupole backoffs)
%   (coefficients hardcoded in .F, replicated in .M when applicable - no doublets)
G.rt = 1;
G.zt = 0;
G.Ttt = 1;

% resistance and mutual inductance
G.Ra = single([0.01180*ones(8,1);... E
               0.03908*ones(8,1);... F
               0.01495-0.00070;...   OH1
               0.01333;...           OH2
              ]);
G.Rv = single([1.2711;1.2738;1.2738;1.2738;...
               1.2738;1.2738;1.2895;1.3213;...
               1.6773;2.0924;2.3368;2.7809;...
               2.6548;2.1638;3.0275;3.5829;...
               3.3532;3.2480;3.4017;3.7648;...
               3.4017;3.2480;3.3153;3.4573;...
               2.9508;2.0713;2.3129;2.5959;...
               2.5119;2.1984;1.7093;1.3174;...
               1.2824;1.2738;1.2738;1.2738;...
               1.2738;1.2738;...
              ]*0.001);
if G.nv == 19
  G.Rv = 1./(.5./G.Rv([38 2:2:36])+1./G.Rv(1:2:37)+.5./G.Rv(2:2:38));
end

if usegreenem
  
  G.Maa = G.Twa.'*            greenem('mut',G.rw,G.zw,[G.aw,G.bw]      )*G.Twa;
  G.Mva = G.Tuv.'*            greenem('mut',G.ru,G.zu,G.rw    ,G.zw    )*G.Twa;
  G.Mvv = G.Tuv.'*            greenem('mut',G.ru,G.zu,[G.au,G.bu]      )*G.Tuv;
  
  G.Mfa =                     greenem('mut',G.rf,G.zf,G.rw    ,G.zw    )*G.Twa;
  G.Mfv =                     greenem('mut',G.rf,G.zf,G.ru    ,G.zu    )*G.Tuv;
  G.Mfx =                     greenem('mut',G.rf,G.zf,G.rrx(:),G.zzx(:));
  
  G.Bma =         (cos(G.am).*greenem('br' ,G.rm,G.zm,G.rw    ,G.zw    ) + ...
    sin(G.am).*greenem('bz' ,G.rm,G.zm,G.rw    ,G.zw    ))*G.Twa;
  G.Bmv =         (cos(G.am).*greenem('br' ,G.rm,G.zm,G.ru    ,G.zu    ) + ...
    sin(G.am).*greenem('bz' ,G.rm,G.zm,G.ru    ,G.zu    ))*G.Tuv;
  G.Bmx =         (cos(G.am).*greenem('br' ,G.rm,G.zm,G.rrx(:),G.zzx(:)) + ...
    sin(G.am).*greenem('bz' ,G.rm,G.zm,G.rrx(:),G.zzx(:)));
 
else
  
  Mww = greenem_('mut',G.rw,G.zw,G.aw,G.bw,4,8);
  G.Maa = G.Twa.'*Mww*G.Twa;
  
  Muw = greenem_('mut',G.ru,G.zu,G.au,G.bu,G.rw,G.zw,G.aw,G.bw,4);
  G.Mva = G.Tuv.'*Muw*G.Twa;
  
  Muu = greenem_('mut',G.ru,G.zu,G.au,G.bu,4,8);
  G.Mvv = G.Tuv.'*Muu*G.Tuv;
  
  Mfw = greenem_('mut',G.rf,G.zf,G.rw,G.zw);
  G.Mfa = Mfw*G.Twa;
  
  Mfu = greenem_('mut',G.rf,G.zf,G.ru,G.zu);
  G.Mfv = Mfu*G.Tuv;
  
  Mfx = greenem_('mut',G.rf,G.zf,G.rrx(:),G.zzx(:));
  G.Mfx = Mfx;
  
  Bmw = cos(G.am).*greenem_('br',G.rm,G.zm,G.rw,G.zw) + ...
        sin(G.am).*greenem_('bz',G.rm,G.zm,G.rw,G.zw);
  G.Bma = Bmw*G.Twa;
  
  Bmu = cos(G.am).*greenem_('br',G.rm,G.zm,G.ru,G.zu) + ...
        sin(G.am).*greenem_('bz',G.rm,G.zm,G.ru,G.zu);
  G.Bmv = Bmu*G.Tuv;
  
  Bmx = cos(G.am).*greenem_('br',G.rm,G.zm,G.rrx(:),G.zzx(:)) + ...
        sin(G.am).*greenem_('bz',G.rm,G.zm,G.rrx(:),G.zzx(:));
  G.Bmx = Bmx;

end
               
G.Mfa = G.Mfa-G.Mfa(1,:);
G.Mfv = G.Mfv-G.Mfv(1,:);
G.Mfx = G.Mfx-G.Mfx(1,:);

G.greenem = @greenem_;

% Fill cache
ii = numel(cache)+1;
cache(ii).nvvel = nvvel;
cache(ii).G     = G;

end

function G = greenem_(mode,r1,z1,varargin)

mu0 = 4e-7*pi;

switch lower(mode)
  case 'self'
    % 'self',r1,z1,[rd1,zd1],ns
    A = varargin{1};if size(A,1) == 2, A = A.';end
    ns = varargin{2};
    G = reshape(selfrc(r1(:),A(:,2),A(:,1),1,ns),size(r1));
  case 'mut'
    if nargin == 5
      % mode,r1,z1,r2,z2
      % Simple mutual between 2 systems with gf
      n1 = numel(r1);
      r2 = varargin{1};
      z2 = varargin{2};
      n2 = numel(r2);
      r1 = repmat(reshape(r1,n1,1),1,n2);
      z1 = repmat(reshape(z1,n1,1),1,n2);
      r2 = repmat(reshape(r2,1,n2),n1,1);
      z2 = repmat(reshape(z2,1,n2),n1,1);
      G = gf(0,0,r1,z1,r2,z2);
      G = G*(-mu0);
    elseif nargin == 7
      % mode,r1,z1,rd1,zd1,nm,ns
      n1 = numel(r1);
      rd1 = varargin{1};
      zd1 = varargin{2};
      nm = varargin{3};
      ns = varargin{4};
      S   = selfrc(r1(:),zd1(:),rd1(:),1,ns);
      r1  = repmat(reshape( r1,n1,1),1,n1);
      z1  = repmat(reshape( z1,n1,1),1,n1);
      rd1 = repmat(reshape(rd1,n1,1),1,n1);
      zd1 = repmat(reshape(zd1,n1,1),1,n1);
      r2  =  r1.';
      z2  =  z1.';
      rd2 = rd1.';
      zd2 = zd1.';
      G = amutlrc(r1,z1,zd1,rd1,1,...
                  r2,z2,zd2,rd2,1,nm);
      G(1:n1+1:end) = S(:);
    elseif nargin == 10
      % mode,r1,z1,rd1,zd1,r2,z2,rd2,zd2,nm
      n1 = numel(r1);
      rd1 = varargin{1};
      zd1 = varargin{2};
      r2  = varargin{3};
      z2  = varargin{4};
      rd2 = varargin{5};
      zd2 = varargin{6};
      nm  = varargin{7};
      n2 = numel(r2);
      r1  = repmat(reshape( r1,n1,1),1,n2);
      z1  = repmat(reshape( z1,n1,1),1,n2);
      rd1 = repmat(reshape(rd1,n1,1),1,n2);
      zd1 = repmat(reshape(zd1,n1,1),1,n2);
      r2  = repmat(reshape( r2,1,n2),n1,1);
      z2  = repmat(reshape( z2,1,n2),n1,1);
      rd2 = repmat(reshape(rd2,1,n2),n1,1);
      zd2 = repmat(reshape(zd2,1,n2),n1,1);
      G = amutlrc(r1,z1,zd1,rd1,1,...
                  r2,z2,zd2,rd2,1,nm);
    else
      error('Wrong number of arguments for case "mut"');
    end
  case 'br'
    n1 = numel(r1);
    r2 = varargin{1};
    z2 = varargin{2};
    n2 = numel(r2);
    r1 = repmat(reshape(r1,n1,1),1,n2);
    z1 = repmat(reshape(z1,n1,1),1,n2);
    r2 = repmat(reshape(r2,1,n2),n1,1);
    z2 = repmat(reshape(z2,1,n2),n1,1);
    [~,G] = gradgf(0,0,r1,z1,r2,z2);
    G = G./r1*mu0/2/pi;
  case 'bz'
    n1 = numel(r1);
    r2 = varargin{1};
    z2 = varargin{2};
    n2 = numel(r2);
    r1 = repmat(reshape(r1,n1,1),1,n2);
    z1 = repmat(reshape(z1,n1,1),1,n2);
    r2 = repmat(reshape(r2,1,n2),n1,1);
    z2 = repmat(reshape(z2,1,n2),n1,1);
    [G,~] = gradgf(0,0,r1,z1,r2,z2);
    G = -G./r1*mu0/2/pi;
end
    

end

% function answ = gf(ineg,nmult,xt,zt,xs,zs)
function answ = gf(~,~,xt,zt,xs,zs)
% calculates poloidal flux at point (xt,zt) due to current
% at location (xs,zs)  . . . returns value in ans
[q0,q1,q2,q3,q4,q5] = deal(-.30685281944e0, .29822748722e0,...
           .00367617100e0, -.01091055499e0, .00860373511e0,...
           .00725598106e0);
[r0,r1,r2,r3,r4,r5] = deal(.25e0, .06250928488e0,...
           .00489241049e0, .01034604435e0, .01358621540e0,...
           .00220893506e0);
% [pi,tpi] = deal(3.1415926535,6.283185308);

zr = zs - zt;
qk2 = 4.*xs.*xt./((xs+xt).^2 + zr.^2);
qn = 1.0 - qk2;
qlg = -log(qn);
bk = q0+qn.*(q1+qn.*(q2+qn.*(q3+qn.*(q4+qn.*q5)))) ...
  + (r0+qn.*(r1+qn.*(r2+qn.*(r3+qn.*(r4+qn.*r5))))).*qlg;
answ = -sqrt(xt.*xs./qk2).*bk*2;
answ(qn <= 0) = 0;
end

function [gradx,gradz] = gradgf(ineg,nmult,xt,zt,xs,zs)

% calculates gradient of the poloidal flux at
% point (xt,zt) due to current at location (xs,zs)
% if current is 1.0 amp, magnetic field is given by
% Bx = (gradz/xt)*(usdi/tpi)
% Bz = -(gradx/xt)*(usdi/tpi)
%                             usdi=4*pi*1.0e-7 and tpi=2*pi

% pi = 3.1415926535897;
ak0 = 1.38629436112;
ak1 = 0.09666344259;
ak2 = 0.03590092383;
ak3 = 0.03742563713;
ak4 = 0.01451196212;
bk0 = 0.5;
bk1 = 0.12498593597;
bk2 = 0.06880248576;
bk3 = 0.03328355346;
bk4 = 0.00441787012;
ae1 = 0.44325141463;
ae2 = 0.0626060122;
ae3 = 0.04757383546;
ae4 = 0.01736506451;
be1 = 0.2499836831;
be2 = 0.09200180037;
be3 = 0.04069697526;
be4 = 0.00526449639;

% tpi = 2.*pi;
zd = zs - zt;
ck2 = 4.*xs.*xt./((xs+xt).^2 + zd.^2);
x1 = 1. - ck2;
x2 = x1.*x1;
x3 = x2.*x1;
x4 = x3.*x1;
ck = sqrt(ck2);

elipk = ak0+ak1*x1+ak2*x2+ak3*x3+ak4*x4 ...
     - (bk0+bk1*x1+bk2*x2+bk3*x3+bk4*x4).*log(x1);

elipe=1.0+ae1*x1+ae2*x2+ae3*x3+ae4*x4 + ...
         - (be1*x1+be2*x2+be3*x3+be4*x4).*log(x1);
mask = abs(x1) > 1.0e-6;
elipe(~mask) = 1.0;

term1 = -2.*elipk;
term2 = (2.-ck2)./(1.-ck2).*elipe;
fac = 1./(4.*sqrt(xs.*xt))./ck.*(term1+term2);
answ = gf(ineg,nmult,xt,zt,xs,zs);
gradx = answ/2./xt - fac.*(2.*xs - ck2.*(xt+xs));
gradz = -fac.*ck2.*(zs-zt);
end   
               
function Mut = amutlrc (a1,zc1,zd1,rd1,t1,a2,zc2,zd2,rd2,t2,n)
  % mutual of two co-axial coils
  % summing flux from filaments
  % dl is the size of the subunit for discretizing
  dl=sqrt((a1-a2).^2+(zc1-zc2).^2)/n;
  dl(dl == 0) = max(dl(:));
  nr1=max(1,fix(rd1./dl+0.5));
  nz1=max(1,fix(zd1./dl+0.5));
  dr1=rd1./nr1;
  dz1=zd1./nz1;
  
  nr2=max(1,fix(rd2./dl+0.5));
  nz2=max(1,fix(zd2./dl+0.5));
  dr2=rd2./nr2;
  dz2=zd2./nz2;
  
  nd = ndims(a1);
  sd = size(a1);
  
  sets = double(unique([nr2(:),nz2(:),nr1(:),nz1(:)],'rows'));
  
  sflx = NaN(sd);
  for ii = 1:size(sets,1)
    nr2m = sets(ii,1); nz2m = sets(ii,2);
    nr1m = sets(ii,3); nz1m = sets(ii,4);
    
    mask = (nr2(:) == nr2m) & (nz2(:) == nz2m) & (nr1(:) == nr1m) & (nz1(:) == nz1m);
    size12 = sum(double(mask));
    
    ir2 = reshape(1:nr2m,[ones(1,nd),nr2m]);
    iz2 = reshape(1:nz2m,[ones(1,nd),nz2m]);
    ir1 = reshape(1:nr1m,[ones(1,nd),nr1m]);
    iz1 = reshape(1:nz1m,[ones(1,nd),nz1m]);
    
    r2=(ir2-0.5).*dr2(mask)-0.5*rd2(mask)+a2(mask);
    r2=r2.*(1.+(dr2(mask)./r2).^2/24.);
    z2=(iz2-0.5).*dz2(mask)-0.5*zd2(mask)+zc2(mask);
    r1=(ir1-0.5).*dr1(mask)-0.5*rd1(mask)+ a1(mask);
    r1=r1.*(1.+(dr1(mask)./r1).^2/24.);
    z1=(iz1-0.5).*dz1(mask)-0.5*zd1(mask)+zc1(mask);
    r2 = repmat(reshape(r2,[size12,nr2m,   1,   1,   1]),[1,   1,nz2m,nr1m,nz1m]);
    z2 = repmat(reshape(z2,[size12,   1,nz2m,   1,   1]),[1,nr2m,   1,nr1m,nz1m]);
    r1 = repmat(reshape(r1,[size12,   1,   1,nr1m,   1]),[1,nr2m,nz2m,   1,nz1m]);
    z1 = repmat(reshape(z1,[size12,   1,   1,   1,nz1m]),[1,nr2m,nz2m,nr1m,   1]);
    %
    flx = sfilfx(r1,z1,1.0,r2,z2);
    sflx(mask) = sum(reshape(flx,[size12,nr2m*nz2m*nr1m*nz1m]),2);
  end
  
  Mut=t1.*t2.*sflx./(nr1.*nz1.*nr2.*nz2);
end

function Sel = selfrc(a,zd,rd,t,n)
% self-inductance of coil
% summing flux from filaments
% a/n is the subunit for discretizing
mu0 = 12.56637e-7;

nr=max(1,fix(rd./(a/n)+0.5));
nz2=fix(0.5*zd./(a/n)+0.5);
nz=max(2,2*nz2);
dr=rd./nr;
dz=zd./nz;
sr=sqrt(dr.*dz/3.14159);

nd = ndims(a);
sd = size(a);

nrm = max(nr(:));
nzm = max(nz(:));

ir = reshape(1:nrm,[ones(1,nd),nrm]);
iz = reshape(1:nzm,[ones(1,nd),nzm]);

r=(ir-0.5).*dr-0.5*rd+a;
z=(iz-0.5).*dz-0.5*zd;
r2 = repmat(reshape(r,[sd,nrm,  1,  1,  1]),[ones(1,nd),  1,nzm,nrm,nzm]);
z2 = repmat(reshape(z,[sd,  1,nzm,  1,  1]),[ones(1,nd),nrm,  1,nrm,nzm]);
r1 = repmat(reshape(r,[sd,  1,  1,nrm,  1]),[ones(1,nd),nrm,nzm,  1,nzm]);
z1 = repmat(reshape(z,[sd,  1,  1,  1,nzm]),[ones(1,nd),nrm,nzm,nrm,  1]);
%
% Eliminate extra filaments
r2(r2 > a + 0.5*rd) = 0;
r2(z2 >     0.5*zd) = 0;
r2(r1 > a + 0.5*rd) = 0;
r2(z1 >     0.5*zd) = 0;
%
flx = sfilfx(r1,z1,1.0,r2,z2);
% 
flx = reshape(flx,numel(a),[]);
r1  = reshape( r1,numel(a),[]);
r1s = r1(:,1:(nrm*nzm+1):end);
r2  = reshape( r2,numel(a),[]);
r2s = r2(:,1:(nrm*nzm+1):end);
r1s(r2s == 0) = 0;
flx(:,1:(nrm*nzm+1):end) = mu0*r1s.*(log(8.0*max(r1s,eps)./sr(:))-1.75);

sflx = sum(reshape(flx,[sd,nrm*nzm*nrm*nzm]),nd+1);

Sel=t.^2.*sflx./(nr.*nz).^2;
end

function fx = sfilfx(rf,zf,at,rp,zp)
c1=12.56637e-7;
c2=c1*at;
z=(zp-zf)./rf;
r=rp./rf;
p2=(1.0+r).*(1.0+r)+z.*z;
p=sqrt(p2);
a2=p2-4.0*r;

h=4.0*r./p2;
v=a2./p2;

eka=(((0.0145119621 *v+0.0374256371).*v+0.0359009238).*v+0.0966634426).*v+1.38629436;
ekb=(((0.00441787012*v+0.0332835535).*v+0.0688024858).*v+0.1249859360).*v+0.5;
eea=(((0.0173650645 *v+0.0475738355).*v+0.0626060122).*v+0.443251415 ).*v+1.0;
eeb=(((0.00526449639*v+0.0406969753).*v+0.0920018004).*v+0.249983683 ).*v;

y=log(v);
ce=eea-y.*eeb;
ck=eka-y.*ekb;
fx=p.*((1.0-0.5.*h).*ck-ce).*c2.*rf;

fx(rp == 0) = 0;
fx(at == 0) = NaN;
fx(a2 <= 0) = sign(at)*1e20;
end

function [nprob,nloop,nr,nz,ngroup,ri,ro,zu,...
  rvf,zvf,hvf,wvf,tvf,kgroup,isvf,isvl,itype,rbee,zbee,thbee,rflu,zflu] = read_fbtmeas_

% Truncate to avoid round-off errors leading to differences with .f
A = single(importdata('fbtmeas.dat'));

% nruns,icur,nprob,nloop,nr,nz,iallf,ngroup
sizes = num2cell(A(1,:));
% [nruns,icur,nprob,nloop,nr,nz,iallf,ngroup] = deal(sizes{:});
[~,~,nprob,nloop,nr,nz,iallf,ngroup] = deal(sizes{:});
% ri,ro,zu,rhocu,filfac,rhovv
scalars = num2cell(A(2,1:6));
% [ri,ro,zu,rhocu,filfac,rhovv] = deal(scalars{:});
[ri,ro,zu,~,~,~] = deal(scalars{:});
iA = 2;

jallf = ceil(iallf/6);
tmp = A(iA+(1:jallf),1:6).';
rvf = tmp(1:iallf).';
iA = iA + jallf;
tmp = A(iA+(1:jallf),1:6).';
zvf = tmp(1:iallf).';
iA = iA + jallf;
tmp = A(iA+(1:jallf),1:6).';
hvf = tmp(1:iallf).';
iA = iA + jallf;
tmp = A(iA+(1:jallf),1:6).';
wvf = tmp(1:iallf).';
iA = iA + jallf;
tmp = A(iA+(1:jallf),1:6).';
tvf = tmp(1:iallf).';
iA = iA + jallf;
jallf = ceil(iallf/12);
if (iallf - (jallf-1)*12 <= 8), jallf = 2*jallf-1;
else,                           jallf = 2*jallf;
end
tmp = A(iA+(1:jallf),1:8).';
tmp = tmp(mod(0:jallf*8-1,16) < 12);
kgroup = tmp(1:iallf).';
iA = iA + jallf;

jgroup = ceil(ngroup/12);
if (ngroup - (jgroup-1)*12 <= 8), jgroup = 2*jgroup-1;
else,                             jgroup = 2*jgroup;
end
tmp = A(iA+(1:jgroup),1:8).';
tmp = tmp(mod(0:jgroup*8-1,16) < 12);
isvf = tmp(1:ngroup).';
iA = iA + jgroup;
tmp = A(iA+(1:jgroup),1:8).';
tmp = tmp(mod(0:jgroup*8-1,16) < 12);
isvl = tmp(1:ngroup).';
iA = iA + jgroup;
tmp = A(iA+(1:jgroup),1:8).';
tmp = tmp(mod(0:jgroup*8-1,16) < 12);
itype = tmp(1:ngroup).';
iA = iA + jgroup;

jprob = ceil(nprob/6);
tmp = A(iA+(1:jprob),1:6).';
rbee = tmp(1:nprob).';
iA = iA + jprob;
tmp = A(iA+(1:jprob),1:6).';
zbee = tmp(1:nprob).';
iA = iA + jprob;
tmp = A(iA+(1:jprob),1:6).';
thbee = tmp(1:nprob).';
iA = iA + jprob;

jloop = ceil(nloop/6);
tmp = A(iA+(1:jloop),1:6).';
rflu = tmp(1:nloop).';
iA = iA + jloop;
tmp = A(iA+(1:jloop),1:6).';
zflu = tmp(1:nloop).';
% iA = iA + jloop;

end

function [nprob,nloop,nr,nz,ngroup,ri,ro,zu,...
  rvf,zvf,hvf,wvf,tvf,kgroup,isvf,isvl,itype,rbee,zbee,thbee,rflu,zflu] = fbtmeas_(nfast,nvvel)

ri = single(0.58);
ro = single(1.18);
zu = single(0.794839);
nr = 30;
nz = 32;

nfilae = 4;
nfilaf = 2;
nloop = 38;
nprob = 38;
if nfast>1, nshafa = 19;
else,       nshafa = 18;end
ngroup = nshafa + nvvel;

% rhocu = single(2.0e-8);
% filfac = single(0.625);
% rhovv = single(0.75e-6);

roh = single([.4225,.4225,.4225,.4225,.4225,.4225,...
              .4225,.4225,.4225,.4225,.4225,.4225,...
              .4458,.4458,.4458,.4458,.6215,.6215,1.1765,1.1765]);
zoh = single([-.726,-.594,-.462,-.330,-.198,-.066,...
              .066,.198,.330,.462,.594,.726,...
              -.983,-.889,.889,.983,-1.110,1.110,-1.170,1.170]);
toh = single([11.917,11.917,11.917,11.917,11.917,11.917,...
              11.917,11.917,11.917,11.917,11.917,11.917,...
              14.5,14.5,14.5,14.5,12.0,12.0,8.0,8.0]);
hoh = single([.132,.132,.132,.132,.132,.132,.132,.132,.132,.132,...
              .132,.132,.094,.094,.094,.094,.120,.120,.090,.090]);
woh = single([.063,.063,.063,.063,.063,.063,.063,.063,.063,.063,...
              .063,.063,.1095,.1095,.1095,.1095,.066,.066,.066,.066]);


rshap16 = single([.505,.505,.505,.505,.505,.505,.505,.505,...
                  1.3095,1.3095,1.3095,1.3095,1.3095,1.3095,1.3095,1.3095]);
zshap16 = single([-.7,-.5,-.3,-.1,.1,.3,.5,.7,...
                  -.77,-.61,-.31,-.15,.15,.31,.61,.77]);
tshap16 = single([34.,34.,34.,34.,34.,34.,34.,34.,...
                  36.,36.,36.,36.,36.,36.,36.,36.]);
hshap16 = single([.180,.180,.180,.180,.180,.180,.180,.180,...
                  .075,.075,.075,.075,.075,.075,.075,.075]);
wshap16 = single([.051,.051,.051,.051,.051,.051,.051,.051,...
                  .120,.120,.120,.120,.120,.120,.120,.120]);



rfast6 = single([1.099,1.114,1.129,1.129,1.114,1.129]);
zfast6 = single([-.648,-.630,-.612,0.612,0.630,0.648]);
hfast6 = single([.015,.015,.015,.015,.015,.015]);
wfast6 = single([.015,.015,.015,.015,.015,.015]);
tfast6 = single([-1.,-1.,-1.,1.,1.,1.]);

rfast2 = single([1.114,1.114]);
zfast2 = single([-.630,0.630]);
hfast2 = single([.020,.020]);
wfast2 = single([.020,.020]);
tfast2 = single([-1.,1.]);


rbeeh = single([.606,.606,.606,.606,.606,.606,.606,.6294,.695,.790,...
                .885,.980,1.0421,1.1188,1.154,1.154,1.154,1.154,1.154,...
                1.154,1.154,1.154,1.154,1.154,1.154,1.1188,1.0421,.980,.885,...
                .790,.695,.6294,.606,.606,.606,.606,.606,.606]);
zbeeh = single([.000,.115,.230,.345,.460,.575,.690,.7456,.764,.764,...
                .764,.764,.7348,.6705,.575,.460,.345,.230,.115,0.0,-.115,...
                -.230,-.345,-.460,-.575,-.6705,-.7348,-.764,-.764,-.764,...
                -.764,-.7456,-.690,-.575,-.460,-.345,-.230,-.115]);
thbeeh = single([90.,90.,90.,90.,90.,90.,90.,45.,0.,0.,0.,0.,-40.,-40.,...
                 -90.,-90.,-90.,-90.,-90.,-90.,-90.,-90.,-90.,-90.,-90.,...
                 -140.,-140.,-180.,-180.,-180.,-180.,-225.,-270.,-270.,-270.,...
                 -270.,-270.,-270.]);
rfluh = single([.584,.584,.584,.584,.584,.584,.584,.600,.695,.790,.885,...
                .980,1.0595,1.1361,1.176,1.176,1.176,1.176,1.176,1.176,1.176,...
                1.176,1.176,1.176,1.176,1.1361,1.0595,.980,.885,.790,.695,...
                .600,.584,.584,.584,.584,.584,.584]);
zfluh = single([.000,.115,.230,.345,.460,.575,.690,.772,.791,.791,...
                .791,.791,.7555,.6912,.575,.460,.345,.230,.115,0.0,...
                -.115,-.230,-.345,-.460,-.575,-.6912,-.7555,-.791,-.791,...
                -.791,-.791,-.772,-.690,-.575,-.460,-.345,-.230,-.115]);

rbee  =  rbeeh(1:nprob).';
zbee  =  zbeeh(1:nprob).';
thbee = thbeeh(1:nprob).';

rflu  = rfluh(1:nloop).';
zflu  = zfluh(1:nloop).';

rvvelb = single([...
     0.5925,0.5925,0.5925,0.5925,0.5925,0.5925,0.5925,0.5950,...
     0.6280,0.7425,0.8375,0.9325,1.0130,1.0900,1.1675,1.1675,...
     1.1675,1.1675,1.1675,1.1675,1.1675,1.1675,1.1675,1.1675,...
     1.1675,1.1675,1.0900,1.0130,0.9325,0.8375,0.7425,0.6280,...
     0.5950,0.5925,0.5925,0.5925,0.5925,0.5925,0.5925]);
zvvelb = single([...
     -.0575,0.0575,0.1725,0.2875,0.4025,0.5175,0.6325,0.7470,...
     0.7800,0.7800,0.7800,0.7800,0.7800,0.7135,0.6470,0.5175,...
     0.4025,0.2875,0.1725,0.0575,-.0575,-.1725,-.2875,-.4025,...
     -.5175,-.6470,-.7135,-.7800,-.7800,-.7800,-.7800,-.7800,...
     -.7470,-.6325,-.5175,-.4025,-.2875,-.1725,-.0575]);
wvvelh = single([...
     .015,.015,.015,.015,.015,.015,.016,.020,.020,.020,...
     .020,.020,.020,.020,.016,.015,.015,.015,.015,.015,...
     .015,.015,.015,.015,.016,.020,.020,.020,.020,.020,...
     .020,.020,.016,.015,.015,.015,.015,.015]);
   
[itype,isvl,isvf] = deal(zeros(ngroup,1,'int32'));

% E coils
[rvfe,zvfe,hvfe,wvfe,tvfe] = deal(zeros(8*nfilae,1,'single'));
kgroupe = zeros(8*nfilae,1,'int32');
ii = 1;
for k = 1:8
  itype(k) = 1;
  isvf(k)  = ii;
  for kf = 1:nfilae
    hvfe(ii) = hshap16(k)/single(nfilae);
    wvfe(ii) = wshap16(k);
    rvfe(ii) = rshap16(k);
    zvfe(ii) = zshap16(k) - 0.5*hshap16(k) + (kf-0.5)*hvfe(ii);
    tvfe(ii) = tshap16(k)/single(nfilae);
    kgroupe(ii) = k;
    ii = ii+1;
  end
  isvl(k) = ii-1;
end

% F coils
[rvff,zvff,hvff,wvff,tvff] = deal(zeros(8*nfilaf,1,'single'));
kgroupf = zeros(8*nfilaf,1,'int32');
ii = 1;
for k = 9:16
  itype(k) = 1;
  isvf(k)  = ii + isvl(8);
  for kf = 1:nfilaf
    hvff(ii) = hshap16(k);
    wvff(ii) = wshap16(k)/single(nfilaf);
    rvff(ii) = rshap16(k) - 0.5*wshap16(k) + (kf-0.5)*wvff(ii);
    zvff(ii) = zshap16(k);
    tvff(ii) = tshap16(k)/single(nfilaf);
    kgroupf(ii) = k;
    ii = ii+1;
  end
  isvl(k) = ii-1 + isvl(8);
end

% OH coils
itype(17:18) = 2;
isvf(17:18) = isvl(16)+cumsum(int32([1,12]));
rvfoh = roh(1:20).';
zvfoh = zoh(1:20).';
hvfoh = hoh(1:20).';
wvfoh = woh(1:20).';
tvfoh = toh(1:20).';
kgroupoh = [17*ones(12,1,'int32');18*ones(8,1,'int32')];
isvl(17:18) = isvl(16)+cumsum(int32([12,8]));

% fast coils
if nfast>1
  itype(19) = 3;
  isvf(19) = isvl(18)+1;
  switch nfast
    case 2
      rvffast = rfast2.';
      zvffast = zfast2.';
      hvffast = hfast2.';
      wvffast = wfast2.';
      tvffast = tfast2.';
    case 6
      rvffast = rfast6.';
      zvffast = zfast6.';
      hvffast = hfast6.';
      wvffast = wfast6.';
      tvffast = tfast6.';
    otherwise
      rvffast = [];
      zvffast = [];
      hvffast = [];
      wvffast = [];
      tvffast = [];
  end
  kgroupfast = 19*ones(size(rvffast),'int32');
  isvl(19) = isvf(19)+numel(rvffast)-1;
else
  [rvffast,zvffast,hvffast,wvffast,tvffast,kgroupfast] = deal([]);
end

% vacuum vessel filaments
itype(nshafa+(1:nvvel)) = 4;
wvvel = wvvelh(1:38);
hvvel = sqrt(diff(rvvelb).^2 + diff(zvvelb).^2);
nfvel = int32(fix(hvvel./wvvel));
ii = 1;
[isvfvel,isvlvel,kgroupvel] = deal(zeros(38,1,'int32'));
[hvfvel,wvfvel,rvfvel,zvfvel] = deal(zeros(sum(nfvel),1,'single'));
for k = 1:38
  isvfvel(k) = ii;
  for kf = 1:nfvel(k)
    hvfvel(ii) = hvvel(k)/single(nfvel(k));
    wvfvel(ii) = wvvel(k);
    filafr = (single(kf)-0.5)/single(nfvel(k));
    rvfvel(ii) = rvvelb(k) + (rvvelb(k+1) - rvvelb(k))*filafr;
    zvfvel(ii) = zvvelb(k) + (zvvelb(k+1) - zvvelb(k))*filafr;
    kgroupvel(ii) = k;
    ii = ii+1;
  end
  isvlvel(k) = ii-1;
end
nvfvel = isvlvel(38);

switch nvvel
  case 38
    iorder = 1:nvfvel;
    ifirst = isvfvel;
    ilast  = isvlvel;
  case 19
    ilast(1:19) = isvfvel(2:2:38) + int32(fix(single(isvlvel(2:2:38)-isvfvel(2:2:38))/2));
    ifirst([2:19,1]) = ilast(1:19)+1;
    ishift = nvfvel - ifirst(1) + 1;
    iorder = mod((1:nvfvel) - ishift -1,nvfvel)+1;
    ifirst = mod(ifirst     + ishift -1,nvfvel)+1;
    ilast  = mod(ilast      + ishift -1,nvfvel)+1;
  otherwise, error('fbtmeas_ nvvel value %d unsupported',nvvel);
end

rvfv = rvfvel(iorder);
zvfv = zvfvel(iorder);
hvfv = hvfvel(iorder);
wvfv = wvfvel(iorder);
tvfv = zeros(size(rvfv),'single');
kgroupv = zeros(size(rvfv),'int32');
%
confil = hvfv.*wvfv./rvfv;
for k = 1:nvvel
  mask = ifirst(k):ilast(k);
  tvfv(mask) = confil(mask)/sum(confil(mask));
  kgroupv(mask) = k + nshafa;
  itype(nshafa+k) = 4;
  isvf(nshafa+k) = isvl(nshafa) + ifirst(k);
  isvl(nshafa+k) = isvl(nshafa) + ilast(k);
end

rvf    = vertcat(   rvfe,   rvff,   rvfoh,   rvffast,   rvfv);
zvf    = vertcat(   zvfe,   zvff,   zvfoh,   zvffast,   zvfv);
wvf    = vertcat(   wvfe,   wvff,   wvfoh,   wvffast,   wvfv);
hvf    = vertcat(   hvfe,   hvff,   hvfoh,   hvffast,   hvfv);
tvf    = vertcat(   tvfe,   tvff,   tvfoh,   tvffast,   tvfv);
kgroup = vertcat(kgroupe,kgroupf,kgroupoh,kgroupfast,kgroupv);

end

function z=round_(x,n)

if verLessThan('MATLAB','8.4')
  y = 10.^ceil(log10(abs(x))-n);
  z = round(x./y).*y;
else
  z = round(x,n,'significant');
end

end