function [rv,zv,wv,hv,Tivs,Rs,Rv,dims,dimv] = vertices2meq(Conds,doplot)
% [rv,zv,wv,hv,Tvs,Rs,Rv,dims,dimv] = vertices2meq(V,doplot)
% define square vessel filaments from vertices of quadrilaterals.
%
% Conds: Array of quadrilateral conductor element description structures
% Conds(i).Rvertex   R coordinate of vertex [m] (4x1)
% Conds(i).Zvertex   Z coordinate of vertex [m] (4x1)
% Conds(i).Res       Resistance of element [Ohm] [optional - can also specify Resy]
% Conds(i).Resy      Resistivity of element [Ohm*m] [optional - can also specify Res]
%                NB: If both Res and Resy are specified, Res is used.
%
% set doplot = true to plot
%
% Outputs: documented in meqg.m
% 
% Method:
%   Find the longest edge, define the aspect ratio (long height / short height);
%   The number of filaments will be floor(aspectratio).
%   The width of the square filaments are chosen to maintain the total area
%   The filament centers are placed on the line joining the midpoints of the
%   two edges adjacent to the longest one, equally spaced.
%   
%   If total element resistance is given, the resistance of each filament
%   is chosen so that a constant voltage yields the same total current
%   in the filaments as in the original single element.
%  
%   If the resistivity is given, the resistance of each filament is
%   directly calculated as Res=resistivity*l/A
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

if nargin==1, doplot=false; end

nelem = numel(Conds);
% Create final vector/matrices from concatenating individual discretized
% elements in cell arrays
rv = cell( nelem, 1 );
zv = rv;
wv = rv;
hv = rv;
Tivs = rv;
Tvs = rv;
Rv = rv;

for ielem = 1:nelem
  cond = Conds(ielem);
  if any(isfield(cond,{'Rcenter','Zcenter'}))
    warning('Fields ''Rcenter'',''Zcenter'' are no longer used, please remove them to avoid confusion.'); 
  end
  % rr,zz are corners of elements to keep
  assert(numel(cond.Rvertex)==4,'only works for quadrilaterals')
  assert(numel(cond.Zvertex)==4,'only works for quadrilaterals')
  [rr,zz] = order_clockwise(cond.Rvertex,cond.Zvertex);
  [Rcentroid,Zcentroid,Area] = polygon_properties(rr,zz);
  
  if doplot
    plot(rr,zz,'ok'); hold on;
    patch([rr,rr(1)],[zz,zz(1)],'b');
    plot(Rcentroid,Zcentroid,'*k'); axis equal;
  end

  N = 4; wrapN = @(x) (1 + mod(x-1, N)); % aux function for wrapping indices
  % find the longest edge
  Vertices = [rr',zz'];
  [base,ibase] = max(sqrt(sum(diff(Vertices(wrapN(1:N+1),:)).^2,2)));

  % find aspect ratio as if it was a rectangle
  aspect = base.^2/Area;
  
  % number of filaments to use
  nfil = ceil(aspect-100*eps); % round upward to avoid gaps, avoiding roundoff errors
  
  % take vertices of the two edges adjacent to the longest one
  Xshort1 = Vertices(wrapN([ibase-1,ibase]),:);
  Xshort2 = Vertices(wrapN([ibase+1,ibase+2]),:);
  
  % vectors of these same edges
  short1_midpoint = mean(Xshort1);
  short2_midpoint = mean(Xshort2);
  
  fracint = (1:2:(2*nfil))/(2*nfil); % fractional spacing of centroids along chord
  % vector between short edge midpoints, oriented towards center from short1_midpoint
  vv = short2_midpoint - short1_midpoint;
  vv = vv*sign(dot(short1_midpoint-[Rcentroid,Zcentroid],vv));
  
  % place rectangle midpoints
  rv{ielem} = short1_midpoint(1) - fracint'*vv(1);
  zv{ielem} = short1_midpoint(2) - fracint'*vv(2);
  
  % filament width to maintain total surface
  wv{ielem} = repmat(sqrt(Area/nfil),nfil,1);
  hv{ielem} = wv{ielem}; % square filaments
  
  % connections
  T = ones(1,nfil); % connection matrix
  % Ohm's law
  %Yv = 1 ./ Rv; % 1/Resistivity
  %Iv(1) = Yv(1)/sum(Yv(1:n)) Is(1) ,... => Iv = Yvv*T'*inv(T*Yvv*T')*Is
  %                                         Iv = Tvs                 *Is
  % with Tvs=Yvv*T'*inv(T*Yvv*T');
  
  % assuming constant resistivity, Rv~R(radius) so Yv~1/Radius
  Yvv = diag(1./rv{ielem}); % constant scaling does not matter
  myTvs = Yvv*T'/(T*Yvv*T');
  myTsv = myTvs';

  % Resistance per vessel element
  if isfield(cond,'Res') && ~isempty(cond.Res)
    % resistance given
    k = myTsv*diag(1./rv{ielem})*myTvs; % scaling
    Rv{ielem} = cond.Res/k * 1./rv{ielem};
  elseif isfield(cond,'Resy') && ~isempty(cond.Resy)
    % resistivity given: R=resy*L/A;
    Rv{ielem}   = cond.Resy * 2*pi*rv{ielem} ./ (wv{ielem}.*hv{ielem});
  else
    error('must specify either resistance or resistivity')
  end
  Tivs{ielem} = T;
  Tvs{ielem} = myTvs;
  
  if doplot
    for iii = 1:numel(rv{ielem})
      xx = rv{ielem}(iii)+wv{ielem}(iii)*[-1 -1 1 1]/2;
      yy = zv{ielem}(iii)+hv{ielem}(iii)*[-1 1 1 -1]/2;
      patch(xx,yy,'r','FaceAlpha',0.3);
      plot(rv{ielem}(iii),zv{ielem}(iii),'.b')
    end
    axis equal;
    drawnow;
  end
end
% Join as a single vector
rv = vertcat(rv{:});
zv = vertcat(zv{:});
hv = vertcat(hv{:});
wv = vertcat(wv{:});
Rv = vertcat(Rv{:});
Tivs = blkdiag(Tivs{:});
Tvs = blkdiag(Tvs{:});

Rs = diag(Tvs'*diag(Rv)*Tvs);

%% Vessel element names
% Element subdivision per component
dims = cell(nelem,1);
dimv = cell(numel(rv),1);
for idims = 1:numel(dims)
  % string for segment
  segstr = sprintf('Segment%03d',idims);
  dims{idims} = segstr;
  % for all segments of this type, assign vessel label
  ivv = any((Tvs(:,idims)~=0),2); nnv = sum(ivv);
  dimv(ivv) = cellstr(num2str((1:nnv)',[segstr,'_%03d']));
end
  
end

function [rr,zz] = order_clockwise(rin,zin)
% sorts rin,zin vertices clockwise 
rc = mean(rin); zc = mean(zin);
angle = atan2(zin-zc,rin-rc);
% sort by increasing angle
[~,isrt] = sort(angle,'ascend');
rr = rin(isrt); zz = zin(isrt);

if iscolumn(rr), rr=rr'; end
if iscolumn(zz), zz=zz'; end
end

function [rc,zc,A] = polygon_properties(rr,zz)
% calculate centroid of a polygon
% Needs rr,zz to be ordered along the polygon contour

r2 = [rr(2:end),rr(1)];
z2 = [zz(2:end),zz(1)];

% signed area using shoelace formula
Ai = rr.*z2 - r2.*zz;
A = 0.5*sum(Ai);

% centroid
rc = 1/(6*A) * sum( (rr + r2) .* Ai ); 
zc = 1/(6*A) * sum( (zz + z2) .* Ai ); 

A = abs(A); % output absolute value
end
