classdef solveF_test < meq_test
  % Tests for general F(x)=0 solver
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  properties
    tol = 1e-8;
  end

  properties(TestParameter)
    algoF = {'jfnk','newton'} % broyden is tested separately
    algoGMRES = {'direct_inversion', 'matlab_gmres', 'sim', 'giv', 'aya'};
    userowmask = {false,true};
    precon = {'off','one_half', 'one', 'jacobi', 'inverse', 'ilu','one_handle'};
    % Broyden parameters
    use_inverse={false,true};
    large_system={false,true};
    group_size={1,2,3,4};
  end

  methods(Test, TestTags={'Unit'})
    function test_quadratic(testCase,algoF,algoGMRES,userowmask)
      testCase.assumeFalse(strcmp(algoF, 'jfnk') && strcmp(algoGMRES, 'direct_inversion'), ...
        sprintf('invalid combination algoF=%s algoGMRES=%s',algoF,algoGMRES))

      % Test a quadratic function with newton method
      [F,x0]=solveF_test.function_factory('quadratic');

      P = struct('robust_on',1, 'algoF', algoF,'algoGMRES',algoGMRES,...
        'mkryl', 200, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', 150, 'tolF', testCase.tol, 'userowmask', userowmask);

      x = solveF(F,x0,P);
      opts.dojacx=false;
      % Check that max|F(x)|<tol
      testCase.verifyLessThan(max(abs(F(x,opts))),testCase.tol,...
        'Norm exceeds error tolerance');
    end

    function test_quadratic_broyden(testCase,use_inverse,large_system,group_size,userowmask)
      testCase.assumeFalse(~use_inverse && large_system, ...
        sprintf('invalid combination use_inverse=%b large_system=%s',mat2str(use_inverse),mat2str(large_system)))
      % Test a quadratic function with Broyden method
      [F,x0] = solveF_test.function_factory('quadratic');
      P = struct('robust_on',1,'algoF','broyden','anajac',true,...
        'mkryl', 200, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', 150, 'tolF', testCase.tol, 'userowmask', userowmask, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',group_size);

      x = solveF(F,x0,P);
      opts.dojacx=false;
      % Check that max|F(x)|<tol
      testCase.verifyLessThan(max(abs(F(x,opts))),testCase.tol,...
        'Norm exceeds error tolerance');
    end

    function test_linear(testCase,algoF,algoGMRES,precon,userowmask)
      testCase.assumeFalse(strcmp(algoF, 'jfnk') && strcmp(algoGMRES, 'direct_inversion'), ...
        sprintf('invalid combination algoF=%s algoGMRES=%s',algoF,algoGMRES))
      testCase.assumeFalse(~strcmp(algoF, 'newton') ...
        && strcmp(precon, 'ilu'),sprintf('invalid combination algo=%s-%s and precon=%s',algoF,algoGMRES,precon));
      % Test a linear function with newton method
      [F,x0,opt] = solveF_test.function_factory('linear');

      P = struct('robust_on',1, 'algoF', algoF,'algoGMRES',algoGMRES,...
        'mkryl', 200, 'relax', 0, 'nrestarts',0,...
        'debug', 2, 'kmax', 150,...
        'epsilon_d',1,... % linear system with A~1 b~1 means large epsilon_d is more accurate
        'tolF', testCase.tol,'userowmask',userowmask);
      P = testCase.add_precon_option(P, precon, opt.A);

      [x,solverinfo] = solveF(F,x0,P);
      opts.dojacx=false;
      % Check that max|F(x)|<tol
      testCase.verifyLessThan(max(abs(F(x,opts))),testCase.tol,...
        'Norm exceeds error tolerance');

      testCase.verifyLessThan(solverinfo.niter,5,'Solver took too many iterations?')
    end

    function test_linear_broyden(testCase,use_inverse,large_system,group_size,userowmask)
      testCase.assumeFalse(~use_inverse && large_system, ...
        sprintf('invalid combination use_inverse=%b large_system=%s',mat2str(use_inverse),mat2str(large_system)))
      % Test a linear function with Broyden method
      [F,x0]=solveF_test.function_factory('linear');
      P = struct('robust_on',1,'algoF','broyden','anajac',true,...
        'mkryl', 200, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', 150, 'tolF', testCase.tol, 'userowmask', userowmask, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',group_size);

      x = solveF(F,x0,P);
      opts.dojacx=false;
      % Check that max|F(x)|<tol
      testCase.verifyLessThan(max(abs(F(x,opts))),testCase.tol,...
        'Norm exceeds error tolerance');
    end

    function test_rosenbrock_deriv(testCase,use_inverse,large_system,group_size)
      testCase.assumeFalse(~use_inverse && large_system, ...
        sprintf('invalid combination use_inverse=%b large_system=%s',mat2str(use_inverse),mat2str(large_system)))
      % Test a Rosenbrock function with Broyden method
      [F,x0]=solveF_test.function_factory('rosenbrock');
      P = struct('robust_on',1,'algoF','broyden','anajac',true,...
        'mkryl', 200, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', 150, 'tolF', testCase.tol, 'userowmask', false, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',group_size);
      x = solveF(F,x0,P);
      opts.dojacx=false;
      % Check that max|F(x)|<tol
      testCase.verifyLessThan(max(abs(F(x,opts))),testCase.tol,...
        'Norm exceeds error tolerance');
    end

    function test_NaN(testCase,use_inverse)
      % Test that the solver fails if function returns NaN
      [F,x0]=solveF_test.function_factory('nan');
      P = struct('robust_on',1,'algoF','broyden','anajac',true,...
        'mkryl', 200, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', 150, 'tolF', testCase.tol, 'userowmask', false, ...
        'use_inverse',use_inverse,'large_system',false, ...
        'is_factored',false,'group_size',4);
      [~,solverinfo] = solveF(F,x0,P);
      testCase.verifyFalse(solverinfo.isconverged,'Solver should fail');
    end

    function test_broyden_large_system(testCase,group_size,userowmask)
      % Test that large system=true or =false gives the same output after
      % n iteration (n small)
      % Tested on a quadratic function
      [F,x0]=solveF_test.function_factory('quadratic');
      n=5;
      P = struct('robust_on',1,'algoF','broyden','anajac',true,...
        'mkryl', group_size, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', n, 'tolF', testCase.tol, 'userowmask', userowmask, ...
        'use_inverse',true,'large_system',true, ...
        'is_factored',false,'group_size',group_size);

      x1 = solveF(F,x0,P);

      P = struct('robust_on',1,'algoF','broyden','anajac',true,...
        'mkryl', group_size, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', n, 'tolF', testCase.tol, 'userowmask', userowmask, ...
        'use_inverse',true,'large_system',false, ...
        'is_factored',false,'group_size',group_size);

      x2 = solveF(F,x0,P);

      testCase.verifyEqual(x1,x2,'RelTol',1e-15,'Different solutions found')

    end

    function test_broyden_inverse(testCase,group_size)
      % Test that inverse=true or =false gives the same output after
      % n iteration (n small)
      [F,x0]=solveF_test.function_factory('circle_parabol');

      n=3;
      % ratio_gb must be 1 (only implemented in the inverse version)
      P = struct('algoF','broyden','anajac',true,...
        'mkryl', group_size, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', n, 'tolF', testCase.tol, 'userowmask', false, ...
        'use_inverse',true,'large_system',false, ...
        'is_factored',false,'group_size',group_size,'ratio_gb',1);

      x1 = solveF(F,x0,P);

      % ratio_gb must be 1 (only implemented in the inverse version)
      P = struct('algoGMRES','direct_inversion','algoF','broyden','anajac',true,...
        'mkryl', group_size, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', n, 'tolF', testCase.tol, 'userowmask', false, ...
        'use_inverse',false,'large_system',false, ...
        'is_factored',false,'group_size',group_size,'ratio_gb',1);

      x2 = solveF(F,x0,P);
      % Set tol = 1e-10 because of numerical approximations (inversion of matrices)
      testCase.verifyEqual(x1,x2,'RelTol',1e-10,'Different solutions found')

    end

    function test_rowmask(testCase,use_inverse,group_size,large_system)
      % Test that userowmask=true or =false gives the same output after
      % n iteration (n small)
      % Tested on a linear function for which userowmask is used
      k=5;
      [F,x0]=solveF_test.function_factory('linear');
      % Broyden method uses P.kmax*P.mkryl as total number of iterations
      P = struct('algoGMRES','direct_inversion','algoF','broyden','anajac',true,...
        'mkryl', 1, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', k*group_size, 'tolF', testCase.tol, 'userowmask', true, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',group_size);

      x1 = solveF(F,x0,P);

      P = struct('algoGMRES','direct_inversion','algoF','broyden','anajac',true,...
        'mkryl', 1, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', k*group_size, 'tolF', testCase.tol, 'userowmask', false, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',group_size);

      x2 = solveF(F,x0,P);

      % Set tol = 1e-12 because of numerical approximations
      testCase.verifyEqual(x1,x2,'RelTol',1e-12,'Different solutions found')

    end

    function test_inner_ortho_broyden(testCase,use_inverse,group_size,large_system,userowmask)
      % Orthogonalization process is not defined for userowmask=true and
      % use_inverse=false
      testCase.assumeFalse(~use_inverse && userowmask, ...
        sprintf('invalid combination use_inverse=%s userowmask=%s',mat2str(use_inverse),mat2str(userowmask)))
      testCase.assumeFalse(~use_inverse && large_system, ...
        sprintf('invalid combination use_inverse=%s large_system=%s',mat2str(use_inverse),mat2str(large_system)))
      % Check that the error after K iteration with group_size 1 is equal than the one after 1 iteration
      % with group size K
      k=group_size;
      [F,x0]=solveF_test.function_factory('quadratic2');
      % Broyden method uses P.kmax*P.mkryl as total number of iterations
      % 'ratio_gb' should be 1
      P = struct('algoGMRES','direct_inversion','algoF','broyden','anajac',true,...
        'mkryl', 1, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', k, 'tolF', testCase.tol, 'userowmask', userowmask, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',1,'ratio_gb',1);

      x1 = solveF(F,x0,P);

      % 'ratio_gb' should be 1
      P = struct('algoGMRES','direct_inversion','algoF','broyden','anajac',true,...
        'mkryl', 1, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', group_size, 'tolF', testCase.tol, 'userowmask', userowmask, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',group_size,'ratio_gb',1);

      x2 = solveF(F,x0,P);

      testCase.verifyEqual(x1,x2,'AbsTol',1e-12*norm(x1),'Different solutions found')

    end

    function test_convergence_jac(testCase,use_inverse,large_system,group_size)
      testCase.assumeFalse(~use_inverse && large_system, ...
        sprintf('invalid combination use_inverse=%s large_system=%s',mat2str(use_inverse),mat2str(large_system)))
      % Check that the approximation of the hessian is correct on
      % descending directions (i.e., -H*dx=-H_approx*dx)
      % Other directions are not updated and therefore H*v might not
      % be equal to H_approx*v
      [F,x0,opt]=solveF_test.function_factory('quadratic');
      P = struct('algoGMRES','direct_inversion','algoF','broyden','anajac',true,...
        'mkryl', 1, 'Prec', [], 'relax', 0, 'debug', 2,...
        'kmax', 150, 'tolF', 1e-14, 'userowmask', false, ...
        'use_inverse',use_inverse,'large_system',large_system, ...
        'is_factored',false,'group_size',group_size);

      [~,solverinfo] = solveF(F,x0,P);
      % Moving only in this direction (so other direction of the Jacobian are not updated)
      dx=opt.xSol-x0;
      if use_inverse
        testCase.verifyLessThan(norm(solverinfo.prev_H.H\dx),1e-6*norm(dx),'Low rank approximation did not converge')
      else
        testCase.verifyLessThan(norm(solverinfo.prev_H.H*dx),1e-6*norm(dx),'Low rank approximation did not converge')
      end

    end

    function test_reduce_step(testCase)
      % Test that method to estimate jacobian in direction dx will reduce
      % the step size until it manages to compute F successfully.

      algoF_ = 'jfnk';    % Newton does not use this feature
      algoGMRES_ = 'aya'; % GMRES variant is unimportant

      [F,x0,opt]=solveF_test.function_factory('epsilon_step');

      % With this setup the initial residual is b=-F(x0)=-xSol/epsilon_d^2=-0.4
      % The first krylov vector is V(:,1)=b/norm(b)=-1
      % We evaluate the jacobian at x0=0 with the direction V(:,1)
      %   We start with dxstep=epsilon_d*V(:,1)=-epsilon_d
      %   We evaluate F(x0+dxstep)=F(-epsilon_d)=NaN so we have to reduce the step size
      %   We set dxstep=epsilon_d/10*V(:,1)=-epsilon_d
      %   Now the evaluation of F succeeds and the algorithm can continue

      P = struct('robust_on',1, 'algoF', algoF_,'algoGMRES',algoGMRES_,...
        'mkryl', 200, 'Prec', [], 'relax', 0, 'debug', 0,...
        'kmax', 150, 'tolF', testCase.tol, 'epsilon_d', opt.epsilon_d);

      x = solveF(F,x0,P);
      opts.dojacx=false;
      testCase.verifyLessThan(max(abs(F(x,opts))),testCase.tol,...
        'Norm exceeds error tolerance');
    end
  end

  methods(Static)
    function [F,x0,opt]=function_factory(name)
      % Generate a function to test solveF
      switch name
        case 'quadratic'
          % Quadratic function of 2 variables
          xSol = [1 -1]'; x0=[0 0]';
          % anonymous function for F, extended to return all output arguments
          % First input is F itself, second is the jacobian, third is rowmask giving diagonal elements of the jacobian of F
          jac=@(x) 2*diag(x-xSol);
          F = @(x,opts) solveF_test.extendF_jac((x-xSol).^2,jac(x),2*(x-xSol),opts);
          opt.jac=jac;
          opt.xSol=xSol;
        case 'linear'
          % Linear function
          n = 100;
          A = rand(n); b=rand(n,1);
          x0 = rand(n,1);
          % Introduce diagonal block
          A(1:10,:) = [eye(10),zeros(10,n-10)];
          % Jacobian (A) diagonal elements when they are the only ones in their row
          Adiag = [ones(10,1);zeros(n-10,1)];
          opt.A=A;
          F = @(x,opts) solveF_test.extendF_jac(A*x-b,A,Adiag,opts);
        case 'rosenbrock'
          % Find the minimum of the Rosenbrock function (or equivalently solve
          % Grad F = 0):
          %     F(x,y)=(a-x)^2+b(y-x^2)^2
          % The solution is (a,a^2).
          x0=[1.2;1.05];
          a=1;
          b=100;
          h=@(x)[2-4*b*x(2)+12*b*x(1).^2, -4*b*x(1);-4*b*x(1),2*b];
          F = @(x,opts) solveF_test.extendF_jac( [2*(x(1)-a)-4*b*x(1)*(x(2)-x(1).^2);2*b*(x(2)-x(1).^2)], ...
            h(x),[],opts);
          opt.jac=h;
        case 'circle_parabol'
          % Find the solution of the intersection between a
          % circle and a parabola
          x0=[1 0]';
          f = @(x) [x(1)^2+x(2)^2-1; x(2)-x(1)^2];
          h = @(x) 2*[x(1) x(2);-x(1) 0.5];
          F = @(x,opts) solveF_test.extendF_jac(f(x),h(x),[],opts);
        case 'quadratic2'
          % A quadratic function that takes into account rowmask (diagonal entries in Jacobian)
          % Function is
          % f(x)=A(x+(B*x).^2) for all entries k>m, f(x)=A*x for
          % the first m-1 entries
          n = 100;
          m = 10;
          A = eye(n);
          x0 = randn(n,1);
          % Introduce diagonal block
          A(1:m,:) = [eye(m),zeros(m,n-m)];
          % Jacobian (A) diagonal elements when they are the only ones in their row
          Adiag = [ones(m-1,1);zeros(n-m+1,1)];
          B=2*diag(ones(n,1))-diag(ones(n-1,1),-1)-diag(ones(n-1,1),1);
          % anonymous function for F, extended to return all output arguments.
          jac=@(x) diag(Adiag)+2*(B*[zeros(m,1);x(m+1:end)]).*B;
          F = @(x,opts) solveF_test.extendF_jac(A*([x(1:m,1);zeros(n-m,1)]+(B*[zeros(m,1);x(m+1:end)]).^2),jac(x),Adiag,opts);
          opt.jac=jac;
        case 'epsilon_step'
          epsilon_d = 0.5;
          xSol = 0.1; x0 = 0;
          opt.epsilon_d=epsilon_d;
          F = @(x,opts) solveF_test.extendF((x-xSol)/(x-epsilon_d)/(x+epsilon_d));
        case 'nan'
          % function that returns nan values
          n=10;
          a=zeros(n,1);
          a(:)=NaN;
          b=zeros(n,n);
          b(:)=NaN;
          x0 = randn(n,1);
          f=@(x)a;
          h=@(x)b;
          F = @(x,opts)solveF_test.extendF_jac(f(x),h(x),[],opts);
        otherwise
          error(sprinf('Unknow function %s',name))
      end
    end

    function [res, LY, Jx, Ju, Jxdot, rowmask] = extendF(res,rowmask)
      % Add the required number of output arguments by solveF
      LY=[];
      Jx=[];
      Ju=[];
      Jxdot=[];
      if nargin<2
        rowmask=zeros(numel(res),1);
      end
    end

    function [res, LY, Jx, Ju, Jxdot, rowmask] = extendF_jac(res,jac,rowmask,opts)
      % Add the required number of output arguments by solveF
      LY=[];
      if opts.dojacx
        Jx=jac;
      else
        Jx=[];
      end
      Ju=[];
      Jxdot=[];
      if isempty(rowmask)
        rowmask=zeros(numel(res),1);
      end
    end
    
    function P = add_precon_option(P, precon_option, A)
      P.prectype = 'user';
      P.precupdate = 'once';
      switch precon_option
        case 'off'
          P.Prec = [];
        case 'one_half'
          P.Prec = 0.5;
        case 'one'
          P.Prec = 1;
        case 'jacobi'
          P.Prec = spdiags(1./diag(A), 0, size(A, 1), size(A, 2));
        case 'inverse'
          P.Prec = inv(A);
        case 'ilu'
          P.Prec = [];
          P.prectype = 'ilu';
          P.ilu_algo = 'crout';
          P.ilu_droptol = 1e-3;
        case 'one_handle'
          P.Prec = @(v) v;
      end
      P.prec = preconditioner_init(P,A,size(A,1));
    end
  end
end
