classdef rzp_legacy_test < meq_test
  %% Test class for rzpl
  % Test the old (in the rzip repository of codes) versus new rzip model
  % matrices.
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties
    verbosity = 0; % Make plot 1 to make the plots

    tok       = 'tcv';
    selu      = 'v';
    nu        = 256;
    rtol      = 1e-3;    %tolerance on the elements relative difference
    atol      = 1.5e-8;  %tolerance on the elements absolute difference

    P_args = {'selx','X','iterq',20,'cde','OhmTor_rigid_0D','agcon',{{'bp','li'}},'lin',true};

    old_rzp, new_rzp;

    %number of eigen to compare
    n_eig = 4;
  end
  
  properties (ClassSetupParameter)
    shot = struct('pos_triang', 61400, 'neg_triang', 69393);
    t    = struct('pos_triang', 1    , 'neg_triang', 0.5  );
  end
  
  methods (TestClassSetup, ParameterCombination = 'sequential')
    function setup_old_rzp_path(testCase)
      curpath = path;
      testCase.addTeardown(@() path(curpath));
      
      % Call to the repository where the old rzip code is placed.
      old_rzp_dir = '/home/codes/rzip';
      assert(logical(exist(old_rzp_dir,'dir')),...
        ['Path ',old_rzp_dir,' doesn''t exist']);
      addpath(old_rzp_dir);
      add_rzip_paths
    end

    function setup_output(testCase,shot,t)
      %% Setup the output for the comparison between the old and new rzip

      L  = rzp (testCase.tok,shot,t,'selu',testCase.selu,'nu',testCase.nu,testCase.P_args{:});
      LX = rzpx(testCase.tok,t,L);
      
      [sys_old,sysinfo] = make_rzip_model(testCase.selu,testCase.nu,...
        'plasmasource',{L,LX},...
        'shot',LX.shot,...
        'time',LX.t,...
        'full_output',true);
      
      LX.Rp = sysinfo.Rp;
      LX.Ia(contains(L.G.dima,'G_001')) = 0; %no current in the coil G
      L.Re(startsWith(L.G.dima,'G_001')) = 0.032; % change to hard-coded G coil resistance
      L = rzpl(L,LX);
      
      [sys_new,ind_new] = fgess(L);
      
      if testCase.verbosity
        meqplott(L,LX)
      end

      %% Set up the permutations used to multiply sysinfo
      % The order of the active coils between the old and new rzip is
      % different. The matrices ipermute, II and III are used to multiply
      % the matrices of the old rzip to recover the same ordering.
      iEn  = startsWith( L.G.dima,'E'  ); iEo  = startsWith( sysinfo.G.dima,'E'  );
      iFn  = startsWith( L.G.dima,'F'  ); iFo  = startsWith( sysinfo.G.dima,'F'  );
      iGn  = startsWith( L.G.dima,'G'  ); iGo  = startsWith( sysinfo.G.dima,'G'  );
      iOHn = startsWith( L.G.dima,'OH' ); iOHo = startsWith( sysinfo.G.dima,'OH' );
      
      ipermute            = zeros( L.G.na  );
      ipermute(iEn,iEo)   = eye( sum(iEn)  );
      ipermute(iFn,iFo)   = eye( sum(iFn)  );
      ipermute(iGn,iGo)   = eye( sum(iGn)  );
      ipermute(iOHn,iOHo) = eye( sum(iOHn) );
      
      II       = blkdiag(ipermute,eye(L.G.nu+L.np));
      III      = blkdiag(ipermute,eye(L.G.nu)     );
      
      %% Extract the quantities for the new rzip
      %Compute and store quantities useful for the comparison of the old

      ixy = L.ind.ixGS;
      ixg = L.ind.ixg;  % Index of ag variables
      iry = L.ind.irGS;
      irC = L.ind.irC;  % Index of ag constraint residuals
      irp = L.ind.irp;  % Index of CDE residual
      irS = [iry,irC];  % Index of static residuals
      ixIp = ixg(1);    % Index of Ip variable (part of ag)
      ixe = [L.ind.ixa,L.ind.ixu];
      ire = [L.ind.ira,L.ind.iru];
      kLe = [ind_new.iSIa,ind_new.iSIu];
      kLp =  ind_new.iSIp;
      iuC = L.ind.iuC;

      % Jacobian
      [~,Jx,Ju] = rzpFlin(L,LX);

      % Static part of the jacobian
      L.lin.dFdIy = 1./L.resscal(irS).*Jx(irS,iry)./L.xscal(ixy).';
      L.lin.dFdag = 1./L.resscal(irS).*Jx(irS,ixg)./L.xscal(ixg).';
      L.lin.dFdIe = 1./L.resscal(irS).*Jx(irS,ixe)./L.xscal(ixe).';
      L.lin.dFdCo = 1./L.resscal(irS).*Ju(irS,iuC);
      L.lin.dFdIp = 1./L.resscal(irS).*Jx(irS,ixIp)./L.xscal(ixIp).';

      % Linearization
      L.lin.dagdIe = L.xscal(ixg).*L.lin.dxdxL(ixg,kLe)./L.xscal(ixe).';
      L.lin.dagdCo = L.xscal(ixg).*L.lin.dxdu (ixg,iuC);
      L.lin.dIydIe = L.xscal(ixy).*L.lin.dxdxL(ixy,kLe)./L.xscal(ixe).';
      L.lin.dIydCo = L.xscal(ixy).*L.lin.dxdu (ixy,iuC);

      L.lin.Mpe = (L.Mey*LX.Iy(:))'./LX.Ip;

      L.lin.Xee =  1./L.resscal(ire).*L.lin.S   (kLe,kLe)./L.xscal(ixe).' - L.Mee;
      L.lin.Xeo = -1./L.resscal(ire).*L.lin.Udot(kLe,iuC);
      L.lin.Xpe =  1./L.resscal(irp).*L.lin.S   (kLp,kLe)./L.xscal(ixe).' - L.lin.Mpe;
      L.lin.Xpo = -1./L.resscal(irp).*L.lin.Udot(kLp,iuC);

      %rzip
      testCase.new_rzp.L      = L;
      testCase.new_rzp.ss     = sys_new;
      testCase.new_rzp.ss_ind = ind_new;
      testCase.new_rzp.Iy     = LX.Iy; %store Iy in the linearisation points
      
      %Esp dxdt = Asp x + Vsp u
      testCase.new_rzp.Esp = 1./L.resscal([ire,irp]).*L.lin.S([kLe,kLp],[kLe,kLp])./L.xscal([ixe,ixIp]).';
      testCase.new_rzp.Asp = 1./L.resscal([ire,irp]).*L.lin.K([kLe,kLp],[kLe,kLp])./L.xscal([ixe,ixIp]).';
      testCase.new_rzp.Vsp = 1./L.resscal([ire,irp]).*L.lin.U([kLe,kLp],1:L.G.na);

      %% Compute the quantities useful for comparison form the old rzip
      IP0 = sysinfo.pcurrents.IP0;
      %Static matrix calculation
      testCase.old_rzp.drdrMxa = sysinfo.G.dmdrdrxa(:,1:19)*ipermute';
      testCase.old_rzp.drdzMxa = sysinfo.G.dmdrdzxa(:,1:19)*ipermute';
      testCase.old_rzp.dzdzMxa = sysinfo.G.dmdzdzxa(:,1:19)*ipermute';
      testCase.old_rzp.drdrMxu = sysinfo.G.dmdrdrxv;
      testCase.old_rzp.drdzMxu = sysinfo.G.dmdrdzxv;
      testCase.old_rzp.dzdzMxu = sysinfo.G.dmdzdzxv;
      testCase.old_rzp.Mee = III*sysinfo.matrices.M11*III';
      testCase.old_rzp.Re  = [ipermute*sysinfo.G.Ra(1:L.G.na); sysinfo.G.Rv];
      testCase.old_rzp.Mey = [sysinfo.G.Mxa(L.lxy(:),1:L.G.na)*ipermute' sysinfo.G.Mxv(L.lxy(:),:)]';
      testCase.old_rzp.rry = sysinfo.G.rx(L.lxy(:));
      testCase.old_rzp.zzy = sysinfo.G.zx(L.lxy(:));
      testCase.old_rzp.Re(contains(L.G.dima,'G')) = 0.032; %old rzp hardcoded
      
      %Plasma parameters calculation
      testCase.old_rzp.Ia  = ipermute*sysinfo.pcurrents.Ia;
      testCase.old_rzp.Iu  = sysinfo.pcurrents.Iu;
      testCase.old_rzp.Ip  = IP0;
      testCase.old_rzp.Iy  = sysinfo.pcurrents.Ipx0(L.lxy(:));
      testCase.old_rzp.rIp = sysinfo.pcurrents.R0*IP0;
      testCase.old_rzp.zIp = sysinfo.pcurrents.z0*IP0;
      testCase.old_rzp.bp  = sysinfo.pcurrents.betap;
      
      % Static part of the jacobian
      %dFdIy
      testCase.old_rzp.dFdIy = [eye(L.ny);...
                                zeros(L.nC,L.ny)];
      %dFdag
      dFydIp = -sysinfo.pcurrents.Ipx0(L.lxy(:))'./IP0;
      dFydRc = -sysinfo.pcurrents.dIpdR0(L.lxy(:))';
      dFydZc = -sysinfo.pcurrents.dIpdz0(L.lxy(:))';
      dFrdIp = sysinfo.matrices.M34*IP0;
      dFrdRc = sysinfo.matrices.M33*IP0^2;
      dFrdZc = sysinfo.matrices.M32*IP0^2;
      dFzdIp = sysinfo.matrices.M24*IP0;
      dFzdRc = sysinfo.matrices.M23*IP0^2;
      dFzdZc = sysinfo.matrices.M22*IP0^2; 
      testCase.old_rzp.dFdag = [dFydIp dFydRc dFydZc;...
                                dFrdIp dFrdRc dFrdZc;...
                                dFzdIp dFzdRc dFzdZc];
      %dFdIe
      testCase.old_rzp.dFdIe = [zeros(L.ny,L.ne); ...
                                sysinfo.matrices.M31*III';...
                                sysinfo.matrices.M21*III'] * IP0;
      %dFdCo
      dFydCo = zeros(L.ny,L.nC);
      dFrdCo = mu0*IP0^2*[1/2, 1/4];
      dFzdCo = zeros(1,L.nC);
      testCase.old_rzp.dFdCo = [dFydCo;...
                                dFrdCo;...
                                dFzdCo];
      
      % Linearization of the static part of the jacobian
      agscal = [1;IP0;IP0];
      testCase.old_rzp.dagdIe = [zeros(1,L.ne); ...
                                 flipud(-(sysinfo.matrices.M_22)\(sysinfo.matrices.M_21(:,1:L.ne)*III'))]./agscal;
      testCase.old_rzp.dagdCo = [0, 0;...
                                 -testCase.old_rzp.dFdag(irC,2:3)\testCase.old_rzp.dFdCo(irC,:)];
      testCase.old_rzp.dIydIe = -testCase.old_rzp.dFdag(iry,:)*testCase.old_rzp.dagdIe;
      testCase.old_rzp.dIydCo = -testCase.old_rzp.dFdag(iry,:)*testCase.old_rzp.dagdCo;
      
      %Xee, Xeo (,Xpe, Xpo)
      testCase.old_rzp.Xee = [zeros(L.ne,1),fliplr(III*sysinfo.matrices.M_12(1:L.ne,:))] * (testCase.old_rzp.dagdIe.*agscal);
      testCase.old_rzp.Xeo = [zeros(L.ne,1),fliplr(III*sysinfo.matrices.M_12(1:L.ne,:))] * (testCase.old_rzp.dagdCo.*agscal);
      testCase.old_rzp.Xpe = [0                   , sysinfo.matrices.M43, sysinfo.matrices.M42]*(testCase.old_rzp.dagdIe.*agscal);
      testCase.old_rzp.Xpo = [sysinfo.matrices.M44, sysinfo.matrices.M43, sysinfo.matrices.M42]*(testCase.old_rzp.dagdCo.*agscal);
      
      %Esp dxdt = Asp x + Vsp u
      testCase.old_rzp.Asp = II*sysinfo.Asp*II';
      testCase.old_rzp.Esp = II*sysinfo.Esp*II';
      testCase.old_rzp.Vsp = II*sysinfo.Vsp*ipermute';
      
      %A and B
      testCase.old_rzp.A = II*sys_old.A*II';
      testCase.old_rzp.B = II*sys_old.B*ipermute';
    end
  end
  
  methods (Test, TestTags = {'rzp'})
    function rzp_G_entries(testCase)
      %% Test the static matrices and the grid points
      tol = {'drdrMxa','drdrMxu','drdzMxa','drdzMxu','dzdzMxa','dzdzMxu'};
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(testCase.new_rzp.L.G.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end
      
      tol = {'Mee','Re','Mey'};
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(testCase.new_rzp.L.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end
      
      tol = {'rry','zzy'};
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(testCase.new_rzp.L.(tol{ii})(:),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end
    end
    
    function rzp_linearisation_point(testCase)
      %% Test the linearisation equilibrium points
      L_ = testCase.new_rzp.L;
      L_.lin.Iy = testCase.new_rzp.Iy(:)';
      
      tol = {'Ia','Iu','Ip','Iy','rIp','zIp','bp'};
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(L_.lin.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end
    end
    
    function rzp_linear_force(testCase)
      %% Test the linearisation of the force operator
      tol = {'dFdIy','dFdag','dFdIe','dFdCo'};
      % dFdIp not needed anymore as it is now part of dFdag
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(testCase.new_rzp.L.lin.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end
    end
    
    function rzp_singular_perturbation(testCase)
      %% Test singular perturbation
      tol = {'dagdIe','dagdCo','dIydIe','dIydCo'};
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(testCase.new_rzp.L.lin.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end

      tol = {'Xee','Xeo'};
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(testCase.new_rzp.L.lin.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end
      
      if testCase.new_rzp.L.icde && strcmp(testCase.new_rzp.L.P.cde,'OhmTor_rigid_0D')
        tol = {'Xpe','Xpo'};
        for ii=1:numel(tol)
          st = sprintf('%s new vs old tolerance exceeded', tol{ii});
          testCase.verifyEqual(testCase.new_rzp.L.lin.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
        end
      end
    end
    
    function rzp_ss_matrix(testCase)
      %% Test ss matrices  
      %Compare Esp dxdt = Asp x + Vsp u 
      tol = {'Asp','Esp','Vsp'};
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(testCase.new_rzp.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',testCase.atol,'RelTol',testCase.rtol,st)
      end 
      
      % Compare A and B matrices
      L = testCase.new_rzp.L;
      ix = [L.ind.ixa,L.ind.ixu,L.ind.ixg(1)];
      iS = [testCase.new_rzp.ss_ind.iSIa,testCase.new_rzp.ss_ind.iSIu,testCase.new_rzp.ss_ind.iSIp];
      new.A = L.xscal(ix).*testCase.new_rzp.ss.A(iS,iS)./L.xscal(ix)';
      new.B = L.xscal(ix).*testCase.new_rzp.ss.B(iS,testCase.new_rzp.ss_ind.iUVa); %in the old rzip, only the input voltages Va were considered
      tol = {'A','B'};
      % The matrix Esp is ill conditioned, therefore the absolute tolerance
      % for this test has been increased to 1.5e-6
      for ii=1:numel(tol)
        st = sprintf('%s new vs old tolerance exceeded', tol{ii});
        testCase.verifyEqual(new.(tol{ii}),testCase.old_rzp.(tol{ii}),'AbsTol',1.5e-6,'RelTol',testCase.rtol,st)
      end
      
      %Make plots
      if testCase.verbosity
        clf;
        subplot(231)
        imagesc(log(abs(Asp)))
        subplot(232)
        imagesc(log(abs(II*sysinfo.Asp*II')))
        subplot(233)
        imagesc(log(abs(II*sysinfo.Asp*II' + Asp))); colorbar;

        subplot(234)
        imagesc(log(abs(Esp)))
        subplot(235)
        imagesc(log(abs(II*sysinfo.Esp*II')))
        subplot(236)
        imagesc(log(abs(II*sysinfo.Esp*II'+ Esp))); colorbar
        
        subplot(234)
        imagesc(log(abs(Vsp)))
        subplot(235)
        imagesc(log(abs(II*sysinfo.Vsp)))
        subplot(236)
        imagesc(log(abs(II*sysinfo.Vsp+ Vsp))); colorbar
      end
    end
    
    function rzp_eigenvalues(testCase)
      %% Test eigenvalues
      %Calculate the eigenvalues of matrix Esp and A
      eigen_Esp = [esort(eig( testCase.new_rzp.Esp  )), esort(eig( testCase.old_rzp.Esp ))];
      eigen_A   = [esort(eig( testCase.new_rzp.ss.A )), esort(eig(  testCase.old_rzp.A  ))];
      
      testCase.verifyEqual( eigen_Esp(1:testCase.n_eig,2), eigen_Esp(1:testCase.n_eig,1), 'RelTol', testCase.rtol, 'AbsTol', testCase.atol, sprintf('Eigen of Esp new vs old rel tolerance exceeded'));
      testCase.verifyEqual( eigen_A(1:testCase.n_eig,2)  , eigen_A(1:testCase.n_eig,1)  , 'RelTol', testCase.rtol, 'AbsTol', testCase.atol, sprintf('Eigen of A new vs old rel tolerance exceeded')  );
    end

  end

end
