classdef (SharedTestFixtures={mexm_fixture}) rtci_robust_test < meq_test
  % Tests for RTCI (real-time contour integrals)
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    verbosity = 0;
  end

  properties (TestParameter)
    method = {'rtcimex','rtcimexm','rtcics'};
  end

  methods(Test, TestTags = {'Unit'})
    function test_rtci_saddle(testCase,method)
      % This test checks behaviour of rtci around saddle points

      switch method
        case 'rtcics'
          PP = {'icsint',true,'ilim',3};
        otherwise
          PP = {'icsint',false,'ilim',1};
      end

      L = fbt('ana',1,[],'iterq',20,PP{:});

      % Custom map with saddle point at rX,zX
      rX = 1;
      zX = 0;
      Fx = (L.rrx-rX).^2 - (L.zzx-zX).^2;
      Opy = reshape(int8(Fx(L.lxy)>0 & L.rrx(L.lxy)>1),L.nzy,L.nry); % Plasma on the right hand side only

      % Setup contours, r=rA-a*cos(oq), z=zA+a*sin(oq)
      rA = 2;
      zA = 0.1;
      FA = 1; % Indicative value of flux "on-axis"
      Opo = int8(1); % Identify origin as plasma domain (-> contours not gaps)
      oq = pi/12*(-3:3).'/3; % Narrow range around X-point direction
      crq = -cos(oq);
      czq =  sin(oq);
      cdrq = crq*L.idrx;
      cdzq = czq*L.idzx;
      F = 0; % Flux level to find (separatrix)

      aq0 = 1.2*ones(size(oq)); % Initial guess, outside of plasma domain, past X-point

      switch method
        case 'rtcics'
          [aq,s] = rtcics(aq0,Fx,0,Opy,rA,zA,Opo,crq,czq,L);
        case {'rtcimex','rtcimexm'}
          ero=(rA-L.rx(1))*L.idrx;
          ezo=(zA-L.zx(1))*L.idzx;
          aq = aq0;
          s = false;
          for ki = 1:L.P.iterq
            [aq,daq] = feval(method,aq,ero,ezo,Fx,F,cdrq,cdzq,Opy,FA,Opo,L.drx);
            if daq < L.drx*L.P.tolq, s = true; break; end
          end
      end

      % Expected results
      %   Solve (rA-a*cos(oq)-rX)^2-(zA+a*sin(oq)-zX)^2 = 0
      oX = atan2(zX-zA,rX-rA);
      aX = sqrt((zA-zX).^2 + (rX-rA).^2);
      aq_ = (-cos(oq-oX) - sqrt(cos(oq-oX).^2-cos(2*oq)*cos(2*oX)))./cos(2*oq).*aX;
      
      % Checks
      if strcmp(method,'rtcics')
        tola = 2*L.drx*L.P.tolq;
      else
        tola = L.drx; % rtci fails to find contours accurately around  critical points
      end
      testCase.verifyTrue(s,sprintf('%s did not converge',method));
      testCase.verifyEqual(aq,aq_,'AbsTol',tola,sprintf('%s results differ from expected results',method));

      % Debug plots
      % Only make figure visible if verbose
      visible = testCase.verbosity>0;
      fig = figure('Visible',ifc(visible,'on','off')); % Fix for R2017a
      % Always close invisible figures
      if ~visible, clean = onCleanup(@() close(fig)); end
      ax = axes(fig);
      hold(ax,'all');
      axis(ax,'equal');
      contourf(ax,L.rx,L.zx,Fx,21);
      contour(ax,L.rx,L.zx,Fx,F*[1,1],'-r','Linewidth',2);
      plot(ax,rA+aq0.*crq,zA+aq0.*czq,'.r');
      plot(ax,rA+aq .*crq,zA+aq .*czq,'.c');
      plot(ax,rA+aq_.*crq,zA+aq_.*czq,'om');
      title(ax,method);
    end
  end
end
