classdef meqpdom_jacobian_test < meq_jacobian_test
  % test the different analytical boundary jacobian computed using
  % meqpdom
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    verbosity = 0;
    tok = 'ana';
  end
    
  properties(ClassSetupParameter)
    shot = struct('circular',1,'diverted',2,'diverted2',3,...
      'squashed',5,meq_jacobian_test.doublets_under_test{:});
    usecs = struct('true',true,'false',false);
  end

  properties (TestParameter)
    convergence_test = struct('false',false,'true',true);
  end
  
  methods(TestClassSetup)
    function get_LLX(testCase,shot,usecs)
      % Compute FGS initial guess (FBT solution)
      t = 0;
      testCase.get_fgs_LLX(shot,t);
      if usecs
        testCase.L.P.icsint = true;
        testCase.L.P.ilim   = 3;
        testCase.L = fgsc(testCase.L.P,testCase.L.G);
      end
      
      if shot == 1
        % Make perturbation smaller for this limited case
        testCase.meshDeltaFx = logspace(-7,-6,3);
      elseif shot == 2
        % Convergence saturates early here
        testCase.meshDeltaFx = logspace(-5,-4,3);
      elseif shot == 82
        % Doublet flux on limiter is very sensitive
        testCase.deltaFx = 1e-8;
        testCase.meshDeltaFx = logspace(-8,-7,3);
      end
    end
  end

  methods(Test,TestTags={'Jacobian'})
    function test_boundary_jacobian(testCase,convergence_test)

      L = testCase.L;
      LX = testCase.LX;

      % Prepare arguments
      fun = @meqpdom;
      x0 = {LX.Fx,LX.Ip,L.P.isaddl,L};

      % Call meqpdom
      [rA,zA,FA,~,~,~,~,~,FX,~,~,~,...
        rB,zB,FB,~,~,~,~,~,~,~,~,dF0dFx,dF1dFx] = ...
        meqpdom(x0{:});

      if L.P.icsint
        % Check we aren't too close to cell boundaries
        mindz = min(abs([zA;zB] - L.zx.'),[],[1,2]);
        mindr = min(abs([rA;rB] - L.rx.'),[],[1,2]);
        testCase.assumeTrue(min(mindr,mindz)>1e-4,'Skipping test as axis/boundary point is too close to grid cell boundary');
      end
      
      % F0 sizes
      nA = numel(FA);
      nX = numel(FX);
      nB = numel(FB);

      szF0 = [repmat({nA},1,6),repmat({nX},1,6),repmat({nB},1,5),{L.ny},repmat({L.nD},1,2)];

      % Check axis and boundary flux (checking stencils takes too much time
      % with cubic spline interpolation)
      names_out = {'F0','F1'};
      iargout = [19,20];
      names_in = {'Fx'};
      iargin = 1;
      J0 = {dF0dFx;dF1dFx};
      
      jacfd_args = {'szF0',szF0};

      if convergence_test
        % Test convergence
        epsval = {testCase.meshDeltaFx};
      else
        % Test baseline error
        epsval = {testCase.deltaFx};
      end
      % Call generic function
      testCase.test_analytical_jacobian(...
        fun,x0,J0,jacfd_args,iargout,names_out,iargin,names_in,epsval);
    end
  end
  
end
