classdef meqcde_jacobian_test < meq_jacobian_test
  % Test to verify the jacobians of the different meqcde options
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    verbosity = 0;  
    tok = 'ana';
    nu = 30; %
    wcompa = 0; % no regularization in compensation currents
  end

  properties (TestParameter)
    % Convergence test skipped due to conditioning problems
    %   (res is small but precision on res is poor)
    convergence_test = struct('false',false);
  end
  
  properties (ClassSetupParameter)
    shot = struct('circular',1,'diverted',2,'diverted2',3,'squashed',5,...
      meq_jacobian_test.doublets_under_test{:});
    cdetype = {'OhmTor_0D','OhmTor_rigid_0D','OhmTor_rigid_0D_noLpext','cde_ss_0D','static_Iy_1D','static_cde_1D','cde_OhmTor_1D','cde_1D'}
  end
  
  methods(TestClassSetup)
    function testSetup(testCase, shot, cdetype)
      iLpext = true;
      if contains(cdetype, 'noLpext') % for OhmTor_rigid without Lpext
        cdetype = erase(cdetype,'_noLpext');
        iLpext = false;
      end

      % Running FGE is necessary to have 2 different time slices
      if strcmp(cdetype, 'static_cde_1D') && floor(shot/80)==1
        % static_cde_1D needs extra constraints for doublet shots
        % otherwise, initialization is degenerate
        agcon = {'bp','ag'};
      elseif endsWith(cdetype,'1D')
        agcon = {'bp'};
      else
        agcon = {'bp','qA'};
      end
      [L,LX,LY] = fge(testCase.tok,shot,[0,1e-6], ...
        'usepreconditioner',false,...
        'ssinit',false,...
        'selu','e','nu',testCase.nu,...
        'cde',cdetype,...
        'agcon',agcon,...
        'wcompa',testCase.wcompa,...
        'iLpext',iLpext);

      testCase.L  = L;
      testCase.LX = LX;
      testCase.LY = LY;
    end
  end
  
  methods(Test,TestTags={'Jacobian'})
    function meqcde_finite_difference_test(testCase,convergence_test)
      L = testCase.L;
      LX  = meqxk(testCase.LX,2);
      LYp = meqxk(testCase.LY,1);
      LY  = meqxk(testCase.LY,2);

      % Prepare arguments
      F0   = LY.F0;
      F1   = LY.F1;
      Opy  = LY.Opy;
      Iy   = LY.Iy;
      Fx   = LY.Fx;
      ag   = LY.ag;
      Ie   = [LY.Ia; LY.Iu];
      
      % Prepare inputs
      fun = @meqcde;
      x0 = {L,LX,LYp,Fx,ag,Iy,F0,F1,Ie,Opy};

      % orders of meqcde derivatives output
      names_in = {'Fx','ag','Iy','F0','F1','Ie'};
      nin = numel(names_in);
      % Index of variables in meqcde inputs
      iargin = [4,5,6,7,8,9];

      J0 = cell(1, nin); % the residual and derivatives are returned
      [~, J0{:}] = fun(x0{:});

      % Scaling of inputs
      xscal = 10.^(log10(cellfun(@(x) max(abs(x(:))),x0(iargin))));
      xscal(xscal==0) = 1;

      jacfd_args = {'szF0',{L.np}};

      if convergence_test
        % Test convergence
        epsval = arrayfun(@(x) x*testCase.meshDeltaFx, xscal, 'UniformOutput', false);
      else
        % Test baseline error
        epsval = num2cell(xscal*testCase.deltaFx);
      end
      % Call generic function
      testCase.test_analytical_jacobian(...
        fun,x0,J0,jacfd_args,1,{'rescde'},iargin,names_in,epsval);
      
    end
  end
end
