classdef meq_toolbox_checks < meq_test
  % Generic toolbox-level checks
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  methods(Test,TestTags = {'Unit'})
    function test_version(testCase)
      % Check that toolbox version and release date from Contents.m matches CHANGELOG information
      testCase.assumeFalse(verLessThan('matlab','9.6'),'Skip version test for matlab<9.6')
      % Get from Contents.m using ver()
      meqpath = fullfile(fileparts(mfilename('fullpath')),'..');
      v = ver(meqpath);
      testCase.assertNotEmpty(v,'could not get version information for meq toolbox')
      
      % Get from CHANGELOG.md by parsing
      str = fileread('CHANGELOG.md');
      tokens = regexp(str,'## \[(\d+\.\d+\.\d+)\] \- (\d*\-\d*\-\d*)','tokens');
      logvers=tokens{1}{1}; logdate=tokens{1}{2};
      unreleased = contains(str,'## Unreleased');
      
      if unreleased
        % This is a development version
        % Verify that version from Contents is posterior to CHANGELOG

        % Decompose version numbers
        logvers_ = sscanf(logvers, '%d.%d.%d')';
        % Pad missing components
        if numel(logvers_) < 3
          logvers_(end+1:3) = 0;
        end

        % Decompose version number
        tbxvers_ = sscanf(v.Version, '%d.%d.%d')';
        % Pad missing components
        if numel(tbxvers_) < 3
          tbxvers_(end+1:3) = 0;
        end
        
        if logvers_(1) ~= tbxvers_(1)
          check = logvers_(1) < tbxvers_(1);
        elseif logvers_(2) ~= tbxvers_(2)
          check = logvers_(2) < tbxvers_(2);
        else
          check = logvers_(3) < tbxvers_(3);
        end

        testCase.verifyTrue(check,sprintf('CHANGELOG.md last version %s is not older than Contents.m version %s',logvers,v.Version));
        
      else
        % This is a released version
        % Verify that versions are the same
        testCase.verifyEqual(v.Version,logvers,...
          sprintf('CHANGELOG.md last version %s does not match Contents.m version %s',logvers,v.Version));
    
        % Verify that dates are the same
        dateconvert = datestr(v.Date,'yyyy-mm-dd'); % convert into same format
        testCase.verifyEqual(dateconvert,logdate,...
          sprintf('CHANGELOG.md release date %s does not match Contents.m date %s',logdate,dateconvert));
      end
    end    
  end
end