classdef meqFx_test < meq_test
  % Test to ensure that the gsxe option leads to similar results
  % in the inversion of the GS operator meqFx
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    L
    LX
    Fx
    tol = 1e-2;
  end
  
  properties(TestParameter)
    gsxe = {1,2,3};
    gsxetol = {0,1e-8,1e-6,1e-4};
  end
  
  properties (ClassSetupParameter)
    machine = {'ana'};
    shot = struct('limited',1,'diverted',2,'double_null',3,'squashed',5,'doublet',82);
  end
  
  methods(TestClassSetup)
    function testSetup(testCase, machine, shot)
      [L_, LX_] = fgs(machine, shot,[]);
      LX_.Iu = 1e-4*L_.Iu0.*(2*rand(L_.G.nu,1)-1);
      Ie = [LX_.Ia; LX_.Iu];
      testCase.L = L_;
      testCase.LX = LX_;
      testCase.Fx = meqFx(L_, LX_.Iy, Ie);
    end
  end
  
  methods(Test,TestTags={'Unit'})
    
    function test_meqFx_with_gsxe_options(testCase, gsxe)
      % Verify different gsxe option values yield similar answers
      Iy = testCase.LX.Iy;
      Ie = [testCase.LX.Ia; testCase.LX.Iu];
      Fx_baseline = testCase.Fx;
      
      L_ = fgs(testCase.L.P.tok, testCase.L.P.shot, [], 'gsxe', gsxe);
      
      Fx_test = meqFx(L_, Iy, Ie);
      testCase.verifyEqual(Fx_baseline, Fx_test, 'AbsTol', testCase.tol*testCase.L.Fx0);
    end
    
    function test_meqFx_with_gsxetol(testCase, gsxetol)
      % Verify different gsxetol values yield similar answers
      Iy = testCase.LX.Iy;
      Ie = [testCase.LX.Ia; testCase.LX.Iu];
      Fx_baseline = testCase.Fx;
      
      L_ = fgs(testCase.L.P.tok, testCase.L.P.shot, [], 'gsxe', 2, 'gsxetol', gsxetol);
      
      Fx_test = meqFx(L_, Iy, Ie);
      testCase.verifyEqual(Fx_baseline, Fx_test, 'AbsTol', testCase.tol*testCase.L.Fx0);
    end

    function test_meqFx_with_Fbext(testCase)
      % Verify passing Ie or Fb_ext yields same answer
      L_ = testCase.L;
      Iy = testCase.LX.Iy;
      Ie = [testCase.LX.Ia; testCase.LX.Iu];
      Fx_baseline = testCase.Fx;

      Fbe = L_.Mbe*Ie;
      Iyie = reshape(L_.Tye*Ie,L_.nzy,L_.nry);

      % Passing Fb_ext parameter should yield the exact same answer
      Fx_test = meqFx(L_,Iy,Ie,{Fbe,Iyie});
      testCase.verifyEqual(Fx_baseline, Fx_test);
    end

    function test_meqFx_failure(testCase)
      % Verify errors are triggered when expected
      L_ = testCase.L;
      Iy = testCase.LX.Iy;
      Ie = [testCase.LX.Ia; testCase.LX.Iu];
      dz = 1e-4;
      Jh = zeros(0,1);

      errid = 'meqFx:incompatibleInputs';

      % Pass multiple Ie but single Iy
      testCase.verifyError(@() meqFx(L_,Iy,repmat(Ie,1,3)),errid);

      % Pass multiple Ie and non-zero dz (needs all 7 args)
      testCase.verifyError(@() meqFx(L_,Iy,repmat(Ie,1,3),{},dz,false,Jh),errid);

      % Pass non-zero dz and rst=true (needs all 7 args)
      testCase.verifyError(@() meqFx(L_,Iy,Ie,{},dz,true,Jh),errid);

      % Pass too few dz (needs all 7 args)
      testCase.verifyError(@() meqFx(L_,repmat(Iy,1,1,3),Ie,{},dz,false,Jh),errid);

      % Invalid mode
      L_.P.gsxe = NaN;
      testCase.verifyError(@() meqFx(L_,Iy,Ie),'meqFx:invalidmode');
    end
  end
end
