classdef mds2meq_test < meq_test
% tests of MDS retrieval of MEQ structures
%
% [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties(TestParameter)
    source = {'LIUQE.M',   'FBTE','RTLIUQE','LIUQE.M',   'FBTE','RTLIUQE','LIUQE.M',   'FBTE','LIUQE.M',   'FBTE','LIUQE.M',   'FBTE','RTLIUQE'};
    shot   = {    70474,   70474 ,    70474,    70477,   70477 ,    70477,    70563,   70563 ,    73306,    73306,    83850,    83850,    83850};
  end
  
  methods(Test,TestTags={'TCV'},ParameterCombination='sequential')
    function mds2meq_data_retrieval(testCase,source,shot)
      
      skipfields = {'shot','rX','zX','FX'}; % X-points quantities have variable sizes
      always_add_time = {'aq','rq','zq'}; % 4D fields, always add time dimensions for these
      [L,LY] = mds2meq(shot,source);
      nt = numel(LY.t);
      for field=fieldnames(LY)'
        myfield = field{:};
        if any(strcmp(myfield,skipfields)), continue; end       
        expected_size_slice = meqsize(L,myfield);
        if expected_size_slice(end) == 1 && ~ismember(myfield,always_add_time)
          expected_size = [expected_size_slice(1:end-1),nt];
        else
          expected_size = [expected_size_slice,nt];
        end
        % Fix for R2017a verifySize, remove trailing singletons
        ndim = max(find(expected_size~=1,1,'last'),2);
        if isempty(ndim), ndim = 2; end
        expected_size = expected_size(1:ndim);
        % Check
        testCase.verifySize(LY.(myfield),expected_size,...
          sprintf('invalid size of retrieved data for %s',myfield))
      end
    end
    
    function mds2meq_units(testCase,source,shot)
      
      [L,LY] = mds2meq(shot,source);
        
      testCase.assumeTrue(L.nD==1,sprintf('Skipping mds2meq Iy test for shot %d: multiple domains',shot));
      
      % Test slice with largest current
      [~,kt] = max(abs(LY.Ip));

      Fx = LY.Fx(:,:,kt);
      if ~isfield(LY,'Opy')
        if strcmp(source,'FBTE'), isaddl = L.P.isaddl(kt);
        else,                     isaddl = L.P.isaddl;    end
        
        [~,~,FA,~,~,~,~,~,~,~,~,~,...
          ~,~,FB,~,~,Opy,~,~,pdomstat] = meqpdom(Fx,LY.Ip(kt),isaddl,L);
        
        testCase.assumeTrue(pdomstat,sprintf('Skipping mds2meq Iy test for shot %d: meqpdom failed',shot));
      else
        FA  = LY.FA (    kt);
        FB  = LY.FB (    kt);
        Opy = LY.Opy(:,:,kt);
      end
        
      FNy = (Fx(2:end-1,2:end-1)-FA)/(FB-FA);
      Ppy  = interp1(L.pQ.^2,LY.PpQ(:,kt) ,FNy,'spline'); Ppy (~Opy) = 0;
      TTpy = interp1(L.pQ.^2,LY.TTpQ(:,kt),FNy,'spline'); TTpy(~Opy) = 0;
      
      Iy = 2*pi*(L.ry.'.*Ppy + TTpy.*L.iry.'/mu0)*L.dsx;
      
      % Test with low tolerance, allowing for differences in models
      testCase.verifyEqual(Iy,LY.Iy(:,:,kt),'AbsTol',max(max(abs(LY.Iy(:,:,kt))))*1e-2,...
        'Iy from PpQ/TTpQ does not match MDS storage within required tolerance');
      
    end
  end
  
  
end
