classdef liu_tests_advanced < meq_test
  % More advanced LIUQE tests that don't require scanning across many
  % parameters and equilibria
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties
    verbosity = 0;
    L,LX;
  end
  
  properties(TestParameter)
    debug={0,1,2,3};
    algoNL = {'all-nl','all-nl-Fx','Newton-GS'}
  end
  
  methods(TestClassSetup)
    function setup_LLX(testCase)
      [testCase.L,testCase.LX] = liu('ana',2,0,'useSQP',true);
    end
    
    function setup_fig(testCase)
      if testCase.verbosity==0
        Visible = 'off';
      else
        Visible = 'on';
      end
      hf = figure('name',sprintf('%s figure',mfilename),...
        'Visible',Visible);
      testCase.addTeardown(@() delete(hf) );
    end
  end
  
  methods(Test,TestTags={'Unit'})
    function test_general_constraints(testCase)
      L = testCase.L; LX = testCase.LX; %#ok<*PROP,*PROPLC>
      LY0 = liut(L,LX); 
      testCase.verifyTrue(LY0.isconverged,'nominal case did not converge');
      
      for constraint = {'Wk','Ip','li','qA','ag'}
        myconstr = constraint{:};
        nominal_value = LY0.(myconstr);
        LX.(myconstr) = nominal_value*1.01; % scale slightly w.r.t. nominal
        LX = liux(L,LX);
        L.P.liuagcon = {myconstr,'ag','ag'};
        L.P.wCo = [Inf;0;0]; % impose constraint = LX.(constraint), leave other directions free
        L.P.useSQP = true; L.P.wsSQP = 0;
        L = liuc(L.P,L.G);
        LY = liut(L,LX,... % start from previous
          'Iy',LY0.Iy,'Ie',[LY0.Ia;LY0.Iu],'ag',LY0.ag,'dz',LY0.dz,'rst',false);
        testCase.verifyTrue(LY.isconverged,'did not converge')
        tol = 1e-4;
        mask = find(L.P.wCo);
        testCase.verifyEqual(LY.(myconstr)(mask),LX.(myconstr)(mask),'RelTol',tol,...
          sprintf('Too large error constraining %s',myconstr))
      end
    end
    
    function algoNL_test(testCase,algoNL)
      % test that all algoNL options converge
      
      % set algoNL and stabz
      P = testCase.L.P; G = testCase.L.G;
      P.algoNL = algoNL;
      P.stabz = strcmp(algoNL,'all-nl');
      
      % re-consolidate
      L_ = liuc(P,G); 
      
      % run liut
      LY = liut(L_,testCase.LX);
      
      % checks
      testCase.verifyTrue(~isempty(LY));
      testCase.verifyTrue(LY.isconverged)
    end

    function debug_test(testCase,debug)
      % check that all debug levels work
      L = testCase.L; LX = testCase.LX;
      L.P.debug = debug;
      L.P.debugplot = max(0,debug-1);
      L.P.itera = 2; % shorten
      LY = liut(L,LX);
      testCase.verifyTrue(~isempty(LY));
    end
    
  end
end
