classdef liuNLmeas_jacobian_test < meq_jacobian_test
  % Compares FGS and FGE semi-analytical Jacobian with finite-difference Jacobian
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties(TestParameter)
    algosNL = {'all-nl', 'all-nl-Fx', 'Newton-GS'};
    shot = struct('circular',1,'diverted',2,'squashed',5,'elongated',11,...
      meq_jacobian_test.doublets_under_test{:});
    usecs = struct('true',true,'false',false);
    %
    convergence_test = struct('false',false,'true',true);
  end

  properties
    verbosity = 0;
    tok = 'ana';
  end

  methods (Test,ParameterCombination='pairwise',TestTags={'Jacobian'})
    function test_liuNLmeas_jacobian(testCase,algosNL,shot,usecs,convergence_test)

      % Generate baseline eq
      [Lfbt,~,LYfbt] = fbt(testCase.tok,shot,[]);

      if usecs
        % Check we aren't too close to cell boundaries
        mindz = min(abs([LYfbt.zA;LYfbt.zX] - Lfbt.zx.'),[],[1,2]);
        mindr = min(abs([LYfbt.rA;LYfbt.rX] - Lfbt.rx.'),[],[1,2]);
        testCase.assumeTrue(min(mindr,mindz)>1e-6,'Skipping test as singular point is too close to grid cell boundary');
      end

      PP = {'algoNL',algosNL,'debug',testCase.verbosity,...
        'selu','e','nu',30,...
        'wreg',2e-6,'ipm',true,'idml',true,'stabz',false};
      if usecs
        PP = [PP,{'icsint',true ,'ilim',3}];
      else
        PP = [PP,{'icsint',false,'ilim',1}];
      end
      L = liu(testCase.tok,shot,[],PP{:});
      LX = meqxconvert(Lfbt,LYfbt,L,true); % Generate minimal LX
      LX.ag = LYfbt.ag;
      LX.IpD = LX.Ip/L.nD;

      x0 = L.LX2x(LYfbt);
      u0 = [LX.Ia(:);LX.Iu(:);LX2Co(L,LYfbt)];
      aux = cell(1,6);
      aux{6} = true; % Do not compute approximate jacobian
      F = @(x,u,opts) liuNLmeas_wrap(x,L,LX,LX,opts,aux,u);

      % Compute analytical jacobian
      J0 = cell(3,2);
      opts = optsF('dojacx',true,'dojacu',true);
      [~,~,~,J0{:}] = F(x0,u0,opts);

      % Check derivatives with respect to x
      names_out = {'Xt','Xq','Xc'};
      iargout = [1,2,3];
      names_in = {'x','u'};
      iargin = [1,2];

      jacfd_args = {'szF0',{1,L.nq,L.nc}};

      if convergence_test
        % Test convergence
        if strcmp(algosNL, 'all-nl')
          dx = logspace(-4,-2,4);
        else
          dx = logspace(-6,-5,3);
        end
        epsval = repmat({dx},1,numel(iargin));
      else
        % Test baseline error
        if strcmp(algosNL, 'all-nl')
          dx = 0.5e-4;
        else
          dx = 1e-6;
        end
        epsval = repmat({dx},1,numel(iargin));
      end

      % Call generic function
      testCase.test_analytical_jacobian(...
        F,{x0,u0,optsF},J0,jacfd_args,iargout,names_out,iargin,names_in,epsval);
    end
  end

end

function [Xt,Xq,Xc,dXtdx,dXqdx,dXcdx,dXtdu,dXqdu,dXcdu] = ...
  liuNLmeas_wrap(x,L,LX,LYp,opts,aux,u)
% Wrapper function to test jacobians of Xt, Xq, Xc
[~,LY] = fgeF(x,L,LX,LYp,opts,aux,u);
Xt = LY.Xt;
Xq = LY.Xq;
Xc = LY.Xc;
if opts.dojacx || opts.dojacu
  dXtdx = LY.dXtdx;
  dXqdx = LY.dXqdx;
  dXcdx = LY.dXcdx;
  dXtdu = LY.dXtdu;
  dXqdu = LY.dXqdu;
  dXcdu = LY.dXcdu;
end
end
  

