classdef ipm_test < meq_test
  % Tests for IPM
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.
  
  properties
    data; % problem-specific parameters, solution, etc.
    verbosity = 1;
    tol = 1e-12;    % convergence tolerance
    niter = 100;    % max iterations
    check_solution; % flag whether to check ipm solution vs quadprog
    osel = [];      % quadprog optimizer settings
  end
  
  properties(ClassSetupParameter)
    problem = {'SI','I','P','Sp','Sn','Sp1','Sn1','S1p','S1n','S1p1','S1n1',...
      'E','ER','ER2','EI','EB','EI2','EN','EAN'}; 
  end

  properties(TestParameter)
    useoptimized = struct('true',true,'false',false);
    presolve = struct('true',true,'false',false);
  end
  
  methods(TestClassSetup)
    function set_problem(testCase,problem)
      [H,f,A,b,Aeq,beq,lb,ub] = deal([]); %#ok<*PROPLC>
      [x,y,s,z] = deal(zeros(0,1));
      expected_warning = ''; % default
      expected_status  = 1; % default
      % Examples taken from help of quadprog:

      switch problem
        case 'SI' % Scalar
          H = 1;
          f = 0;
          A = 1;
          b = 1;
          modes = [0,1,2];
          % Solution
          x = 0;
          s = 1;
          z = 0;
        case 'I'
          % "Quadratic Program with Linear Inequality Constraints"
          H = [1 -1; -1 2];
          f = [-2; -6];
          A = [1 1; -1 2; 2 1];
          b = [2; 2; 3];
          modes = [0,1,2];
          % Solution
          x = [2;4]/3;
          s = [0;0;1]/3;
          z = [7;1;0]*4/9;
        case 'P'
          % Adapted from I for positive solutions
          H = [1 -1; -1 2];
          f = [-2; -6];
          lb = zeros(2,1);
          ub = Inf(2,1);
          modes = [0,1,2,3];
          % Solution
          x = [10;8];
          s = [x;ub];
          z = [0;0;0;0];
        case {'Sp','Sn','Sp1','Sn1'}
          % Adapted from I for imposed signs
          H = [1 -1; -1 2];
          f = [-2; -6];
          switch problem
            case 'Sp',  a =  2.5;
            case 'Sn',  a = -2.5;
            case 'Sp1', a =  1.0;
            case 'Sn1', a = -1.0;
          end
          A = a*eye(2,2);
          b = zeros(2,1);
          modes = [0,1,2,4];
          % Solution
          if a>0
            x = [0;0];
            s = [0;0];
            z = -f/a;
          else
            x = [10;8];
            s = abs(a)*x;
            z = [0;0];
          end
        case {'S1p','S1n','S1p1','S1n1'}
          % Adapted from I for one imposed sign
          H = [1 -1; -1 2];
          f = [-2; -6];
          switch problem
            case 'S1p',  a =  2.5;
            case 'S1n',  a = -2.5;
            case 'S1p1', a =  1.0;
            case 'S1n1', a = -1.0;
          end
          A = a*[0,1];
          b = 0;
          modes = [0,1,2,5];
          % Solution
          if a>0
            x = [2;0];
            s = 0;
            z = 8/a;
          else
            x = [10;8];
            s = abs(a)*x(2);
            z = 0;
          end
        case 'E'
          % "Quadratic Program with Linear Equality Constraint"
          H = [1 -1; -1 2];
          f = [-2; -6];
          Aeq = [1 1];
          beq = 0;
          modes = [0,1];
          % Solution
          x = [-1;1]*4/5;
          y = 18/5;
        case 'ER'
          % Same as E but with redundant constraints
          H = [1 -1; -1 2];
          f = [-2; -6];
          Aeq = [1 1;2 2];
          beq = [0;0];
          modes = [0,1];
          expected_warning = 'ipm:RedundantEqConstraints';
          % Solution
          x = [-1;1]*4/5;
          y = [1;0]*18/5;
        case 'ER2'
          % Same as E but with redundant constraints involving a single variable
          H = [1 -1; -1 2];
          f = [-2; -6];
          Aeq = [1 0;1 0];
          beq = [0;0];
          modes = [0,1];
          expected_warning = 'ipm:RedundantEqConstraints';
          % Solution
          x = [0;3];
          y = [1;0]*5;
        case 'EI'
          % Same as E but with infeasible constraints
          H = [1 -1; -1 2];
          f = [-2; -6];
          Aeq = [1 1;2 2];
          beq = [0;1];
          modes = [0,1];
          expected_status = -2;
        case 'EI2'
          % Same as E but with infeasible constraints involving a single variable
          H = [1 -1; -1 2];
          f = [-2; -6];
          Aeq = [1 0;1 0];
          beq = [0;1];
          modes = [0,1];
          expected_status = -2;
        case 'EB'
          % "Quadratic Program with Linear Constraints and Bounds"
          H = [ 1,-1, 1
               -1, 2,-2
                1,-2, 4];
          f = [2;-3;1];
          lb = zeros(3,1);
          ub = ones(size(lb));
          Aeq = ones(1,3);
          beq = 1/2;
          modes = [0,1];
          % Solution
          x = [0;1;0]/2;
          y = 2;
          s = [0;1;0;2;1;2]/2;
          z = [7;0;4;0;0;0]/2;
        case 'EN'
          % "Quadratic Program with Null Equality Constraint"
          H = [1 -1; -1 2];
          f = [-2; -6];
          Aeq = [0 0];
          beq = 0;
          modes = [0,1];
          expected_warning = 'ipm:RedundantEqConstraints';
          % Solution
          x = [10;8];
          y = 1;
        case 'EAN'
          % "Quadratic Program with ignored variable and Null Equality Constraint"
          H = [1 -1 0; -1 2 0;0 0 0];
          f = [-2; -6; 0];
          Aeq = [0 0 0];
          beq = 0;
          modes = [0,1];
          expected_warning = 'ipm:RedundantEqConstraints';
          % Solution
          x = [10;8;0];
          y = 1;
      end

      % Separate s,z into each category
      [iin,ilb,iub] = n2k(size(A,1),numel(lb),numel(ub));
      sin = s(iin); zin = z(iin);
      slb = s(ilb); zlb = z(ilb);
      sub = s(iub); zub = z(iub);

      % Set correct sizes
      nx = numel(x);
      if isempty(A  ), A   = zeros(0,nx); end
      if isempty(b  ), b   = zeros(0, 1); end
      if isempty(Aeq), Aeq = zeros(0,nx); end
      if isempty(beq), beq = zeros(0, 1); end
      if isempty(lb ), lb  =  -Inf(nx,1);
                       slb =   Inf(nx,1);
                       zlb = zeros(nx,1);
      end
      if isempty(ub ), ub  =   Inf(nx,1);
                       sub =   Inf(nx,1);
                       zub = zeros(nx,1);
      end
      
      s = [sin;slb;sub]; z = [zin;zlb;zub];

      testCase.data = {H,f,A,b,Aeq,beq,lb,ub,x,y,s,z,modes,expected_status,expected_warning};

      % Check solution only if problem is feasible
      if expected_status ~= 1, return; end

      % Check solution
      rd = H*x + f + Aeq.'*y + A.'*zin - zlb + zub;
      re = Aeq*x - beq;
      rc = z.*s;
      rc(isinf(s)) = 0;
      rp = s + [A*x - b; -x + lb; x - ub];
      rp(isinf(s)) = 0;
      r = [rd;re;rc;rp];
      testCase.assertEqual(r,zeros(size(r)),'AbsTol',10*testCase.tol,'Provided solution does not verify the optimality conditions');
    end
  end
  
  methods(Test, TestTags = {'Unit'})
    function ipmTest(testCase)

      [H,f,A,b,Aeq,beq,lb,ub,x,y,s,z,modes,stat,wngID] = deal(testCase.data{:});

      % Run low-level IPM when expecting success (cases treated by ipmwrapper instead)
      testCase.assumeTrue(isempty(wngID) && stat>=0, 'Skipping low-level IPM call for cases when failure is expected');

      for mode = modes % loop over modes
        [x0,y0,s0,z0,st] = testCase.run_ipm(H,f,A,b,Aeq,beq,lb,ub,mode,...
          testCase.niter,testCase.tol,testCase.verbosity);

        testCase.verifyEqual(st,stat,'ipm did not return expected status flag')
        
        if st ~= 1, continue; end % no further checks necessary for non-converged cases
        tol_ = 10*testCase.tol;
        testCase.verifyEqual(x0,x,'AbsTol',tol_,sprintf('Exact solution and ipm(mode=%d) for x do not match within tolerance',mode));
        testCase.verifyEqual(y0,y,'AbsTol',tol_,sprintf('Exact solution and ipm(mode=%d) for y do not match within tolerance',mode));
        testCase.verifyEqual(s0,s,'AbsTol',tol_,sprintf('Exact solution and ipm(mode=%d) for s do not match within tolerance',mode));
        testCase.verifyEqual(z0,z,'AbsTol',tol_,sprintf('Exact solution and ipm(mode=%d) for z do not match within tolerance',mode));
      end
    end

    function ipmwrapperTest(testCase,useoptimized,presolve)

      [H,f,A,b,Aeq,beq,lb,ub,x,y,~,z,~,stat,wngID] = deal(testCase.data{:});

      % Use meq's ipmwrapper method
      opts = ipmopts('useoptimized',useoptimized,...
                     'niter',testCase.niter,...
                     'tol',testCase.tol,...
                     'debug',logical(testCase.verbosity),...
                     'presolve',presolve);
      funipm = @() ipmwrapper(H,f,A,b,Aeq,beq,lb,ub,[],opts);

      % run IPMWRAPPER while checking warnings
      if isempty(wngID)
        [x0,~,st,~,lambda0] = testCase.verifyWarningFree(funipm);
      else
        [x0,~,st,~,lambda0] = testCase.verifyWarning(funipm,wngID);
      end
      testCase.verifyEqual(st,stat,'ipm did not return expected status flag')

      % If result can be compared to a golden solution, check it
      if stat ~= 1, return; end % no further checks necessary for non-converged cases
      tol_ = 10*testCase.tol;
      msg = 'Exact solution and ipmwrapper values for %s do not match within tolerance';
      testCase.verifyEqual(x0,x,'AbsTol',tol_,sprintf(msg,'x'));
      %
      % Note: When redundant equality constraints are specified, choice
      % of lagrange coefficients is not unique, only Aeq'*y is
      testCase.verifyEqual(Aeq.'*lambda0.eqlin  ,Aeq.'*y  ,'AbsTol',tol_,sprintf(msg,'Aeq''*lambda.eqlin'));
      % Add z for lower and upper bounds when not specified
      [iin,ilb,iub] = n2k(size(A,1),numel(lb),numel(ub));
      if ~isempty(A)
        testCase.verifyEqual(lambda0.ineqlin,z(iin),'AbsTol',tol_,sprintf(msg,'lambda.ineqlin'));
      end
      if ~isempty(lb)
        testCase.verifyEqual(lambda0.lower  ,z(ilb),'AbsTol',tol_,sprintf(msg,'lambda.lower'));
      end
      if ~isempty(ub)
        testCase.verifyEqual(lambda0.upper  ,z(iub),'AbsTol',tol_,sprintf(msg,'lambda.upper'));
      end
    end

    function quadprog_comparison(testCase)

      [H,f,A,b,Aeq,beq,lb,ub] = deal(testCase.data{1:8});

      testCase.assumeTrue(~isempty(which('quadprog')),'Skipping comparison with quadprog');

      % First use meq's ipmwrapper
      opts = ipmopts('niter',testCase.niter,...
                     'tol',testCase.tol,...
                     'debug',logical(testCase.verbosity));
      [x0,~,st0,~,lambda0] = ipmwrapper(H,f,A,b,Aeq,beq,lb,ub,[],opts);

      opts = optimoptions('quadprog','Algorithm','interior-point-convex',...
        'FunctionTolerance',testCase.tol,'MaxIterations',testCase.niter);
      [x1,~,st1,~,lambda1] = quadprog(H,f,A,b,Aeq,beq,lb,ub,[],opts);

      testCase.verifyEqual(st0,st1,'ipmwrapper and quadprog did not return similar status flag')

      % If result can be compared to the quadprog solution, check it
      if st0 ~= 1, return; end % no further checks necessary for non-converged cases
      tol_ = 1e-6;
      msg = 'quadprog and ipmwrapper values for %s do not match within tolerance';
      testCase.verifyEqual(x0,x1,'AbsTol',tol_,sprintf(msg,'x'));
      % Note: When redundant equality constraints are specified, choice
      % of lagrange coefficients is not unique, only Aeq'*y is
      testCase.verifyEqual(Aeq'*lambda0.eqlin,Aeq'*lambda1.eqlin,'AbsTol',tol_,sprintf(msg,'Aeq''*lambda.eqlin'));
      testCase.verifyEqual(lambda0.ineqlin,lambda1.ineqlin,'AbsTol',tol_,sprintf(msg,'lambda.ineqlin'));
      testCase.verifyEqual(lambda0.lower  ,lambda1.lower  ,'AbsTol',tol_,sprintf(msg,'lambda.lower'));
      testCase.verifyEqual(lambda0.upper  ,lambda1.upper  ,'AbsTol',tol_,sprintf(msg,'lambda.upper'));
    end
  end
  
  methods
    function [x,y,s,z,stat] = run_ipm(testCase,H,f,A,b,Aeq,beq,lb,ub,mode,niter,tol,verbosity)
      
      if mode > 1
        testCase.verifyTrue(isempty(Aeq) && isempty(beq),'For modes other than 0 and 1, ipm does not support equality constraints')
      end
      
      nvar = size(H,1);
      % Group bounds with constraints
      Aipm = [-A;eye(nvar*~isempty(lb));-eye(nvar*~isempty(ub))];
      bipm = [-b;lb;-ub];
      % ipm does not support infinite bounds
      mask = isfinite(bipm);
      Aipm = Aipm(mask,:);
      bipm = bipm(mask);
      
      switch mode
        case 3
          test = isequal(Aipm,eye(nvar)) && isequal(bipm,zeros(nvar,1));
          testCase.assertTrue(test,'For ipm mode 3, Aipm,bipm must be equivalent to x>0');
        case 4
          test = isequal(Aipm/Aipm(1,1),eye(nvar)) && isequal(bipm,zeros(nvar,1));
          testCase.assertTrue(test,'For ipm mode 4, Aipm,bipm must be equivalent to a*x>0');
        case 5
          [i,j,s] = find(Aipm);
          test = numel(i) == 1 && b(i) == 0;
          testCase.assertTrue(test,'For ipm mode 5, Aipm,bipm must be equivalent to a*x(b)>0');
          Aipm = s; bipm = j;
      end

      [x,y,s_,z_,~,stat] = ipm(H,f,Aipm,bipm,Aeq,beq,[],[],[],[],mode,niter,tol,verbosity);
      % Restore s and z values for infinite bounds
      s =   Inf(numel(mask),1); s(mask) = s_;
      z = zeros(numel(mask),1); z(mask) = z_;
    end
  end
end
