classdef (SharedTestFixtures={meq_fixture,meqids_fixture}) ...
    ids_tests < matlab.unittest.TestCase
  % Tests for ids routines function
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties
    debugplot = 0; % 2 to display debugging plot
    L = struct; % Standard anamak L
  end

  % Do not display plots by default in test
  methods(TestClassSetup)
    function setFigureVisibilityOff(testCase)
      if testCase.debugplot
        plot_visibility = 'on';
      else
        plot_visibility = 'off';
      end
      original_visibility = get(0, 'DefaultFigureVisible');
      set(0, 'DefaultFigureVisible', plot_visibility); % Put to on to display debugging plot
      addTeardown(testCase, @() set(0, 'DefaultFigureVisible', original_visibility));
    end

    function generateStdAnamakFBT(testCase)
      L = fge('ana',1,0,...
        'fastcoil',true... % Vertical stability coils allows to test Tca IDS parsing for coil with reverse polarity
      );
      testCase.L = L;
    end
  end
  
  properties (TestParameter)
    % Parameters for gen_filaments filamentation
    vvdata = { [0;0;1;1] , [0;0;1;1;0;-45/180*pi]};
    type_coil_description = { 2 , 3};
    num = { 1, 4};
    n_rout = { 1, 4};
    n_zout = { 1, 4};
    aw_exp = { 1, 0.17*ones(4,1) };
    ww_exp = { 1, 0.42*ones(4,1) };
    hw_exp = { 1, 0.42*ones(4,1) };
    vessel_2_ids_key =  { 'wall', 'pf_passive'};
  end
  
  
  methods (Test, TestTags = {'ids'}, ParameterCombination='sequential')
    
    function test_gen_filament(...
        testCase, vvdata, type_coil_description, num,...
          n_rout, n_zout, aw_exp, ww_exp, hw_exp...
        )

      % Test generation of filaments from a coil description
      [rout, zout, aw, ww, hw] = gen_filament(vvdata, num, type_coil_description);
      
      testCase.verifyEqual(numel(rout), n_rout, 'AbsTol', 1e-1);
      testCase.verifyEqual(numel(zout), n_zout, 'AbsTol', 1e-1);
      testCase.verifyEqual(aw, aw_exp,  'AbsTol', 1e-1);
      testCase.verifyEqual(ww, ww_exp, 'AbsTol', 1e-1);
      testCase.verifyEqual(hw, hw_exp, 'AbsTol', 1e-1);
      
    end
    
    function test_ray_trace(testCase)
      % Test the ray tract from center (0,0) to square (1,-1),(1,1),(2,1),(2,-1)
      r = [1,1,2,2];  z = [-1,1,1,-1];
      r0 = 0; z0 = 0; theta = 0;
      rint_exp = [1;2]; zint_exp = [0;0];
      
      [rint,zint] = ray_trace(r,z,r0,z0,theta);
      testCase.verifyEqual(rint, rint_exp, 'AbsTol', 1e-6);
      testCase.verifyEqual(zint, zint_exp, 'AbsTol', 1e-6);
    end
    
    function test_outline2innerouterlines(testCase)
      % Extract inner and outer line wrt r0,z0 center for a rectangle
      
      r = [0;0;0.1;0.1]; z = [-1;1;1;-1]; % Coordinate of a rectangle.
      r0 = -1; z0 = 0; % Ray center.
      
      P = struct;
      P.rayntheta = 2; % Number of rays.
      P.debugplot = testCase.debugplot;
      
      % Expected values
      ri_exp = [0;0]; % r coords of ray intersecting points on the inner liner (facing r0,z0)
      ro_exp = [0.1;0.1]; % r coords of ray intersecting points on the outer liner (facing r0,z0)
      zi_exp = [0.9;-0.9]; % z coords of ray intersecting points on the inner liner (away from r0,z0)
      zo_exp = [1.;-1];% z coords of ray intersecting points on the outer liner (away from  r0,z0)
      [ri,zi,ro,zo] = outline2innerouterlines(r,z,r0,z0, P);
      
      testCase.verifyEqual(ri, ri_exp, 'AbsTol', 1e-2);
      testCase.verifyEqual(ro, ro_exp, 'AbsTol', 1e-2);
      testCase.verifyEqual(zi, zi_exp, 'AbsTol', 1e-2);
      testCase.verifyEqual(zo, zo_exp, 'AbsTol', 1e-2);
    end
    
    function test_patch_inner_outer(testCase)
      % Patch a closed contour geometry provided as inner and outer line
      P = struct;
      P.debugplot = testCase.debugplot;
      P.raydeltadist = 0.5; % [m]
      
      ri = [0,0]; zi = [-1,1];
      ro = [1,1]; zo = [-1,1];
      rr0 = 0; zz0 = 0;
      resistivity = 1;
      
      rv_exp = [0.5;0.5;0.5];
      zv_exp = [-0.6;0.;0.6];
      Rv_exp = [4.7;4.7;4.7];
      hv_exp = [0.8;0.8;0.8];
      wv_exp = [0.8;0.8;0.8];
      
      [rv,zv,hv,wv,Rv] = patch_inner_outer(ri,zi,ro,zo,rr0,zz0,resistivity, P);
      
      testCase.verifyEqual(rv, rv_exp, 'AbsTol', 1e-1);
      testCase.verifyEqual(zv, zv_exp, 'AbsTol', 1e-1);
      testCase.verifyEqual(Rv, Rv_exp, 'AbsTol', 1e-1);
      testCase.verifyEqual(hv, hv_exp, 'AbsTol', 1e-1);
      testCase.verifyEqual(wv, wv_exp, 'AbsTol', 1e-1);
      
    end

    function test_wall_2G_outline(testCase)
      % Test outline case for wall2G
      wall.description_2d{1}.vessel.type.index = 1;
      wall.description_2d{1}.vessel.unit{1}.element{1}.outline.r = [1;1;1.1;1.1];
      wall.description_2d{1}.vessel.unit{1}.element{1}.outline.z = [-0.5;0.5;0.5;-0.5];
      wall.description_2d{1}.vessel.unit{1}.element{1}.resistivity= 1;
         
      P.rayr0 = 0; P.rayz0 = 0; G = struct; P.debugplot = 0;
      G = wall2G(G,wall,P);

      testCase.verifyTrue(~isempty(G.rv));
      testCase.verifyTrue(~isempty(G.zv));
      testCase.verifyTrue(~isempty(G.nv));
      testCase.verifyTrue(~isempty(G.wv));
      testCase.verifyTrue(~isempty(G.hv));
      testCase.verifyTrue(~isempty(G.Rv));
      testCase.verifyTrue(~isempty(G.Rs));
      testCase.verifyTrue(~isempty(G.Tivs));

    end

    function test_G2ids_run(testCase, vessel_2_ids_key)
      % Test G2ids run till completion for anamak standard case
      L = testCase.L;

      % Convert G to ids
      P = struct;
      P.vessel_2_ids_key = vessel_2_ids_key;
      ids = G2ids(L.G, P);

      % Check all fields have been generated
      exp_fields = {'wall', 'pf_active', 'magnetics', 'pf_passive'};
      for ii = 1:numel(exp_fields)
        testCase.verifyTrue(isfield(ids, exp_fields{ii}));
      end
    end

    function test_plot_connection_matrix(testCase)
      L = testCase.L;
      % Convert G to ids
      P = struct;
      ids = G2ids(L.G, P);
      plot_connection_matrices(ids.pf_active);
    end

    function test_G2ids_ids2G_integration(testCase, vessel_2_ids_key)
      % Integration test. 
      % - For anamak compute G2ids and ids2G. 
      % - Check main physical code parameters.
      L = testCase.L;
      tol = 1e-5;
      neig = 10; % Number of eigenvalues to be tested

      % Convert G to ids
      P = struct;
      P.vessel_2_ids_key = vessel_2_ids_key;
      ids = G2ids(L.G, P);

      % G from ids
      G = struct();
      P = struct();
      P.rayr0 = (max(L.G.rv)+min(L.G.rv))/2; 
      P.rayz0 = 0; 
      P.debugplot = 0;
      P.discrtype = 'squares';

      % Extract G information
      G = pf_active2G(G, ids.pf_active, P);
      G = wall2G(G, ids.wall, P);
      G = magnetics2G(G, ids.magnetics, P);
      G = pf_passive2G(G, ids.pf_passive, P);

      % Check that all fields are equal
      testCase.verifyEqual(G.Tca, L.G.Tca, 'AbsTol', tol);
      testCase.verifyEqual(G.rw, L.G.rw, 'AbsTol', tol);
      testCase.verifyEqual(G.zw, L.G.zw, 'AbsTol', tol);
      testCase.verifyEqual(G.rm, L.G.rm, 'AbsTol', tol);
      testCase.verifyEqual(G.zm, L.G.zm, 'AbsTol', tol);
      testCase.verifyEqual(G.am, L.G.am, 'AbsTol', tol);
      testCase.verifyEqual(G.rf, L.G.rf, 'AbsTol', tol);
      testCase.verifyEqual(G.zf, L.G.zf, 'AbsTol', tol);


      % Coils with circular section are not yet implemented in IDS.
      % This has ww, hw = nan. Removing them from the test.
      notcircular = ~isnan(L.G.ww);
      testCase.verifyEqual(G.ww(notcircular), L.G.ww(notcircular), 'AbsTol', tol);
      testCase.verifyEqual(G.hw(notcircular), L.G.hw(notcircular), 'AbsTol', tol);

      % Compute inductances and resistances
      P = fgep(L.P);
      G = fgeg(G, P);
      L_new = fgec(P,G);

      if testCase.debugplot
        meqgplot(G)
      end

      % Check circuit matrices
      testCase.verifyEqual(L_new.G.Ra, L.G.Ra, 'AbsTol', tol);
      testCase.verifyEqual(L_new.G.Twa, L.G.Twa, 'AbsTol', tol);

      % Maa will be slightly different because vertical stability coil is considered squared instead of circular.
      testCase.verifyEqual(L_new.G.Maa, L.G.Maa, 'AbsTol', 1e-5);

      % Test the vessel eigenvalues are correct
      [~,Le] = vveig(L.G.Muu,L.G.Ru);
      [~,Le_new] = vveig(L_new.G.Muu,L_new.G.Ru);
      testCase.verifyEqual(Le(1:neig), Le_new(1:neig), 'AbsTol', tol);

      if testCase.debugplot
        % Plot the vessel eigenvalues
        figure
        hold on
        plot(Le, 'o')
        plot(Le_new, '.')
      end

    end
    
  end % Close methods
end % Close class



