classdef fgs_jacobian_comp_test < fgs_fge_jacobian_comp_test
  % Compares FGS analytical Jacobian with finite-difference Jacobian
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties(TestParameter)
    algosNL = {'all-nl', 'all-nl-Fx', 'Newton-GS'};
    shot = struct('circular',1,'diverted',2,'double_null',4,'squashed',5);
    agcon = struct('Ip_bp_li',{{'Ip','bp','li'}},...  % Need to nest cell arrays to avoid creating structure arrays
                   'Ip_bp_qA',{{'Ip','bp','qA'}},...
                   'Ip_li_qA',{{'Ip','li','qA'}},...
                   'bp_li_qA',{{'bp','li','qA'}});
    usecs = struct('true',true,'false',false);
    % Separate multiple domain cases as they typically run with fewer bf
    shot_multiD = struct(meq_jacobian_test.doublets_under_test{:});
    %
    convergence_test = struct('false',false,'true',true);
  end

  methods
    function setup_code(testCase, algosNL)
      testCase.code = 'fgs';
      testCase.algoNL = algosNL;
      % State perturbation amplitude (using already existing deltaFx)
      if strcmp(algosNL, 'all-nl')
        testCase.deltaFx = 0.5e-4;
        testCase.meshDeltaFx = logspace(-4,-2,4);
      else
        testCase.deltaFx = 1e-6;
        testCase.meshDeltaFx = logspace(-6,-5,3);
      end
    end
  end

  methods(Test,TestTags={'Jacobian-Integration'},ParameterCombination='pairwise')
    function test_code_jacobian_vs_fd_singleD(testCase,algosNL,shot,agcon,usecs,convergence_test)

      % Run setup
      testCase.setup_code(algosNL);

      % Additional arguments for cubic-spline interpolation
      PP = {'agcon',agcon};
      if usecs
        PP = [PP,{'icsint',true,'ilim',3}];
      else
        PP = [PP,{'icsint',false,'ilim',1}];
      end

      % Run initial setup
      [L,LX] = get_L_LX(testCase,shot,testCase.t,PP{:});

      if usecs
        % Check we aren't too close to cell boundaries
        mindz = min(abs([LX.zA;LX.zX] - L.zx.'),[],[1,2]);
        mindr = min(abs([LX.rA;LX.rX] - L.rx.'),[],[1,2]);
        testCase.assumeTrue(min(mindr,mindz)>1e-6,'Skipping test as singular point is too close to grid cell boundary');
      end

      % Call generic routine
      testCase.test_code_jacobian_vs_fd(L,LX,convergence_test);
    end

    function test_code_jacobian_vs_fd_multiD(testCase,algosNL,shot_multiD,convergence_test)

      % Run setup
      testCase.setup_code(algosNL);

      % Run initial setup
      [L,LX] = get_L_LX(testCase,shot_multiD,testCase.t);

      % Call generic routine
      testCase.test_code_jacobian_vs_fd(L,LX,convergence_test);
    end

    function test_code_jacobian_assignIy(testCase,algosNL)
      testCase.setup_code(algosNL)
      shot_ = 2; % just one shot
      convergence_test_ = false;
      % force assigning Iy by setting Ipmin=Inf
      [L,LX] = get_L_LX(testCase,shot_,testCase.t,'Ipmin',Inf);
      testCase.test_code_jacobian_vs_fd(L,LX,convergence_test_);
    end
  end

end
