classdef fgs_fge_broyden_solver_test < meq_jacobian_test
  % Tests Forward-Grad-Shafranov solvers FGS and FGE using semi-analytical Jacobian
  %
  % [+MEQ MatlabEQuilibrium Toolbox+] Swiss Plasma Center EPFL Lausanne 2022. All rights reserved.

  properties
    tok = 'ana';
    t = 0:5e-3:1e-2;
    verbosity = 0;
    agcon = {'bp','li','qA'};
    solution
    time
  end

  properties(ClassSetupParameter)
    codes       = {'fgs','fge'};
    algosNL     = {'all-nl','all-nl-Fx'};%'Newton-GS' not implemented
  end

  properties(TestParameter)
    shot = struct('diverted',2,'double_null',4,'squashed',5,'doublet',82);
    large_system= {false,true};
    group_size  = {1,3};
    ldim        = {5,10};
  end

  methods(TestClassSetup)

    function set_code_solver(testCase,codes,algosNL)
      testCase.assumeFalse(strcmp(algosNL, 'Newton-GS'), ...
        'Newton-GS method not yet supported for broyden method.');
      testCase.code      = codes;
      testCase.algoNL    = algosNL;
      start=tic;
      % Compute the solution using default JFNK to compare with
      [L,LX] = get_L_LX(testCase,2,testCase.t,'algoF','jfnk', ...
        'anajac',true,'usepreconditioner',true);
      LY = get_LY(testCase,L,LX);
      testCase.time=toc(start);
      LY.nfeval=sum(LY.nfeval);
      LY.niter=sum(LY.niter);
      testCase.solution  = LY;
    end
  end

  methods(Test,TestTags={'fge'})
    function test_broyden_errors_anajac(testCase)
      % Checks that errors are thrown for the case anajac=False
      params={'algoF','broyden', ...
        'use_inverse',true,'is_factored',false,'large_system','true',...
        'group_size',4,...
        'anajac',false,'usepreconditioner',false};

      verifyError(testCase,@()get_L_LX(testCase,2,testCase.t,params{:}),'broyden:anajac')
    end

    function test_broyden_errors_newton_gs(testCase)
      % Checks that errors are launch for the case algoNL=Newton-GS
      params={'algoNL','Newton-GS','algoF','broyden', ...
        'use_inverse',true,'is_factored',false,'large_system','true',...
        'group_size',4,...
        'anajac',true,'usepreconditioner',false};

      verifyError(testCase,@()get_L_LX(testCase,2,testCase.t,params{:}), ...
        'broyden:Newton_GS')
    end

    function test_broyden_inverse_convergence(testCase,shot,large_system,group_size)
      % Check the convergence of the Broyden method when
      % 'use_inverse'=true and  'is_factored'=false
      if testCase.verbosity
        fprintf('Testing tok=%s, algoGMRES=%s with the %s Jacobian\n',testCase.tok,testCase.algoGMRES,results_prefix)
      end

      % Run on shot with broyden method
      % is_factored set to False
      [L,LX] = get_L_LX(testCase,shot,testCase.t,'algoF','broyden', ...
        'use_inverse',true,'is_factored',false,'large_system',large_system,...
        'group_size',group_size,...
        'anajac',true,'usepreconditioner',false);
      LY = get_LY(testCase,L,LX);

      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),sprintf('shot %d did not converge',shot));
    end

    function test_broyden_inverse_factored_convergence(testCase,shot,large_system,group_size,ldim)
      % Check the convergence of the Broyden method when
      % 'use_inverse'=true and 'is_factored'=True
      if testCase.verbosity
        fprintf('Testing tok=%s, algoGMRES=%s with the %s Jacobian\n',testCase.tok,testCase.algoGMRES,results_prefix)
      end

      % Run on shot with broyden method
      [L,LX] = get_L_LX(testCase,shot,testCase.t,'algoF','broyden',...
        'use_inverse',true,'is_factored',true,'large_system',large_system,...
        'group_size',group_size,'ldim',ldim,...
        'anajac',true,'usepreconditioner',false);
      LY = get_LY(testCase,L,LX);

      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),sprintf('shot %d did not converge',shot));
    end

    function test_broyden_jac_convergence(testCase,shot,group_size)
      % Check that the convergence of the Broyden method when 'use_inverse'=false
      if testCase.verbosity
        fprintf('Testing tok=%s, algoGMRES=%s with the %s Jacobian\n',testCase.tok,testCase.algoGMRES,results_prefix)
      end

      % Run on shot with broyden method
      [L,LX] = get_L_LX(testCase,shot,testCase.t,'algoF','broyden',...
        'group_size',group_size,'anajac',true,'usepreconditioner',false, 'use_inverse',false);
      LY = get_LY(testCase,L,LX);

      testCase.verifyTrue(meq_test.check_convergence(L,LX.t,LY),sprintf('shot %d did not converge',shot));
    end

    function test_broyden_inverse(testCase,large_system,group_size)
      % Check that the Broyden solution corresponds to one obtained with the jfnk solver
      % Case where 'use_inverse'=true, 'is_factored'=false
      % Tested on shot 2
      shot=2;
      if testCase.verbosity
        fprintf('Testing tok=%s, algoGMRES=%s with the %s Jacobian\n',testCase.tok,testCase.algoGMRES,results_prefix)
      end

      start_time = tic;
      % Run on shot with newton and broyden method
      [L,LX] = get_L_LX(testCase,shot,testCase.t,'algoF','broyden',...
        'use_inverse',true,'is_factored',false,'large_system',large_system,...
        'group_size',group_size,...
        'anajac',true,'usepreconditioner',false);
      LY = get_LY(testCase,L,LX);
      LY.t = toc(start_time);

      % sum over all timesteps
      LY.nfeval=sum(LY.nfeval);
      LY.niter=sum(LY.niter);

      MethodName = [upper(testCase.code), sprintf(' shot=%d group size=%d inv-jac', shot,group_size)];

      MethodName_jfnk=[upper(testCase.code), sprintf(' shot=%d',shot)];
      testCase.check_solution(testCase,LY,MethodName,MethodName_jfnk)
    end

    function test_broyden_inverse_factored(testCase,large_system,group_size,ldim)
      % Check that the Broyden solution corresponds to one obtained with the jfnk solver
      % Case where 'use_inverse'=true, 'is_factored'=true
      % Tested on shot 2
      shot=2;
      if testCase.verbosity
        fprintf('Testing tok=%s, algoGMRES=%s with the %s Jacobian\n',testCase.tok,testCase.algoGMRES,results_prefix)
      end

      start_time = tic;
      % Run on shot with broyden method
      [L,LX] = get_L_LX(testCase,shot,testCase.t,'algoF','broyden',...
        'use_inverse',true,'is_factored',true,'large_system',large_system,...
        'group_size',group_size,'ldim',ldim,...
        'anajac',true,'usepreconditioner',false);
      LY = get_LY(testCase,L,LX);
      LY.t = toc(start_time);

      % sum over all timesteps
      LY.nfeval=sum(LY.nfeval);
      LY.niter=sum(LY.niter);

      MethodName = [upper(testCase.code), sprintf(' shot=%d group size=%d ldim=%d', shot,group_size,ldim)];
      MethodName_jfnk=[upper(testCase.code), sprintf(' shot=%d',shot)];
      testCase.check_solution(testCase,LY,MethodName,MethodName_jfnk)
    end

    function test_broyden_jac(testCase,group_size)
      % Check that the Broyden solution corresponds to one obtained with the jfnk solver
      % Case where 'use_inverse'=false
      % Tested on shot 2
      shot=2;
      if testCase.verbosity
        fprintf('Testing tok=%s, algoGMRES=%s with the %s Jacobian\n',testCase.tok,testCase.algoGMRES,results_prefix)
      end

      start_time = tic;
      % Run on shot with broyden method
      [L,LX] = get_L_LX(testCase,shot,testCase.t,'algoF','broyden',...
        'group_size',group_size,'anajac',true,'usepreconditioner',false,'use_inverse',false);
      LY = get_LY(testCase,L,LX);
      LY.t = toc(start_time);
      % sum over all timesteps
      LY.nfeval=sum(LY.nfeval);
      LY.niter=sum(LY.niter);

      MethodName = [upper(testCase.code), sprintf(' shot=%d group size=%d', shot,group_size)];
      MethodName_jfnk=[upper(testCase.code), sprintf(' shot=%d',shot)];
      testCase.check_solution(testCase,LY,MethodName,MethodName_jfnk)
    end
  end

  methods(Static)
    function check_solution(testCase,LY, MethodName,MethodName_jfnk)
      % test validity of results
      testCase.verifyTrue(all([LY.isconverged,testCase.solution.isconverged], 'all'))
      % tol = sqrt(eps);
      tol = 1e-10;
      testCase.verifyEqual(LY.rB, testCase.solution.rB, 'AbsTol', tol);
      testCase.verifyEqual(LY.zB, testCase.solution.zB, 'AbsTol', tol);
      testCase.verifyEqual(LY.FB, testCase.solution.FB, 'AbsTol', tol);
      testCase.verifyEqual(LY.rA, testCase.solution.rA, 'AbsTol', tol);
      testCase.verifyEqual(LY.zA, testCase.solution.zA, 'AbsTol', tol);
      testCase.verifyEqual(LY.FA, testCase.solution.FA, 'AbsTol', tol);
      testCase.verifyEqual(LY.Fx, testCase.solution.Fx, 'AbsTol', tol);

      % analyze and display results
      Methods   = {[MethodName,' with broyden']; [MethodName_jfnk,' with jfnk']; 'Relative (broyden/jfnk)'};
      ComputationTime = [ LY.t     ; testCase.time     ; LY.t      / testCase.time    ];
      TotalNiter      = [ LY.niter ; testCase.solution.niter ; LY.niter  / testCase.solution.niter  ];
      TotalNfeval     = [ LY.nfeval; testCase.solution.nfeval; LY.nfeval / testCase.solution.nfeval ];
      ComparisonTable = table(Methods,ComputationTime,TotalNiter,TotalNfeval);
      disp(ComparisonTable);
    end
  end

end